﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Text;

namespace Nintendo.InGameEditing.UI
{
    /// <summary>
    /// テキストボックスクラスです。
    /// </summary>
    public class TextBoxModel : ControlModel
    {
        internal static readonly string UiType = "TextBox";
        private static readonly PropertyChangedEventArgs TextEventArgs = new PropertyChangedEventArgs(nameof(Text));
        private static readonly PropertyChangedEventArgs FileDropEventArgs = new PropertyChangedEventArgs(nameof(IsFileDropEnabled));

        private readonly ValueNode<string> node;
        private string currentValue;
        private bool isFileDropEnabled = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="node">元になるノード</param>
        /// <param name="attrs">メタデータに指定された属性の辞書</param>
        internal TextBoxModel(ValueNode<string> node, Dictionary<string, string> attrs) : base(node, attrs)
        {
            this.node = node;
            node.ValueReceived += (s, e) => SetPropertyValue(ref currentValue, e.Value, TextEventArgs);

            var size = default(string);
            var sizeValue = default(int);
            if (attrs.TryGetValue(nameof(size), out size) && int.TryParse(size, out sizeValue))
            {
                Size = sizeValue;
            }
        }

        /// <summary>
        /// テキストを取得設定します。
        /// テキストは UTF-8 換算で Size バイト未満である必要があります。
        /// </summary>
        /// <exception cref="ArgumentException">
        /// 文字列が設定可能なバイトサイズを超えています。
        /// </exception>
        public string Text
        {
            get { return currentValue; }
            set
            {
                var textByteCount = value != null ? Encoding.UTF8.GetByteCount(value) : 0;
                if (textByteCount >= this.Size)
                {
                    throw new ArgumentException($"文字列が設定可能なバイトサイズを超えています。");
                }

                if (SetPropertyValue(ref currentValue, value, TextEventArgs))
                {
                    node.PushValue(value);
                }
            }
        }

        /// <summary>
        /// ファイルドロップが有効かどうかを取得します。
        /// </summary>
        public bool IsFileDropEnabled
        {
            get { return isFileDropEnabled; }
            private set { SetPropertyValue(ref isFileDropEnabled, value, FileDropEventArgs); }
        }

        /// <summary>
        /// 文字列バッファのサイズを取得します。
        /// 設定する文字列は UTF-8 換算で Size バイト未満である必要があります。
        /// </summary>
        public int Size { get; }

        protected override void UpdateAttributes(Dictionary<string, string> attrs)
        {
            var fileDrop = default(string);
            var fileDropValue = default(bool);
            if (attrs.TryGetValue(nameof(fileDrop), out fileDrop) && bool.TryParse(fileDrop, out fileDropValue))
            {
                IsFileDropEnabled = fileDropValue;
            }
            else
            {
                IsFileDropEnabled = false;
            }
        }
    }
}
