﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Linq;
using Nintendo.InGameEditing.Utilities;

namespace Nintendo.InGameEditing.UI
{
    /// <summary>
    /// コントロールのリストです。
    /// </summary>
    public class ControlModelList : ReadOnlyObservableCollection<ControlModel>, INotifyCollectionChanged
    {
        private readonly ImplList impl;

        internal ControlModelList() : base(new ImplList())
        {
            this.impl = (ImplList)Items;
        }

        /// <summary>
        /// nodeList の全ての要素を自動的にコントロールモデルに変換します。
        /// </summary>
        /// <param name="nodeList">監視するノードリスト</param>
        internal ControlModelList(NodeList nodeList) : this()
        {
            nodeList.CollectionChanged += (s, e)
                => e.NewItems?.Cast<Node>().Select(n => n.ToControlModel()).ForEach(Add);
        }

        /// <summary>
        /// コレクションの変更を通知します。
        /// </summary>
        public new event NotifyCollectionChangedEventHandler CollectionChanged;

        event NotifyCollectionChangedEventHandler INotifyCollectionChanged.CollectionChanged
        {
            add { CollectionChanged += value; }
            remove { CollectionChanged -= value; }
        }

        /// <summary>
        /// コレクション変更通知
        /// </summary>
        /// <param name="args">イベント引数</param>
        protected override void OnCollectionChanged(NotifyCollectionChangedEventArgs args) => CollectionChanged?.Invoke(this, args);

        /// <summary>
        /// 元ノードが削除済みのコントロールを削除します。
        /// </summary>
        public void Refresh()
        {
            foreach (var item in impl.Where(model => model.IsRemoved).ToArray())
            {
                (item as IHasChildren)?.Children.Refresh();
                impl.Remove(item);
            }
        }

        internal void Add(ControlModel model)
        {
            if (model == null) throw new ArgumentNullException(nameof(model));
            model.Node.Removed += (s, args) => { if (args.Reason == RemovedReason.Deleted) { impl.Remove(model); } };
            impl.Add(model);
        }

        private class ImplList : ObservableSortedCollection<ControlModel>
        {
            private static readonly ModelComparer comparer = new ModelComparer();
            private readonly object lockObj = new object();

            public ImplList() : base(comparer)
            {
            }

            protected override void InsertItem(int _, ControlModel item)
            {
                if (item == null) { throw new ArgumentNullException(nameof(item)); }
                lock (lockObj) { base.InsertItem(_, item); }
            }

            protected override void RemoveItem(int index)
            {
                lock (lockObj) { base.RemoveItem(index); }
            }

            protected override void ClearItems()
            {
                lock (lockObj) { base.ClearItems(); }
            }

            private class ModelComparer : IComparer<ControlModel>
            {
                public int Compare(ControlModel x, ControlModel y) => Comparer<uint>.Default.Compare(x.Node.Id, y.Node.Id);
            }
        }
    }
}
