﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Reflection;

namespace Nintendo.InGameEditing.Utilities
{
    internal class WeakDelegateManager
    {
        private readonly List<WeakDelegate> list = new List<WeakDelegate>();

        public void Add(Delegate value)
        {
            if (value == null) { throw new ArgumentNullException(nameof(value)); }

            lock (((ICollection)list).SyncRoot)
            {
                list.Add(new WeakDelegate(value));
            }
        }

        public void Remove(Delegate value)
        {
            if (value == null) { throw new ArgumentNullException(nameof(value)); }

            lock (((ICollection)list).SyncRoot)
            {
                foreach (var weakDelegate in list.ToArray())
                {
                    object target;
                    if (weakDelegate.TryGetTarget(out target) && (!Equals(target, value.Target) || weakDelegate.Method != value.Method))
                    {
                        continue; // 生きていて、インスタンスかメソッドが違う、ものは有効;
                    }
                    list.Remove(weakDelegate);
                }
            }
        }

        public void Invoke(params object[] parameters)
        {
            lock (((ICollection)list).SyncRoot)
            {
                list.RemoveAll(weakDelegate => !weakDelegate.TryInvoke(parameters));
            }
        }

        private class WeakDelegate
        {
            private readonly WeakReference<object> weakTarget;

            public WeakDelegate(Delegate source)
            {
                if (source == null) { throw new ArgumentNullException(nameof(source)); }
                if (source.Method == null) { throw new ArgumentNullException(nameof(source.Method)); }

                if (source.Target != null)
                {
                    weakTarget = new WeakReference<object>(source.Target);
                }

                Method = source.Method;
            }

            public bool TryGetTarget(out object target)
            {
                target = null;
                return Method.IsStatic || weakTarget?.TryGetTarget(out target) == true;
            }

            public MethodInfo Method { get; }

            public bool TryInvoke(params object[] parameters)
            {
                object target;
                if (!TryGetTarget(out target)) { return false; }

                Method.Invoke(target, parameters);
                return true;
            }
        }
    }
}
