﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;

namespace Nintendo.InGameEditing.Utilities
{
    public class SortedCollection<T> : Collection<T>
    {
        private readonly IComparer<T> comparer = Comparer<T>.Default;

        public SortedCollection()
        {
        }

        public SortedCollection(IEnumerable<T> collection)
        {
            if (collection == null) { throw new ArgumentNullException(nameof(collection)); }
            foreach (var item in collection) { Add(item); }
        }

        public SortedCollection(IComparer<T> comparer)
            : this(Enumerable.Empty<T>(), comparer)
        {
        }

        public SortedCollection(IEnumerable<T> collection, IComparer<T> comparer)
        {
            if (collection == null) { throw new ArgumentNullException(nameof(collection)); }
            if (comparer == null) { throw new ArgumentNullException(nameof(comparer)); }

            this.comparer = comparer;
            foreach (var item in collection) { Add(item); }
        }

        protected override void InsertItem(int _, T item)
        {
            var index = BinarySerchHelper.BinarySearch(this, 0, Count, item, comparer);
            if (index < 0)
            {
                index = ~index;
            }
            base.InsertItem(index, item);
        }

        protected override void SetItem(int index, T item)
        {
            RemoveItem(index);
            InsertItem(0, item);
        }
    }

    public class ObservableSortedCollection<T> : ObservableCollection<T>
    {
        private readonly IComparer<T> comparer = Comparer<T>.Default;

        public ObservableSortedCollection()
        {
        }

        public ObservableSortedCollection(IEnumerable<T> collection) : base(collection)
        {
        }

        public ObservableSortedCollection(IComparer<T> comparer) : this(Enumerable.Empty<T>(), comparer)
        {
        }

        public ObservableSortedCollection(IEnumerable<T> collection, IComparer<T> comparer)
        {
            if (comparer == null) { throw new ArgumentNullException(nameof(comparer)); }
            this.comparer = comparer;

            // comparer を使用するため base(collection) はできない
            foreach (var item in collection) { Add(item); }
        }

        protected override void InsertItem(int _, T item)
        {
            var index = BinarySerchHelper.BinarySearch(this, 0, Count, item, comparer);
            if (index < 0)
            {
                index = ~index;
            }
            base.InsertItem(index, item);
        }

        protected override void SetItem(int index, T item)
        {
            RemoveItem(index);
            InsertItem(-1, item);
        }

        protected sealed override void MoveItem(int oldIndex, int newIndex)
        {
            // ソート済みコレクションなので Move は無効
        }
    }

    internal static class BinarySerchHelper
    {
        // implementation from ArraySortHelper<T>.InternalBinarySearchHelper
        internal static int BinarySearch<T>(IList<T> list, int index, int length, T value, IComparer<T> comparer = null)
        {
            if (list == null) { throw new ArgumentNullException(nameof(list)); }
            if (index < 0 || length < 0 || list.Count - index < length) { throw new ArgumentException($"{nameof(index)}, {nameof(length)}"); }

            comparer = comparer ?? Comparer<T>.Default;

            int lo = index;
            int hi = index + length - 1;
            while (lo <= hi)
            {
                int i = lo + ((hi - lo) >> 1);
                int order = comparer.Compare(list[i], value);

                if (order == 0) return i;
                if (order < 0)
                {
                    lo = i + 1;
                }
                else
                {
                    hi = i - 1;
                }
            }

            return ~lo;
        }
    }
}
