﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace Nintendo.InGameEditing.Utilities
{
    using System;
    using System.IO;

    /// <summary>
    /// 転送回数を削減するためにキャッシュして出力するストリームクラスです。
    /// </summary>
    internal class CachedStream : Stream
    {
        public const int DefaultWriteCacheSize = 1024;

        private readonly Stream sourceStream;
        private readonly byte[] writeCache;
        private int writeCacheCurrent;

        //-----------------------------------------------------------------

        public CachedStream(Stream sourceStream)
            : this(sourceStream, DefaultWriteCacheSize)
        {
        }

        public CachedStream(Stream sourceStream, int writeCacheSize)
        {
            if (sourceStream == null) { throw new ArgumentNullException(nameof(sourceStream)); }
            if (writeCacheSize <= 0) { throw new ArgumentException(nameof(writeCacheSize)); }

            this.sourceStream = sourceStream;
            writeCache = new byte[writeCacheSize];
        }

        //-----------------------------------------------------------------

        public Stream SourceStream => this.sourceStream;

        public override bool CanRead => this.sourceStream.CanRead;

        public override bool CanSeek => this.sourceStream.CanSeek;

        public override bool CanWrite => this.sourceStream.CanWrite;

        public override long Length => this.sourceStream.Length;

        public override long Position
        {
            get { return this.sourceStream.Position; }
            set { this.sourceStream.Position = value; }
        }

        public int WriteCacheSize => this.writeCache.Length;

        //-----------------------------------------------------------------

        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);

            if (disposing)
            {
                this.sourceStream.Close();
            }
        }

        public override void Flush()
        {
            this.FlushCache();
            this.sourceStream.Flush();
        }

        public override int Read(byte[] buffer, int offset, int count) => this.sourceStream.Read(buffer, offset, count);

        public override long Seek(long offset, SeekOrigin origin) => this.sourceStream.Seek(offset, origin);

        public override void SetLength(long value) => this.sourceStream.SetLength(value);

        public override void Write(byte[] buffer, int offset, int count)
        {
            // 転送回数を削減するためにキャッシュして出力します。
            // バッファサイズを超える場合は、キャッシュ済みの内容を先に出力します。
            // count がバッファより大きい場合は、直接出力します。
            if (this.writeCacheCurrent + count < this.writeCache.Length)
            {
                Array.Copy(buffer, offset, this.writeCache, this.writeCacheCurrent, count);
                this.writeCacheCurrent += count;
                return;
            }

            this.FlushCache();

            if (count > this.writeCache.Length)
            {
                this.sourceStream.Write(buffer, 0, count);
                return;
            }

            Array.Copy(buffer, offset, this.writeCache, 0, count);
        }

        private void FlushCache()
        {
            if (this.writeCacheCurrent <= 0) { return; }

            this.sourceStream.Write(this.writeCache, 0, this.writeCacheCurrent);
            this.writeCacheCurrent = 0;
        }
    }
}
