﻿using System;
using System.Threading;
using System.Threading.Tasks;

namespace Nintendo.InGameEditing.Utilities
{
    /// <summary>
    /// 非同期ロックオブジェクトです。
    /// スレッドアフィニティがないため await と併用することができます。
    /// 同一スレッド内での再帰的なロック取得はできません。
    /// </summary>
    public sealed class AsyncLockObject
    {
        private readonly SemaphoreSlim semaphore = new SemaphoreSlim(1, 1);
        private readonly Task<IDisposable> releaser;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public AsyncLockObject()
        {
            releaser = Task.FromResult((IDisposable)new Releaser(this));
        }

        /// <summary>
        /// 非同期にロックを取得を行い、結果の Dispose でロックを解放します。
        /// using ステートメントで使用することを前提とします。
        /// </summary>
        /// <returns>ロック取得タスク</returns>
        public Task<IDisposable> LockAsync()
        {
            var wait = semaphore.WaitAsync();
            return wait.IsCompleted ? releaser :
                    wait.ContinueWith((_, state) => (IDisposable)state,
                    releaser.Result,
                    CancellationToken.None,
                    TaskContinuationOptions.ExecuteSynchronously,
                    TaskScheduler.Default);
        }

        /// <summary>
        /// 同期的にロックの取得を行い、Dispose でロックを解放します。
        /// </summary>
        /// <returns>Dispose でロックを解放するオブジェクト</returns>
        public IDisposable Lock()
        {
            semaphore.Wait();
            return releaser.Result;
        }

        private sealed class Releaser : IDisposable
        {
            private readonly AsyncLockObject toRelease;

            internal Releaser(AsyncLockObject toRelease)
            {
                this.toRelease = toRelease;
            }

            public void Dispose() => toRelease.semaphore.Release();
        }
    }
}
