﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Linq;
using System.Reactive.Disposables;
using Nintendo.InGameEditing.Utilities;

namespace Nintendo.InGameEditing
{
    /// <summary>
    /// ノードのリストです。
    /// </summary>
    public class NodeList : ReadOnlyObservableCollection<Node>, INotifyCollectionChanged
    {
        private readonly ImplList impl;

        internal NodeList() : base(new ImplList())
        {
            impl = (ImplList)Items;
        }

        /// <summary>
        /// コレクションの変更を通知します。
        /// </summary>
        public new event NotifyCollectionChangedEventHandler CollectionChanged;

        event NotifyCollectionChangedEventHandler INotifyCollectionChanged.CollectionChanged
        {
            add { CollectionChanged += value; }
            remove { CollectionChanged -= value; }
        }

        /// <summary>
        /// コレクション変更通知
        /// </summary>
        /// <param name="args">イベント引数</param>
        protected override void OnCollectionChanged(NotifyCollectionChangedEventArgs args) => CollectionChanged?.Invoke(this, args);

        internal void Add(Node item) => impl.Add(item);

        internal void RemoveItems()
        {
            foreach (var item in this.ToArray())
            {
                item.Disposer.Dispose();
            }
        }

        private class ImplList : ObservableSortedCollection<Node>
        {
            private static readonly NodeOrderComparer comparer = new NodeOrderComparer();
            private readonly object lockObj = new object();

            public ImplList() : base(comparer)
            {
            }

            protected override void InsertItem(int index, Node item)
            {
                if (item == null) { throw new ArgumentNullException(nameof(item)); }
                if (item.Disposer.IsDisposed) { throw new ObjectDisposedException(item.GetType().FullName); }

                lock (lockObj)
                {
                    Disposable.Create(() => this.Remove(item)).AddTo(item.Disposer);

                    base.InsertItem(index, item);
                }
            }

            protected override void RemoveItem(int index)
            {
                lock (lockObj)
                {
                    base.RemoveItem(index);
                }
            }

            protected override void ClearItems()
            {
                lock (lockObj)
                {
                    base.ClearItems();
                }
            }

            private class NodeOrderComparer : IComparer<Node>
            {
                public int Compare(Node x, Node y) => Comparer<uint>.Default.Compare(x.Id, y.Id);
            }
        }
    }
}
