﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using Nintendo.InGameEditing.Communication;

namespace Nintendo.InGameEditing.Messages
{
    internal static class MessageHelper
    {
        private static readonly Dictionary<MessageType, Func<Packet, Message>> parserDictionary;
        private const string ParseMethodName = "Parse";
        private const string MessageTypePropertyName = "MessageType";

        static MessageHelper()
        {
            var flags = BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic;

            parserDictionary = Assembly.GetExecutingAssembly()
                .GetTypes()
                .Select(type => new
                {
                    ParseMethod = type.GetMethod(ParseMethodName, flags, null, new[] { typeof(Packet) }, null),
                    MessageTypeProperty = type.GetProperty(MessageTypePropertyName, flags)
                })
                .Where(pair => pair.ParseMethod != null && pair.MessageTypeProperty?.CanRead == true)
                .ToDictionary(
                    pair => (MessageType)pair.MessageTypeProperty.GetValue(null),
                    pair => (Func<Packet, Message>)pair.ParseMethod.CreateDelegate(typeof(Func<Packet, Message>)));
        }

        internal static Message Parse(this Packet packet)
        {
            if (packet == null) { throw new ArgumentNullException(nameof(packet)); }

            var messageType = packet.ReadMessageType();

            Func<Packet, Message> parseFunc;
            if (!parserDictionary.TryGetValue(messageType, out parseFunc)) { throw new NotSupportedException(); }

            return parseFunc(packet);
        }

        internal static Packet CreatePacket(MessageType type, byte[] data)
        {
            var typeBuffer = type.ToBytes();
            var buffer = new byte[typeBuffer.Length + data?.Length ?? 0];

            Buffer.BlockCopy(typeBuffer, 0, buffer, 0, typeBuffer.Length);

            if (data?.Length > 0)
            {
                Buffer.BlockCopy(data, 0, buffer, typeBuffer.Length, data.Length);
            }

            return new Packet(Const.Signature, buffer);
        }

        internal static MessageType ReadMessageType(this Packet packet) => ReadMessageType(packet.Payload);

        internal static MessageType ReadMessageType(byte[] buffer) => ReadMessageType(buffer, 0);

        internal static MessageType ReadMessageType(byte[] buffer, int startIndex)
            => (MessageType)BitConverter.ToInt32(buffer, startIndex);

        internal static byte[] ToBytes(this MessageType type) => BitConverter.GetBytes((int)type);
    }
}
