﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Globalization;
using System.Xml.Linq;

namespace Nintendo.InGameEditing.Communication.Htcs
{
    /// <summary>
    /// ターゲット情報です。
    /// </summary>
    internal class Target : IEquatable<Target>
    {
        /// <summary>
        /// XML上での要素名です。
        /// </summary>
        internal const string XElementName = "Target";

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Target(string peerType, string htcsPeerName)
        {
            this.PeerType = peerType;
            this.HtcsPeerName = htcsPeerName;
        }

        /// <summary>
        /// パース用コンストラクタ
        /// </summary>
        internal Target(XElement xElement)
        {
            this.PeerType = xElement.Element("PeerType").Value;
            this.HtcsPeerName = xElement.Element("HtcsPeerName").Value;
        }

        /// <summary>
        /// 通信システム名を取得します。
        /// </summary>
        public string PeerType { get; }

        /// <summary>
        /// ハードウェア識別名を取得します。
        /// </summary>
        public string HtcsPeerName { get; }

        /// <summary>
        /// 文字列化します。
        /// </summary>
        /// <returns>インスタンスの内容を表す文字列</returns>
        public override string ToString()
        {
            return string.Format(CultureInfo.InvariantCulture, "{0}, {1}", PeerType, HtcsPeerName);
        }

        /// <summary>
        /// インスタンスが同じ内容を保持しているかを比較します。
        /// </summary>
        /// <param name="other">もう1つのインスタンス</param>
        /// <returns>一致していればtrue,そうでなければfalse.</returns>
        public bool Equals(Target other)
        {
            return this.HtcsPeerName.Equals(other.HtcsPeerName) && this.PeerType.Equals(other.PeerType);
        }

        /// <summary>
        /// XMLに変換します。
        /// </summary>
        internal XElement ToXElement()
        {
            return new XElement(XElementName,
                new XElement("PeerType", PeerType),
                new XElement("HtcsPeerName", HtcsPeerName));
        }
    }
}
