﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Globalization;
using System.Net;
using System.Xml.Linq;

namespace Nintendo.InGameEditing.Communication.Htcs
{
    /// <summary>
    /// Htcsのポート情報です。
    /// </summary>
    public struct HtcsPortDescriptor : IEquatable<HtcsPortDescriptor>, IComparable<HtcsPortDescriptor>, IComparable
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="htcsPeerName">ハードウェア識別名</param>
        /// <param name="htcsPortName">ポート名</param>
        public HtcsPortDescriptor(HtcsPeerName htcsPeerName, string htcsPortName)
            : this()
        {
            this.HtcsPeerName = htcsPeerName;
            this.HtcsPortName = htcsPortName;
        }

        /// <summary>
        /// パース用コンストラクタ
        /// </summary>
        internal HtcsPortDescriptor(XElement xElement)
            : this()
        {
            this.HtcsPeerName = new HtcsPeerName(xElement.Element(HtcsPeerName.XElementName));
            this.HtcsPortName = xElement.Element("HtcsPortName").Value;
        }

        /// <summary>
        /// ハードウェア識別名を取得します。
        /// </summary>
        public HtcsPeerName HtcsPeerName { get; }

        /// <summary>
        /// ポート名を取得します。
        /// </summary>
        public string HtcsPortName { get; }

        /// <summary>
        /// ハッシュ化します。
        /// </summary>
        /// <returns>ハッシュコード</returns>
        public override int GetHashCode()
        {
            return this.HtcsPeerName.GetHashCode() ^ this.HtcsPortName.GetHashCode();
        }

        /// <summary>
        /// 文字列化します。
        /// </summary>
        /// <returns>インスタンスの内容を表す文字列</returns>
        public override string ToString()
        {
            return string.Format(
                CultureInfo.InvariantCulture, "{0}:{1}", this.HtcsPeerName.ToString(), this.HtcsPortName);
        }

        /// <summary>
        /// インスタンスが同じ内容を保持しているかを比較します。
        /// </summary>
        /// <param name="obj">もう1つのインスタンス</param>
        /// <returns>一致していればtrue,そうでなければfalse.</returns>
        public override bool Equals(object obj)
        {
            if (!(obj is HtcsPortDescriptor))
            {
                return false;
            }

            return ((IEquatable<HtcsPortDescriptor>)(this)).Equals((HtcsPortDescriptor)obj);
        }

        /// <summary>
        /// インスタンスが同じ内容を保持しているかを比較します。
        /// </summary>
        /// <param name="other">もう1つのインスタンス</param>
        /// <returns>一致していればtrue,そうでなければfalse.</returns>
        public bool Equals(HtcsPortDescriptor other)
        {
            return this.HtcsPeerName.Equals(other.HtcsPeerName) && this.HtcsPortName.Equals(other.HtcsPortName);
        }

        /// <summary>
        /// インスタンスの内容を比較し、大小関係を返します。
        /// </summary>
        /// <param name="other">もう1つのインスタンス</param>
        /// <returns>等しければ0,このインスタンスがotherより前なら負の値,otherより後ろなら正の値.</returns>
        public int CompareTo(HtcsPortDescriptor other)
        {
            var c1 = this.HtcsPeerName.CompareTo(other.HtcsPeerName);
            if (c1 != 0)
            {
                return c1;
            }

            return string.CompareOrdinal(this.HtcsPortName, other.HtcsPortName);
        }

        /// <summary>
        /// インスタンスの内容を比較し、大小関係を返します。
        /// </summary>
        /// <param name="obj">もう1つのインスタンス</param>
        /// <returns>等しければ0,このインスタンスがobjより前なら負の値,otherより後ろなら正の値.</returns>
        public int CompareTo(object obj)
        {
            if (!(obj is HtcsPortDescriptor))
            {
                throw new ArgumentException("obj must be a HtcsPortDescriptor", nameof(obj));
            }

            return ((IComparable<HtcsPortDescriptor>)(this)).CompareTo((HtcsPortDescriptor)obj);
        }

        #region 演算子オーバーロード

        public static bool operator ==(HtcsPortDescriptor lhs, HtcsPortDescriptor rhs)
        {
            return lhs.Equals(rhs);
        }

        public static bool operator !=(HtcsPortDescriptor lhs, HtcsPortDescriptor rhs)
        {
            return !lhs.Equals(rhs);
        }

        public static bool operator <(HtcsPortDescriptor lhs, HtcsPortDescriptor rhs)
        {
            return lhs.CompareTo(rhs) < 0;
        }

        public static bool operator >(HtcsPortDescriptor lhs, HtcsPortDescriptor rhs)
        {
            return lhs.CompareTo(rhs) > 0;
        }

        #endregion
    }

    /// <summary>
    /// ハードウェア識別名を表します。
    /// </summary>
    public struct HtcsPeerName : IEquatable<HtcsPeerName>, IComparable<HtcsPeerName>, IComparable
    {
        /// <summary>
        /// XML上での要素名です。
        /// </summary>
        internal const string XElementName = "HtcsPeerName";

        /// <summary>
        /// ホストが指定した場合、どのターゲットからの接続も受け付けることを示す値です。
        /// </summary>
        public static readonly HtcsPeerName Any = new HtcsPeerName(string.Empty);

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="idValue">ハードウェア識別名</param>
        public HtcsPeerName(string idValue)
            : this()
        {
            this.Value = idValue;
        }

        /// <summary>
        /// パース用コンストラクタ
        /// </summary>
        internal HtcsPeerName(XElement xElement)
            : this()
        {
            this.Value = xElement.Value;    // TODO: validate
        }

        /// <summary>
        /// ハードウェア識別名を取得します。
        /// </summary>
        public string Value { get; }

        /// <summary>
        /// インスタンスが同じ内容を保持しているかを比較します。
        /// </summary>
        /// <param name="other">もう1つのインスタンス</param>
        /// <returns>一致していればtrue,そうでなければfalse.</returns>
        public bool Equals(HtcsPeerName other)
        {
            return this.Value.Equals(other.Value);
        }

        /// <summary>
        /// インスタンスが同じ内容を保持しているかを比較します。
        /// </summary>
        /// <param name="obj">もう1つのインスタンス</param>
        /// <returns>一致していればtrue,そうでなければfalse.</returns>
        public override bool Equals(object obj)
        {
            return this.Value.Equals(obj);
        }

        /// <summary>
        /// ハッシュ化します。
        /// </summary>
        /// <returns>ハッシュコード</returns>
        public override int GetHashCode()
        {
            return this.Value.GetHashCode();
        }

        /// <summary>
        /// 文字列化します。
        /// </summary>
        /// <returns>インスタンスの内容を表す文字列</returns>
        public override string ToString()
        {
            return this.Value.ToString();
        }

        #region 演算子オーバーロード

        public static bool operator ==(HtcsPeerName lhs, HtcsPeerName rhs)
        {
            return lhs.Equals(rhs);
        }

        public static bool operator !=(HtcsPeerName lhs, HtcsPeerName rhs)
        {
            return !lhs.Equals(rhs);
        }

        public static bool operator <(HtcsPeerName lhs, HtcsPeerName rhs)
        {
            return lhs.CompareTo(rhs) < 0;
        }

        public static bool operator >(HtcsPeerName lhs, HtcsPeerName rhs)
        {
            return lhs.CompareTo(rhs) > 0;
        }

        #endregion

        /// <summary>
        /// インスタンスの内容を比較し、大小関係を返します。
        /// </summary>
        /// <param name="other">もう1つのインスタンス</param>
        /// <returns>等しければ0,このインスタンスがotherより前なら負の値,otherより後ろなら正の値.</returns>
        public int CompareTo(HtcsPeerName other)
        {
            return string.CompareOrdinal(this.Value, other.Value);
        }

        /// <summary>
        /// インスタンスの内容を比較し、大小関係を返します。
        /// </summary>
        /// <param name="obj">もう1つのインスタンス</param>
        /// <returns>等しければ0,このインスタンスがotherより前なら負の値,otherより後ろなら正の値.</returns>
        public int CompareTo(object obj)
        {
            if (!(obj is HtcsPeerName))
            {
                throw new ArgumentException("obj must be a HtcsPeerName", nameof(obj));
            }

            return this.CompareTo((HtcsPeerName)obj);
        }

        /// <summary>
        /// XMLに変換します。
        /// </summary>
        internal XElement ToXElement()
        {
            return new XElement(XElementName, Value);
        }
    }

    /// <summary>
    /// HTCS のポートと WinSock のポートの対応情報です。
    /// </summary>
    internal class PortMapItem : IEquatable<PortMapItem>
    {
        /// <summary>
        /// XML上での要素名です。
        /// </summary>
        internal const string XElementName = "PortMapItem";

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PortMapItem(HtcsPortDescriptor htcsPortDescriptor, IPEndPoint endPoint)
        {
            this.HtcsPortDescriptor = htcsPortDescriptor;
            this.EndPoint = endPoint;
        }

        /// <summary>
        /// パース用コンストラクタ
        /// </summary>
        internal PortMapItem(XElement xElement)
        {
            this.HtcsPortDescriptor = new HtcsPortDescriptor(xElement);

            var address = IPAddress.Parse(xElement.Element("IPAddress").Value);
            var port = int.Parse(xElement.Element("TcpPortNumber").Value, CultureInfo.InvariantCulture);
            this.EndPoint = new IPEndPoint(address, port);
        }

        /// <summary>
        /// Htcsポート記述子を取得します。
        /// </summary>
        public HtcsPortDescriptor HtcsPortDescriptor { get; }

        /// <summary>
        /// WinSockのポート情報を取得します。
        /// </summary>
        public IPEndPoint EndPoint { get; }

        /// <summary>
        /// 文字列化します。
        /// </summary>
        /// <returns>インスタンスの内容を表す文字列</returns>
        public override string ToString()
        {
            return string.Format(CultureInfo.InvariantCulture, "{0}, {1}", HtcsPortDescriptor, EndPoint);
        }

        /// <summary>
        /// インスタンスが同じ内容を保持しているかを比較します。
        /// </summary>
        /// <param name="other">もう1つのインスタンス</param>
        /// <returns>一致していればtrue,そうでなければfalse.</returns>
        public bool Equals(PortMapItem other)
        {
            return this.HtcsPortDescriptor.Equals(other.HtcsPortDescriptor) && this.EndPoint.Equals(other.EndPoint);
        }

        /// <summary>
        /// XMLに変換します。
        /// </summary>
        internal XElement ToXElement()
        {
            return new XElement(XElementName,
                HtcsPortDescriptor.HtcsPeerName.ToXElement(),
                new XElement("HtcsPortName", HtcsPortDescriptor.HtcsPortName),
                new XElement("IPAddress", EndPoint.Address.ToString()),
                new XElement("TcpPortNumber", EndPoint.Port.ToString(CultureInfo.InvariantCulture)));
        }
    }
}
