﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml.Linq;

namespace Nintendo.InGameEditing.Communication.Htcs
{
    /// <summary>
    /// ポートマッピング情報です。
    /// </summary>
    internal class PortMap : IEnumerable<PortMapItem>, IEquatable<PortMap>
    {
        /// <summary>
        /// XML上での要素名です。
        /// </summary>
        internal const string XElementName = "PortMap";

        /// <summary>
        /// ポートマッピング情報のリストです。
        /// </summary>
        private readonly List<PortMapItem> items = new List<PortMapItem>();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PortMap()
        {
            // 何もしない
        }

        /// <summary>
        /// コピー用コンストラクタ
        /// </summary>
        internal PortMap(IEnumerable<PortMapItem> portMapItemEnum)
        {
            this.items.Clear();
            this.items.AddRange(portMapItemEnum);
        }

        /// <summary>
        /// パース用コンストラクタ
        /// </summary>
        internal PortMap(XElement xElement)
        {
            items.AddRange(
                xElement.Elements(PortMapItem.XElementName).Select(x => new PortMapItem(x)));
        }

        /// <summary>
        /// インスタンスが同じ内容を保持しているかを比較します。
        /// </summary>
        /// <param name="other">もう1つのインスタンス</param>
        /// <returns>一致していればtrue,そうでなければfalse.</returns>
        public bool Equals(PortMap other)
        {
            if (this.items.Count != other?.items.Count)
            {
                return false;
            }

            return !this.items.Where((t, i) => !t.Equals(other.items[i])).Any();
        }

        #region IEnumerable<PortMapItem> メンバー

        /// <summary>
        /// Enumeratorを取得します。
        /// </summary>
        public IEnumerator<PortMapItem> GetEnumerator()
        {
            return items.GetEnumerator();
        }

        #endregion

        #region IEnumerable メンバー

        /// <summary>
        /// Enumeratorを取得します。
        /// </summary>
        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return items.GetEnumerator();
        }

        #endregion

        /// <summary>
        /// ポートマッピング情報を追加します。
        /// </summary>
        internal void Add(PortMapItem item)
        {
            items.Add(item);
        }

        /// <summary>
        /// ポートマッピング情報を削除します。
        /// </summary>
        internal void Remove(HtcsPortDescriptor htcsPortDescriptor)
        {
            items.RemoveAt(items.FindIndex(s => s.HtcsPortDescriptor == htcsPortDescriptor));
        }

        /// <summary>
        /// XMLに変換します。
        /// </summary>
        internal XElement ToXElement()
        {
            return new XElement(XElementName,
                items.Select(s => s.ToXElement()));
        }
    }
}
