﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using Nintendo.InGameEditing.Communication.Htcs;

namespace Nintendo.InGameEditing.Communication
{
    /// <summary>
    /// HTCSやそれに類するサービスにより、ターゲットアプリと接続するコネクタを取得する機能を提供します。
    /// </summary>
    public class ConnectionManager : IDisposable
    {
        /// <summary>
        /// コネクタリストです。
        /// </summary>
        private readonly List<TargetConnector> _usingConnectors = new List<TargetConnector>();

        /// <summary>
        /// プロトコル名とコネクタのテーブルです。将来的には動的に解決します。
        /// </summary>
        private readonly Dictionary<string, Type> _connectorTable = new Dictionary<string, Type>
        {
            { "HTCS", typeof(HtcConnector) },
            { "HIO", typeof(Hio.HioConnector) },
        };

        /// <summary>
        /// 破棄済みフラグです。
        /// </summary>
        private volatile bool _disposed = false;

        /// <summary>
        /// ターゲットリストです。
        /// </summary>
        private TargetList _targetList = new TargetList();

        /// <summary>
        /// ポート情報リストです。
        /// </summary>
        private PortMap _portMap = new PortMap();

        /// <summary>
        /// ターゲットとポート情報が階層化されたリストを取得します。
        /// </summary>
        public IReadOnlyList<TargetInfo> TargetInfos
        {
            get
            {
                lock (this._usingConnectors)
                {
                    return this._targetList.Select(target => new TargetInfo(target, this._portMap)).ToList();
                }
            }
        }

        /// <summary>
        /// 対応しているプロトコル名のリストを取得します。
        /// </summary>
        public IReadOnlyList<string> SupportedProtocols
        {
            get
            {
                lock (this._usingConnectors)
                {
                    return this._connectorTable.Select(c => c.Key).ToList();
                }
            }
        }

        /// <summary>
        /// 接続状態に変化があった場合にターゲットリストとポートマッピング情報を通知します。
        /// </summary>
        public event EventHandler<ConnectionInfoUpdatedEventArgs> ConnectionInfoUpdated;

        /// <summary>
        /// デストラクタ
        /// </summary>
        ~ConnectionManager()
        {
            this.Dispose();
        }

        /// <summary>
        /// 接続情報の受信待ちを開始します。
        /// </summary>
        /// <param name="htcSystemFilter">HTCS以外に追加で対応する通信システム名に対してtrueを返す関数(省略可能)</param>
        /// <returns>受信待ちが開始されたらtrue,既に開始済みだったらfalse.</returns>
        public bool Start(Func<string, bool> htcSystemFilter = null)
        {
            if (this._disposed)
            {
                return false;
            }

            lock (this._usingConnectors)
            {
                bool anyStarted = false;
                if (!this._usingConnectors.Any(c => c is HtcConnector))
                {
                    this._usingConnectors.Add(new HtcConnector());
                    this._usingConnectors.Last().ConnectionInfoUpdated += this.ReceivePortMapFromConnectors;
                    anyStarted = this._usingConnectors.Last().Start();
                }

                if (htcSystemFilter == null)
                {
                    return anyStarted;
                }

                foreach (var pair in this._connectorTable.Where(pair => htcSystemFilter(pair.Key)))
                {
                    if (this._usingConnectors.All(c => c.GetType() != pair.Value))
                    {
                        this._usingConnectors.Add((TargetConnector)Activator.CreateInstance(pair.Value));
                        this._usingConnectors.Last().ConnectionInfoUpdated += this.ReceivePortMapFromConnectors;
                        anyStarted = this._usingConnectors.Last().Start();
                    }
                }

                return anyStarted;
            }
        }

        /// <summary>
        /// 指定したプロトコルのみ受信待ちを開始します。
        /// </summary>
        /// <param name="protocolName">プロトコル名(HTCS,HIO,etc...)</param>
        /// <param name="port">ポート番号を上書きしたい場合はその値(省略可能、省略時はデフォルト値を使用)</param>
        /// <returns>受信待ちが開始されたらtrue,既に開始済みかプロトコルが見つからなかったらfalse.</returns>
        public bool Start(string protocolName, int port = -1)
        {
            if (this._disposed)
            {
                return false;
            }

            lock (this._usingConnectors)
            {
                Type connectorType;
                if (!this._connectorTable.TryGetValue(protocolName, out connectorType))
                {
                    return false;
                }

                if (this._usingConnectors.Any(c => c.GetType() == connectorType))
                {
                    return false;
                }

                TargetConnector connector;
                if (port == -1)
                {
                    connector = (TargetConnector)Activator.CreateInstance(connectorType);
                }
                else
                {
                    connector = (TargetConnector)Activator.CreateInstance(connectorType, port);
                }

                connector.ConnectionInfoUpdated += this.ReceivePortMapFromConnectors;
                this._usingConnectors.Add(connector);

                return connector.Start();
            }
        }

        /// <summary>
        /// 指定したプロトコルのみ受信待ちを開始します。
        /// </summary>
        /// <param name="protocolName">プロトコル名(HTCS,HIO,etc...)</param>
        /// <returns>受信待ちが停止されたらtrue,既に停止済みかプロトコルが見つからなかったらfalse.</returns>
        public bool Stop(string protocolName)
        {
            if (this._disposed)
            {
                return false;
            }

            lock (this._usingConnectors)
            {
                Type connectorType;
                if (!this._connectorTable.TryGetValue(protocolName, out connectorType))
                {
                    return false;
                }

                if (this._usingConnectors.All(c => c.GetType() != connectorType))
                {
                    return false;
                }

                var targetConnector = this._usingConnectors.First(c => c.GetType() == connectorType);
                this._usingConnectors.Remove(targetConnector);

                targetConnector.Stop();

                return true;
            }
        }

        /// <summary>
        /// 接続情報の受信待ちを停止します。
        /// </summary>
        public void Stop()
        {
            lock (this._usingConnectors)
            {
                foreach (var connector in _usingConnectors)
                {
                    connector.Stop();
                }

                this._usingConnectors.Clear();
            }
        }

        /// <summary>
        /// リソースを破棄します。
        /// </summary>
        public void Dispose()
        {
            this.Stop();
            this._disposed = true;
            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// 指定プロトコルが利用可能な状態にあるかどうかを調べます。
        /// </summary>
        /// <param name="protocolName">状態を調べたいプロトコル名(省略した場合はHTCS)</param>
        /// <returns>プロトコルが利用可能ならtrue,そうでなければfalse.</returns>
        public bool IsAvailable(string protocolName = "")
        {
            lock (this._usingConnectors)
            {
                // プロトコル名が省略されていたらHTCコネクターの接続状態を返す
                if (string.IsNullOrEmpty(protocolName))
                {
                    protocolName = "HTCS";
                }

                // 対応するコネクタを探して接続状態を返す
                Type connectorType;
                bool result = this._connectorTable.TryGetValue(protocolName, out connectorType);
                if (result)
                {
                    var connector = this._usingConnectors.FirstOrDefault(c => c.GetType() == connectorType);
                    return connector != null && connector.IsConnected;
                }

                return false;
            }
        }

        /// <summary>
        /// ホスト側をサーバにするためにポートマッピング情報をHtcDaemonに登録します。
        /// </summary>
        /// <param name="portMapItem">Htcsポート情報</param>
        /// <param name="endPoint">ホスト側で作成したサーバソケットのIPとポート番号</param>
        /// <returns>成功の場合は0,失敗の場合は対応するエラーコード</returns>
        public uint RegisterPort(HtcsPortDescriptor portMapItem, IPEndPoint endPoint)
        {
            lock (this._usingConnectors)
            {
                var htcConnector = this._usingConnectors.FirstOrDefault(c => c is HtcConnector) as HtcConnector;
                if (htcConnector == null)
                {
                    return uint.MaxValue;
                }

                return htcConnector.RegisterPort(portMapItem.HtcsPeerName, portMapItem.HtcsPortName, endPoint);
            }
        }

        /// <summary>
        /// ホスト側をサーバにするためにポートマッピング情報をHtcDaemonに登録します。
        /// </summary>
        /// <param name="htcsPeerName">プラットフォーム識別子</param>
        /// <param name="htcsPortName">クライアント側が問い合わせるポート名</param>
        /// <param name="endPoint">ホスト側で作成したサーバソケットのIPとポート番号</param>
        /// <returns>成功の場合は0,失敗の場合は対応するエラーコード</returns>
        public uint RegisterPort(HtcsPeerName htcsPeerName, string htcsPortName, IPEndPoint endPoint)
        {
            HtcsPortDescriptor htcsPortDesc = new HtcsPortDescriptor(htcsPeerName, htcsPortName);
            return RegisterPort(htcsPortDesc, endPoint);
        }

        /// <summary>
        /// HtcDaemonに登録したポートマッピング情報を解除します。
        /// </summary>
        /// <param name="htcsPortDescriptor">ポートマッピング情報</param>
        /// <returns>成功の場合は0,失敗の場合は対応するエラーコード</returns>
        public uint UnregisterPort(HtcsPortDescriptor htcsPortDescriptor)
        {
            lock (this._usingConnectors)
            {
                var htcConnector = this._usingConnectors.FirstOrDefault(c => c is HtcConnector) as HtcConnector;
                if (htcConnector == null)
                {
                    return uint.MaxValue;
                }

                return htcConnector.UnregisterPort(htcsPortDescriptor);
            }
        }

        /// <summary>
        /// HtcDaemonに登録したポートマッピング情報を解除します。
        /// </summary>
        /// <param name="htcsPeerName">プラットフォーム識別子</param>
        /// <param name="htcsPortName">クライアント側が問い合わせるポート名</param>
        /// <returns>成功の場合は0,失敗の場合は対応するエラーコード</returns>
        public uint UnregisterPort(HtcsPeerName htcsPeerName, string htcsPortName)
        {
            HtcsPortDescriptor htcsPortDesc = new HtcsPortDescriptor(htcsPeerName, htcsPortName);
            return UnregisterPort(htcsPortDesc);
        }

        /// <summary>
        /// コネクタから接続情報を収集し、マージしてイベントを発生させます。
        /// </summary>
        /// <param name="sender">使用しません</param>
        /// <param name="args">使用しません</param>
        private void ReceivePortMapFromConnectors(object sender, ConnectionInfoUpdatedEventArgs args)
        {
            lock (this._usingConnectors)
            {
                var oldTargets = this._targetList;
                var oldMap = this._portMap;

                var newTargets = new TargetList();
                var newMap = new PortMap();

                foreach (var connector in _usingConnectors)
                {
                    newTargets = new TargetList(newTargets.Concat(connector.TargetList));
                    newMap = new PortMap(newMap.Concat(connector.PortMap));
                }

                // イベント発生前と同じ内容だったらイベントの発生をキャンセルする
                if (newTargets.Equals(oldTargets) && newMap.Equals(oldMap))
                {
                    return;
                }

                this._targetList = newTargets;
                this._portMap = newMap;

                ConnectionInfoUpdated?.Invoke(this, new ConnectionInfoUpdatedEventArgs(this._targetList, this._portMap));
            }
        }
    }
}
