﻿using Nintendo.Foundation.Contracts;
using System;
using System.Diagnostics.CodeAnalysis;
using System.IO;
using System.Runtime.InteropServices;

namespace Nintendo.Atk.Binary
{
    /// <summary>
    /// Opus データ内の情報ヘッダです。
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 1)]
    [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1121:UseBuiltInTypeAlias", Justification = "バイナリのサイズを明示するため")]
    public class AudioOpusInfoHeader
    {
        /// <summary>
        /// ヘッダのサイズ。
        /// </summary>
        public static readonly int StructSize = Marshal.SizeOf(typeof(AudioOpusInfoHeader));

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="type">OpusInfo のタイプ。</param>
        /// <param name="size">OpusInfo のサイズ。</param>
        public AudioOpusInfoHeader(AudioOpusInfoType type, int size)
        {
            Ensure.Argument.True(size >= StructSize);

            this.Type = type;
            this.Size = size;
        }

        /// <summary>
        /// バイナリデータから、AudioOpusInfoHeader を生成します。
        /// </summary>
        /// <param name="reader">BinaryReader。</param>
        /// <returns>生成した AudioOpusInfoHeader。</returns>
        public static AudioOpusInfoHeader FromBinary(BinaryReader reader)
        {
            Ensure.Argument.NotNull(reader);

            return new AudioOpusInfoHeader(
                (AudioOpusInfoType)reader.ReadInt32(),
                reader.ReadInt32());
        }

        /// <summary>
        /// OpusInfo のタイプ。
        /// </summary>
        public AudioOpusInfoType Type { get; private set; }

        /// <summary>
        /// OpusInfo のサイズ。
        /// ヘッダサイズは含みません。
        /// </summary>
        public Int32 Size { get; private set; }
    }
}
