﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "gfxutil/FontRenderer.h"
#include "AtkPlayerBuildConfig.h"

#if  defined( ATKPLAYER_BUILD_CONFIG_ENABLE_GFX )
#include <nn/gfx.h>
#include <nn/vi.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_GraphicsFramework.h>

namespace gfxutil
{

//! @brief      Gfx に関するインスタンスを扱います
class GfxContext
{
public:
    //! @brief  初期化します
    void Initialize(void* pMemory, size_t memorySize, int displayWidth, int displayHeight) NN_NOEXCEPT;
    //! @brief  終了処理をします
    void Finalize() NN_NOEXCEPT;

    //! @biref  描画を開始します
    void Begin() NN_NOEXCEPT;
    //! @biref  描画を終了します
    void End() NN_NOEXCEPT;

    //! @biref  四角形を描画します
    void DrawQuad(const nn::util::Vector3fType& position, const nn::util::Float2& size, const nn::util::Uint8x4& color) NN_NOEXCEPT;
    //! @biref  四角形を描画します
    void DrawQuad(const nn::util::Vector3fType& position, const nn::util::Float2& size, const nn::util::Uint8x4& topColor, const nn::util::Uint8x4& bottomColor) NN_NOEXCEPT;
    //! @biref  線分を描画します
    void DrawLine(const nn::util::Vector3fType& begin, const nn::util::Vector3fType& end, const nn::util::Uint8x4& color) NN_NOEXCEPT;
    //! @biref  点を結ぶように連続して線分を描画します
    void DrawLinkedPoint(const nn::util::Float3* pPoints, int pointCount, const nn::util::Uint8x4& color) NN_NOEXCEPT;

    //! @biref  FontRenderer を取得します
    FontRenderer& GetFontRenderer() NN_NOEXCEPT;

private:
    //  DrawLinkedPoint で描ける点の最大数
    static const int DrawalbeLinkedPointCountMax = 1024;
    //  DrawLinkedPoint を呼べる回数
    static const int DrawLinkedPointFunctionCallCount = 6;

    //! @biref  PrimitiveRenderer に渡す Allocate 関数
    static void* AllocateForPrimitiveRenderer(size_t size, size_t alignment, void* pUserData) NN_NOEXCEPT;
    //! @biref  PrimitiveRenderer に渡す Free 関数
    static void FreeForPrimitiveRenderer(void* ptr, void* pUserData) NN_NOEXCEPT;

private:
    nn::mem::StandardAllocator m_Allocator;
    int                        m_CurrentBufferIndex;

    nns::gfx::GraphicsFramework            m_GraphicsFramework;
    nns::gfx::PrimitiveRenderer::Renderer* m_pPrimitiveRenderer;
    FontRenderer                           m_FontRenderer;

    nns::gfx::PrimitiveRenderer::PrimitiveMesh m_LineMesh[DrawLinkedPointFunctionCallCount];
    nns::gfx::GpuBuffer                        m_LineMeshBuffer;
    int                                        m_LineMeshIndex;
    nn::gfx::MemoryPool                        m_LineMeshMemoryPool;

    nn::util::Matrix4x3fType m_ViewMatrix;
    nn::util::Matrix4x4fType m_ProjectionMatrix;
    nn::util::Matrix4x3fType m_ModelMatrix;
};


}

#else

#include <stddef.h>

namespace gfxutil
{

//  グラフィクスを使用しないときのダミー
class GfxContext
{
public:
    //! @brief  初期化します
    void Initialize(void* pMemory, size_t memorySize, int displayWidth, int displayHeight) NN_NOEXCEPT;
    //! @brief  終了処理をします
    void Finalize() NN_NOEXCEPT;

    //! @biref  描画を開始します
    void Begin() NN_NOEXCEPT;
    //! @biref  描画を終了します
    void End() NN_NOEXCEPT;
};


}

#endif  //  ATKPLAYER_BUILD_CONFIG_ENABLE_GFX
