﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "gfxutil/Panel.h"
#include "gfxutil/Label.h"
#include "HidPad.h"
#include "SoundRecorder.h"


//! @brief  波形出力用のパネルです
class SoundOutputPanel
{
public:
    //! @brief  必要なメモリ量を取得します
    size_t GetRequiredMemorySize() NN_NOEXCEPT;
    //! @brief  初期化します
    void Initialize(void* buffer, size_t bufferSize, float positionX, float positionY, float sizeX, float sizeY) NN_NOEXCEPT;
    //! @brief  終了します
    void Finalize() NN_NOEXCEPT;
    //! @brief  更新します
    void Update() NN_NOEXCEPT;
    //! @brief  入力による更新を行います
    void UpdateByHid(const HidPad& hidPad) NN_NOEXCEPT;
    //! @brief  描画します
    void Draw(gfxutil::GfxContext& gfxContext) NN_NOEXCEPT;

    //! @brief  フォーカスを設定します
    void SetFocused(bool isFocused) NN_NOEXCEPT;

private:
    enum ChannelIndex
    {
        ChannelIndex_FL,
        ChannelIndex_FR,
        ChannelIndex_FC,
        ChannelIndex_LFE,
        ChannelIndex_RL,
        ChannelIndex_RR,
        ChannelIndex_Count
    };

    //! @brief  1 チャンネル分の波形を描画します
    class WaveDrawer
    {
    public:
        //! @brief  必要なメモリ量を取得します
        size_t GetRequiredMemorySize(uint32_t sampleCountPerSec) NN_NOEXCEPT;
        //! @brief  初期化します
        void Initialize(void* buffer, size_t bufferSize, uint32_t sampleCountPerSec) NN_NOEXCEPT;

        //! @brief  サンプルを追加します
        void Push(const int16_t* pSamples, size_t sampleCount, int offset, int stride) NN_NOEXCEPT;
        //! @brief  波形を無音状態にします
        void Clear() NN_NOEXCEPT;
        //! @brief  背景を描画します
        void DrawBackGround(gfxutil::GfxContext& gfxContext, const nn::util::Float2& position, const nn::util::Float2& size) NN_NOEXCEPT;
        //! @brief  time に設定された時間分の波形を描画します
        void DrawWave(gfxutil::GfxContext& gfxContext, const nn::util::Float2& position, const nn::util::Float2& size, const nn::TimeSpan& time) NN_NOEXCEPT;

        //! @brief  ラベルの文字を設定します
        void SetText(const char* text) NN_NOEXCEPT;
        //! @brief  波形の振幅のスケールを設定します
        void SetAmpScale(float scale) NN_NOEXCEPT;

    private:
        //  波形を構成する線分の数
        static const int LineCount = 500;

        //! @brief  波形を構成する点のバッファサイズを取得します
        size_t GetRequiredMemorySizeForWaveDrawerPoints() const NN_NOEXCEPT;
        //! @brief  サンプルを保存するバッファのサイズを取得します
        size_t GetRequiredMemorySizeForWaveDrawerSampleBuffer(uint32_t sampleCountPerSec) const NN_NOEXCEPT;

    private:
        gfxutil::Label    m_Label;
        nn::util::Float3* m_pPoints;  //  波形の線分を構成する点

        int16_t* m_pSample;
        int      m_SampleCount;
        int      m_BlockSampleSize;
        int      m_WritePosition;

        float m_AmpScale;
    };

    //! @brief  1 アプリケーションフレームで読むサンプル数を取得します
    size_t GetSampleCountPerApplicationFrame() const NN_NOEXCEPT;

private:
    gfxutil::Panel m_Panel;
    gfxutil::Label m_InfoLabel;

    SoundRecorder  m_Recorder;
    WaveDrawer     m_WaveDrawer[ChannelIndex_Count];
    int16_t*       m_SampleBuffer;

    nn::TimeSpan   m_TimeScale;
    float          m_AmpScale;
};
