﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/mem/mem_StandardAllocator.h>
#include <nn/atk.h>
#include "SelectItemList.h"

//! @brief  Effect のパネルで使用する項目です
class EffectItem
{
public:
    //! @brief  初期化します
    void Initialize(float parameterClientAreaSizeX, float labelWidth) NN_NOEXCEPT;
    //! @brief  終了します
    void Finalize(nn::mem::StandardAllocator* pAllocator) NN_NOEXCEPT;

    //! @brief  更新します
    void Update() NN_NOEXCEPT;
    //! @brief  パラメータを更新します
    void UpdateParameters(const HidPad& hidPad) NN_NOEXCEPT;
    //! @brief  項目を描画します
    void DrawItem(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT;
    //! @brief  パラメータを描画します
    void DrawParameters(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT;

    //! @brief  パラメータがあるかどうかを取得します
    bool IsHasParameters() const NN_NOEXCEPT;
    //! @brief  使用メモリサイズを取得します
    size_t GetUsingMemorySize() const NN_NOEXCEPT;
    //! @brief  エフェクトを適用する必要があるかどうかを取得します
    bool IsNeedToAppend() const NN_NOEXCEPT;
    //! @brief  エフェクトを変更します
    void ChangeEffect() NN_NOEXCEPT;
    //! @brief  エフェクトを適用します
    void Append(nn::atk::AuxBus bus, nn::mem::StandardAllocator* pAllocator) NN_NOEXCEPT;
    //! @brief  エフェクトを消す準備をします。完了したとき true を返します
    bool PrepareToClear() NN_NOEXCEPT;

private:
    //  エフェクト処理の抽象クラス
    class EffectBase
    {
    public:
        //! @brief  初期化します
        virtual void Initialize(float parameterClientAreaSizeX) NN_NOEXCEPT = 0;
        //! @brief  終了します
        virtual void Finalize() NN_NOEXCEPT = 0;

        //! @brief  エフェクトを適用します
        virtual void Append(nn::atk::AuxBus bus, void* buffer, size_t bufferSize) NN_NOEXCEPT = 0;
        //! @brief  更新します
        virtual void Update() NN_NOEXCEPT = 0;
        //! @brief  パラメータを更新します
        virtual void UpdateParameters(const HidPad& hidPad) NN_NOEXCEPT = 0;
        //! @brief  パラメータを描画します
        virtual void DrawParameters(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT = 0;

        //! @brief  エフェクトを消す準備をします。完了したとき true を返します
        virtual bool PrepareToClear() NN_NOEXCEPT = 0;
        //! @brief  使用するメモリ量を取得します
        virtual size_t GetRequiredMemSize() NN_NOEXCEPT = 0;
        //! @brief  エフェクトを適用する必要があるかどうかを取得します
        virtual bool IsNeedToAppend() const NN_NOEXCEPT = 0;

        //! @brief  コンストラクタ
        EffectBase(const char* itemText, const nn::util::Unorm8x4& itemColor) NN_NOEXCEPT;
        //! @brief  項目を描画します
        void DrawItem(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT;
        //! @brief  ラベルの幅を設定します
        void SetLabelWidth(float width);

    private:
        gfxutil::Label m_ItemLabl;
    };
    //  何も行わないエフェクト
    class NoEffect : public EffectBase
    {
    public:
        //! @brief  コンストラクタ
        NoEffect() NN_NOEXCEPT;
        //! @brief  初期化します
        virtual void Initialize(float parameterClientAreaSizeX) NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  終了します
        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE;

        //! @brief  エフェクトを適用します
        virtual void Append(nn::atk::AuxBus bus, void* buffer, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  更新します
        virtual void Update() NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  パラメータを更新します
        virtual void UpdateParameters(const HidPad& hidPad) NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  パラメータを描画します
        virtual void DrawParameters(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT NN_OVERRIDE;

        //! @brief  エフェクトを無効化します。完了したとき true を返します
        virtual bool PrepareToClear() NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  使用するメモリ量を取得します
        virtual size_t GetRequiredMemSize() NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  エフェクトを適用する必要があるかどうかを取得します
        virtual bool IsNeedToAppend() const NN_NOEXCEPT NN_OVERRIDE;

    private:
        gfxutil::Label m_ParamLabel;
    };
    //  リバーブ
    class Reverb : public EffectBase
    {
    public:
        //! @brief  コンストラクタ
        Reverb() NN_NOEXCEPT;
        //! @brief  初期化します
        virtual void Initialize(float parameterClientAreaSizeX) NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  終了します
        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE;

        //! @brief  エフェクトを適用します
        virtual void Append(nn::atk::AuxBus bus, void* buffer, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  更新します
        virtual void Update() NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  パラメータを更新します
        virtual void UpdateParameters(const HidPad& hidPad) NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  パラメータを描画します
        virtual void DrawParameters(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT NN_OVERRIDE;

        //! @brief  エフェクトを無効化します。完了したとき true を返します
        virtual bool PrepareToClear() NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  使用するメモリ量を取得します
        virtual size_t GetRequiredMemSize() NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  エフェクトを適用する必要があるかどうかを取得します
        virtual bool IsNeedToAppend() const NN_NOEXCEPT NN_OVERRIDE;

    private:
        enum ParamType
        {
            ParamType_OutGain,
            ParamType_ReverbGain,
            ParamType_EarlyGain,
            ParamType_DryGain,
            ParamType_LateGain,
            ParamType_EarlyMode,
            ParamType_LateMode,
            ParamType_DecayTime,
            ParamType_PredelayTime,
            ParamType_Coloration,
            ParamType_HighFrequencyDecayRatio,
            ParamType_Count,
        };

        nn::atk::EffectReverb m_Reverb;
        SelectItemList<ParamType_Count> m_ParamList;
    };
    //  ディレイ
    class Delay : public EffectBase
    {
    public:
        //! @brief  コンストラクタ
        Delay() NN_NOEXCEPT;
        //! @brief  初期化します
        virtual void Initialize(float parameterClientAreaSizeX) NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  終了します
        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE;

        //! @brief  エフェクトを適用します
        virtual void Append(nn::atk::AuxBus bus, void* buffer, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  更新します
        virtual void Update() NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  パラメータを更新します
        virtual void UpdateParameters(const HidPad& hidPad) NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  パラメータを描画します
        virtual void DrawParameters(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT NN_OVERRIDE;

        //! @brief  エフェクトを無効化します。完了したとき true を返します
        virtual bool PrepareToClear() NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  使用するメモリ量を取得します
        virtual size_t GetRequiredMemSize() NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  エフェクトを適用する必要があるかどうかを取得します
        virtual bool IsNeedToAppend() const NN_NOEXCEPT NN_OVERRIDE;

    private:
        enum ParamType
        {
            ParamType_DelayTime,
            ParamType_DelayTimeMax,
            ParamType_InGain,
            ParamType_DryGain,
            ParamType_FeedbackGain,
            ParamType_LowPassAmount,
            ParamType_Count,
        };

        nn::atk::EffectDelay m_Delay;
        SelectItemList<ParamType_Count> m_ParamList;
        nn::TimeSpan m_DelayTimeMax;
    };
    //  I3dl2 リバーブ
    class I3dl2Reverb : public EffectBase
    {
    public:
        //! @brief  コンストラクタ
        I3dl2Reverb() NN_NOEXCEPT;
        //! @brief  初期化します
        virtual void Initialize(float parameterClientAreaSizeX) NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  終了します
        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE;

        //! @brief  エフェクトを適用します
        virtual void Append(nn::atk::AuxBus bus, void* buffer, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  更新します
        virtual void Update() NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  パラメータを更新します
        virtual void UpdateParameters(const HidPad& hidPad) NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  パラメータを描画します
        virtual void DrawParameters(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT NN_OVERRIDE;

        //! @brief  エフェクトを無効化します。完了したとき true を返します
        virtual bool PrepareToClear() NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  使用するメモリ量を取得します
        virtual size_t GetRequiredMemSize() NN_NOEXCEPT NN_OVERRIDE;
        //! @brief  エフェクトを適用する必要があるかどうかを取得します
        virtual bool IsNeedToAppend() const NN_NOEXCEPT NN_OVERRIDE;

    private:
        enum ParamType
        {
            ParamType_RoomGain,
            ParamType_RoomHfGain,
            ParamType_LateReverbHfDecayRatio,
            ParamType_LateReverbDecayTime,
            ParamType_ReflectionsGain,
            ParamType_ReflectionDelayTime,
            ParamType_ReverbGain,
            ParamType_LateReverbDelayTime,
            ParamType_LateReverbDiffusion,
            ParamType_LateReverbDensity,
            ParamType_HfReference,
            ParamType_DryGain,
            ParamType_Count,
        };

        nn::atk::EffectI3dl2Reverb m_I3dl2Reverb;
        SelectItemList<ParamType_Count> m_ParamList;
    };

    //  エフェクトの種類
    enum EffectType
    {
        EffectType_NoEffect,
        EffectType_Reverb,
        EffectType_Delay,
        EffectType_I3dl2Reverb,
        EffectType_Count
    };

private:
    //! @brief  今選択されている EffectBase の参照を取得します
    EffectBase& GetCurrentEffect() NN_NOEXCEPT;

private:
    EffectBase* m_pEffect[EffectType_Count];

    int m_CurrentEffectType;
    int m_NextEffectType;
    bool m_IsNeedToAppend;

    void* m_Buffer;
    size_t m_BufferSize;
    nn::audio::MemoryPoolType m_MemoryPool;

    NoEffect    m_NoEffect;
    Reverb      m_Reverb;
    Delay       m_Delay;
    I3dl2Reverb m_I3dl2Reverb;
};
