﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include "SoundArchivePlayerPanel.h"
#include "SoundParametersPanel.h"
#include "AtkPlayerBuildConfig.h"
#include "gfxutil/GfxContext.h"
#include "FlightRecorder.h"

namespace
{
    //  L ボタンを押しながらサウンドを左右キーで選択するときの進む量
    const uint32_t SoundIndexIncrementBoost = 0x100;

    //  パネルのタイトル
    const char* PanelTitle = "SoundArchivePlayer";

    //  ラベルに設定する再生中, 停止中, ポーズ中を表すテキスト、色
    const char* SoundPlayingText = "PLAY";
    const char* SoundStoppedText = "STOP";
    const char* SoundPausedText  = "PAUS";
    const nn::util::Unorm8x4 SoundPlayingColor = GetUnorm8x4( 192, 192,  64, 255 );
    const nn::util::Unorm8x4 SoundStoppedColor = GetUnorm8x4( 192, 192, 192, 255 );
    const nn::util::Unorm8x4 SoundPausedColor  = GetUnorm8x4(  32, 192,  64, 255 );
    //  サウンドアーカイブにサウンドがないときに表示されるテキスト
    const char* SoundNotFoundText = "------";
}
namespace
{
    //  SoundHandle の状態によって表示する文字列を返します
    const char* GetSoundHandleStateText(const WrappedSoundHandle& handle) NN_NOEXCEPT
    {
        if( handle.IsPlaying() )
        {
            return SoundPlayingText;
        }
        else if( handle.IsPaused() )
        {
            return SoundPausedText;
        }
        else
        {
            return SoundStoppedText;
        }
    }
    //  SoundHandle の状態によって表示する色を返します
    const nn::util::Unorm8x4& GetSoundHandleStateColor(const WrappedSoundHandle& handle) NN_NOEXCEPT
    {
        if( handle.IsPlaying() )
        {
            return SoundPlayingColor;
        }
        else if( handle.IsPaused() )
        {
            return SoundPausedColor;
        }
        else
        {
            return SoundStoppedColor;
        }
    }
    //  ラベルのテキストを設定します
    void SetLabelText(gfxutil::Label& label, const WrappedSoundHandle& handle) NN_NOEXCEPT
    {
        const char* name = handle.GetSoundLabel();
        if( name == nullptr )
        {
            name = SoundNotFoundText;
        }

        const char* state = GetSoundHandleStateText( handle );
        const uint32_t index = handle.GetSoundIndex();

        label.SetText( "[%s] %0.6x %s", state, index, name );
        label.SetColor( GetSoundHandleStateColor( handle ) );
    }
    //  サウンドの index を diff だけ変更します
    void ShiftSoundIndex(WrappedSoundHandle& handle, uint32_t diff, uint32_t soundCount) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_GREATER( soundCount, 0u );
        NN_ABORT_UNLESS_MINMAX( diff, 0u, soundCount );

        const uint32_t index = handle.GetSoundIndex();
        const uint32_t next = ( index + diff ) % soundCount;

        FlightRecorder::GetInstance().WriteLog( "[SndArcPlyr] handle[%d] Shift SoundIndex : %d -> %d, state:%s", handle.GetHandleIndex(), index, next, GetSoundHandleStateText( handle ) );
        handle.SetSoundIndex( next );

        //  再生中であるならば変更後の SoundIndex で再生する
        //  一時停止中ならば停止する
        if( handle.IsPaused() )
        {
            handle.Stop();
        }
        else if( handle.IsPlaying() )
        {
            handle.Start();
        }
    }
}

NN_DEFINE_STATIC_CONSTANT( const int SoundArchivePlayerPanel::SoundHandleCount );

//  初期化します
void SoundArchivePlayerPanel::Initialize(float positionX, float positionY, float sizeX, float sizeY) NN_NOEXCEPT
{
    m_Panel.SetTitle( PanelTitle );
    m_Panel.SetPosition( positionX, positionY );
    m_Panel.SetSize( sizeX, sizeY );

    const nn::util::Float2 panelPos = m_Panel.GetClientPositionLeftTop();
    const float itemSizeX = m_Panel.GetClientPositionRightBottom().x - panelPos.x;
    m_ItemList.Initialize( itemSizeX );
    m_ItemList.SetSelectedItemIndex( 0 );
    for( int i = 0; i < SoundHandleCount; i++ )
    {
        m_ItemList.GetValueLabel( i ).SetText( "" );
    }

    SetFocused( false );
}
//  更新します
void SoundArchivePlayerPanel::Update(const SoundArchiveContext& soundArchiveContext) NN_NOEXCEPT
{
    for( int i = 0; i < SoundHandleCount; i++ )
    {
        const WrappedSoundHandle& handle = soundArchiveContext.GetSoundHandle( i );
        SetLabelText( m_ItemList.GetNameLabel( i ), handle );
    }
}
//! @brief  入力による更新を行います
void SoundArchivePlayerPanel::UpdateByHid(SoundArchiveContext& soundArchiveContext, const HidPad& hidPad) NN_NOEXCEPT
{
    m_ItemList.UpdateByHid( hidPad );

    if( m_Panel.IsFocused() )
    {
        WrappedSoundHandle& handle = GetCurrentHandle( soundArchiveContext );
        if( hidPad.IsContinue( HidPad::Button_Left ) )
        {
            //  サウンドの選択
            const uint32_t soundCount = soundArchiveContext.GetSoundCount();

            if( soundCount > 0 )
            {
                const uint32_t boost = SoundIndexIncrementBoost % soundCount;
                const uint32_t diff = hidPad.IsHold( HidPad::Button_L ) ? boost : 1;
                ShiftSoundIndex( handle, soundCount - diff, soundCount );
            }
        }
        else if( hidPad.IsContinue( HidPad::Button_Right ) )
        {
            //  サウンドの選択
            const uint32_t soundCount = soundArchiveContext.GetSoundCount();

            if( soundCount > 0 )
            {
                const uint32_t boost = SoundIndexIncrementBoost % soundCount;
                const uint32_t diff = hidPad.IsHold( HidPad::Button_L ) ? boost : 1;
                ShiftSoundIndex( handle, diff, soundCount );
            }
        }

        //  再生, 停止, ポーズ
        if( hidPad.IsTrigger( HidPad::Button_A ) )
        {
            FlightRecorder::GetInstance().WriteLog( "[SndArcPlyr] Start handle[%d]", handle.GetHandleIndex() );
            handle.Start();
        }
        else if( hidPad.IsTrigger( HidPad::Button_B ) )
        {
            FlightRecorder::GetInstance().WriteLog( "[SndArcPlyr] Stop handle[%d]", handle.GetHandleIndex() );
            handle.Stop();
        }
        else if( hidPad.IsTrigger( HidPad::Button_Y ) )
        {
            FlightRecorder::GetInstance().WriteLog( "[SndArcPlyr] Pause handle[%d]", handle.GetHandleIndex() );
            handle.Pause();
        }
    }
}
//  描画します
void SoundArchivePlayerPanel::Draw(gfxutil::GfxContext& gfxContext) NN_NOEXCEPT
{
    m_Panel.Draw( gfxContext );

    const nn::util::Float2 panelPos = m_Panel.GetClientPositionLeftTop();
    m_ItemList.Draw( gfxContext, panelPos.x, panelPos.y );
}

//  選択中のハンドルを返します
WrappedSoundHandle& SoundArchivePlayerPanel::GetCurrentHandle(SoundArchiveContext& soundArchiveContext) NN_NOEXCEPT
{
    const int index = m_ItemList.GetSelectedItemIndex();
    NN_ABORT_UNLESS_RANGE( index, 0, SoundHandleCount );
    return soundArchiveContext.GetSoundHandle( index );
}
//  選択中のハンドルを返します
const WrappedSoundHandle& SoundArchivePlayerPanel::GetCurrentHandle(const SoundArchiveContext& soundArchiveContext) const NN_NOEXCEPT
{
    const int index = m_ItemList.GetSelectedItemIndex();
    NN_ABORT_UNLESS_RANGE( index, 0, SoundHandleCount );
    return soundArchiveContext.GetSoundHandle( index );
}

//  フォーカスを設定します
void SoundArchivePlayerPanel::SetFocused(bool isFocused) NN_NOEXCEPT
{
    m_Panel.SetFocused( isFocused );
    m_ItemList.SetFocused( isFocused );
}
