﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include "SelectItemList.h"
#include "gfxutil/GfxContext.h"

namespace
{
    //  選択されていない項目の色
    const nn::util::Uint8x4 ItemDefaultBackColor         = GetUint8x4(  0,  0,  0,   0 );
    //  選択されている && フォーカスがある項目の色
    const nn::util::Uint8x4 ItemSelectedFocusedBackColor = GetUint8x4( 24, 28, 48, 255 );
    //  選択されている && フォーカスがない項目の色
    const nn::util::Uint8x4 ItemSelectedBackColor        = GetUint8x4( 18, 20, 28, 255 );

#if defined( ATKPLAYER_BUILD_CONFIG_ENABLE_GFX )
    //  選択中の項目のマージン幅
    const float ItemSelectedMarginWidth = 4.0f;
#endif  //  ATKPLAYER_BUILD_CONFIG_ENABLE_GFX
}

//  コンストラクタです
SelectItemListImpl::SelectItemListImpl(int itemCount, gfxutil::Label* pNameLabel, gfxutil::Label* pValueLabel)
    : m_pName( pNameLabel )
    , m_pValue( pValueLabel )
    , m_ItemCount( itemCount )
{
    NN_ABORT_UNLESS_GREATER_EQUAL( itemCount, 0 );
    NN_ABORT_UNLESS_NOT_NULL( pNameLabel );
    NN_ABORT_UNLESS_NOT_NULL( pValueLabel );

    //  その他の変数の初期化を行う
    Initialize( 0.0f );
}

//  初期化します
void SelectItemListImpl::Initialize(float sizeX) NN_NOEXCEPT
{
    m_SizeX = sizeX;

    m_Current = 0;
    m_isFocused = false;

    for( int i = 0; i < m_ItemCount; i++ )
    {
        m_pName[i].SetBackColor( ItemDefaultBackColor );
        m_pValue[i].SetBackColor( ItemDefaultBackColor );
        m_pValue[i].SetDrawAlign( gfxutil::Label::DrawAlign_Left );
    }
}
//  入力による更新を行います
void SelectItemListImpl::UpdateByHid(const HidPad& hidPad) NN_NOEXCEPT
{
    if( m_isFocused )
    {
        if( hidPad.IsContinue( HidPad::Button_Down ) )
        {
            if( hidPad.IsHold( HidPad::Button_L ) )
            {
                m_Current = m_ItemCount - 1;
            }
            else
            {
                m_Current = GetNextValueOnLoopSelection( m_Current + 1, m_ItemCount );
            }
            NN_ABORT_UNLESS_RANGE( m_Current, 0, m_ItemCount );
        }
        else if( hidPad.IsContinue( HidPad::Button_Up ) )
        {
            if( hidPad.IsHold( HidPad::Button_L ) )
            {
                m_Current = 0;
            }
            else
            {
                m_Current = GetNextValueOnLoopSelection( m_Current - 1, m_ItemCount );
            }
            NN_ABORT_UNLESS_RANGE( m_Current, 0, m_ItemCount );
        }
    }
}
//  描画します
void SelectItemListImpl::Draw(gfxutil::GfxContext& gfxContext, float positionX, float positionY) NN_NOEXCEPT
{
#if defined( ATKPLAYER_BUILD_CONFIG_ENABLE_GFX )
    gfxutil::FontRenderer& fontRenderer = gfxContext.GetFontRenderer();

    //  項目を表示
    for( int i = 0; i < m_ItemCount; i++ )
    {
        gfxutil::Label& name = m_pName[i];
        gfxutil::Label& value = m_pValue[i];
        const float height = std::max( name.CalculateDrawSize( fontRenderer ).y, value.CalculateDrawSize( fontRenderer ).y );

        //  ハイライトを表示
        if( i == m_Current )
        {
            gfxContext.DrawQuad(
                nn::util::Vector3f( positionX - ItemSelectedMarginWidth, positionY, 0.0f ),
                GetFloat2( m_SizeX + 2.0f * ItemSelectedMarginWidth, height ),
                m_isFocused ? ItemSelectedFocusedBackColor : ItemSelectedBackColor
            );
        }

        name.SetPosition( positionX, positionY );
        name.Draw( gfxContext );

        value.SetPosition( positionX + m_SizeX, positionY );
        value.Draw( gfxContext );

        positionY += height;
    }
#else
    NN_UNUSED( gfxContext );
    NN_UNUSED( positionX );
    NN_UNUSED( positionY );

    const char focus = m_isFocused ? '|' : ' ';
    for( int i = 0; i < m_ItemCount; i++ )
    {
        gfxutil::Label& name = m_pName[i];
        gfxutil::Label& value = m_pValue[i];

        if( i == m_Current )
        {
            NN_LOG( "%c > %s %s\n", focus, name.GetText(), value.GetText() );
        }
        else
        {
            NN_LOG( "%c   %s %s\n", focus, name.GetText(), value.GetText() );
        }
    }
#endif  //  ATKPLAYER_BUILD_CONFIG_ENABLE_GFX
}

//  フォーカスを設定します
void SelectItemListImpl::SetFocused(bool isFocused) NN_NOEXCEPT
{
    m_isFocused = isFocused;
}
//  選択中の項目 index を返します
int SelectItemListImpl::GetSelectedItemIndex() const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RANGE( m_Current, 0, m_ItemCount );
    return m_Current;
}
//  選択中の項目 index を設定します
void SelectItemListImpl::SetSelectedItemIndex(int index) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RANGE( index, 0, m_ItemCount );
    m_Current = index;
}
//  index 番目の項目の名前ラベルを取得します
gfxutil::Label& SelectItemListImpl::GetNameLabel(int index) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RANGE( index, 0, m_ItemCount );
    return m_pName[index];
}
//  index 番目の項目の値ラベルを取得します
gfxutil::Label& SelectItemListImpl::GetValueLabel(int index) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RANGE( index, 0, m_ItemCount );
    return m_pValue[index];
}
