﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "EffectItem.h"
#include "FlightRecorder.h"

namespace
{
    //  Reverb パラメータの増加値 (減少値)
    const float ReverbIncrement_OutGain                 = 0.01f;
    const float ReverbIncrement_ReverbGain              = 0.01f;
    const float ReverbIncrement_EarlyGain               = 0.01f;
    const float ReverbIncrement_DryGain                 = 0.01f;
    const float ReverbIncrement_LateGain                = 0.01f;
    const int   ReverbIncrement_EarlyMode               = 1;
    const int   ReverbIncrement_LateMode                = 1;
    const nn::TimeSpan ReverbIncrement_DecayTime        = nn::TimeSpan::FromMilliSeconds( 100 );
    const nn::TimeSpan ReverbIncrement_PredelayTime     = nn::TimeSpan::FromMilliSeconds( 1 );
    const float ReverbIncrement_Coloration              = 0.01f;
    const float ReverbIncrement_HighFrequencyDecayRatio = 0.01f;

    //  Reverb パラメータの L キーによる加速された増加値 (減少値)
    const float ReverbIncrementBoost_OutGain                 = 0.1f;
    const float ReverbIncrementBoost_ReverbGain              = 0.1f;
    const float ReverbIncrementBoost_EarlyGain               = 0.1f;
    const float ReverbIncrementBoost_DryGain                 = 0.1f;
    const float ReverbIncrementBoost_LateGain                = 0.1f;
    const int   ReverbIncrementBoost_EarlyMode               = nn::atk::EffectReverb::EarlyMode_Count;
    const int   ReverbIncrementBoost_LateMode                = nn::atk::EffectReverb::LateMode_Count;
    const nn::TimeSpan ReverbIncrementBoost_DecayTime        = nn::TimeSpan::FromMilliSeconds( 1000 );
    const nn::TimeSpan ReverbIncrementBoost_PredelayTime     = nn::TimeSpan::FromMilliSeconds( 10 );
    const float ReverbIncrementBoost_Coloration              = 0.1f;
    const float ReverbIncrementBoost_HighFrequencyDecayRatio = 0.1f;

    //  Reverb パラメータの最小値
    const float ReverbMin_OutGain                 = nn::atk::EffectReverb::OutGainMin;
    const float ReverbMin_ReverbGain              = nn::atk::EffectReverb::ReverbGainMin;
    const float ReverbMin_EarlyGain               = nn::atk::EffectReverb::EarlyGainMin;
    const float ReverbMin_DryGain                 = nn::atk::EffectReverb::DryGainMin;
    const float ReverbMin_LateGain                = nn::atk::EffectReverb::LateGainMin;
    const int   ReverbMin_EarlyMode               = 0;
    const int   ReverbMin_LateMode                = 0;
    const nn::TimeSpan ReverbMin_DecayTime        = nn::atk::EffectReverb::GetDecayTimeMin();
    const nn::TimeSpan ReverbMin_PredelayTime     = nn::atk::EffectReverb::GetPredelayTimeMin();
    const float ReverbMin_Coloration              = nn::atk::EffectReverb::ColorationMin;
    const float ReverbMin_HighFrequencyDecayRatio = nn::atk::EffectReverb::HighFrequencyDecayRatioMin;

    //  Reverb パラメータの最大値
    const float ReverbMax_OutGain                 = nn::atk::EffectReverb::OutGainMax;
    const float ReverbMax_ReverbGain              = nn::atk::EffectReverb::ReverbGainMax;
    const float ReverbMax_EarlyGain               = nn::atk::EffectReverb::EarlyGainMax;
    const float ReverbMax_DryGain                 = nn::atk::EffectReverb::DryGainMax;
    const float ReverbMax_LateGain                = nn::atk::EffectReverb::LateGainMax;
    const int   ReverbMax_EarlyMode               = nn::atk::EffectReverb::EarlyMode_Count - 1;
    const int   ReverbMax_LateMode                = nn::atk::EffectReverb::LateMode_Count - 1;
    const nn::TimeSpan ReverbMax_DecayTime        = nn::atk::EffectReverb::GetDecayTimeMax();
    const nn::TimeSpan ReverbMax_PredelayTime     = nn::atk::EffectReverb::GetPredelayTimeMax();
    const float ReverbMax_Coloration              = nn::atk::EffectReverb::ColorationMax;
    const float ReverbMax_HighFrequencyDecayRatio = nn::atk::EffectReverb::HighFrequencyDecayRatioMax;
}
namespace
{
    //  Delay パラメータの増加値 (減少値)
    const nn::TimeSpan DelayIncrement_DelayTime     = nn::TimeSpan::FromMilliSeconds( 10 );
    const nn::TimeSpan DelayIncrement_DelayTimeMax  = DelayIncrement_DelayTime;
    const float DelayIncrement_InGain               = 0.01f;
    const float DelayIncrement_DryGain              = 0.01f;
    const float DelayIncrement_FeedbackGain         = 0.01f;
    const float DelayIncrement_LowPassAmount        = 0.01f;

    //  Delay パラメータの L キーによる加速された増加値 (減少値)
    const nn::TimeSpan DelayIncrementBoost_DelayTime     = nn::TimeSpan::FromMilliSeconds( 100 );
    const nn::TimeSpan DelayIncrementBoost_DelayTimeMax  = DelayIncrementBoost_DelayTime;
    const float DelayIncrementBoost_InGain               = 0.1f;
    const float DelayIncrementBoost_DryGain              = 0.1f;
    const float DelayIncrementBoost_FeedbackGain         = 0.1f;
    const float DelayIncrementBoost_LowPassAmount        = 0.1f;

    //  Delay パラメータの最小値
    const nn::TimeSpan DelayMin_DelayTime     = nn::TimeSpan::FromMilliSeconds( 0 );
    const nn::TimeSpan DelayMin_DelayTimeMax  = nn::TimeSpan::FromMilliSeconds( 0 );
    const float DelayMin_InGain               = nn::atk::EffectDelay::InGainMin;
    const float DelayMin_DryGain              = nn::atk::EffectDelay::DryGainMin;
    const float DelayMin_FeedbackGain         = nn::atk::EffectDelay::FeedbackGainMin;
    const float DelayMin_LowPassAmount        = nn::atk::EffectDelay::LowPassAmountMin;

    //  Delay パラメータの最大値
    const nn::TimeSpan DelayMax_DelayTimeMax  = nn::TimeSpan::FromMilliSeconds( 5 * 1000 );
    const float DelayMax_InGain               = nn::atk::EffectDelay::InGainMax;
    const float DelayMax_DryGain              = nn::atk::EffectDelay::DryGainMax;
    const float DelayMax_FeedbackGain         = nn::atk::EffectDelay::FeedbackGainMax;
    const float DelayMax_LowPassAmount        = nn::atk::EffectDelay::LowPassAmountMax;
}
namespace
{
    //  I3dl2Reverb パラメータの増加値 (減少値)
    const float        I3dl2ReverbIncrement_RoomGain = 10.0f;
    const float        I3dl2ReverbIncrement_RoomHfGain = 10.0f;
    const nn::TimeSpan I3dl2ReverbIncrement_ReflectionDelayTime = nn::TimeSpan::FromMilliSeconds( 1 );
    const float        I3dl2ReverbIncrement_ReflectionsGain = 10.0f;
    const nn::TimeSpan I3dl2ReverbIncrement_LateReverbDecayTime = nn::TimeSpan::FromMilliSeconds( 10 );
    const float        I3dl2ReverbIncrement_LateReverbHfDecayRatio = 0.01f;
    const nn::TimeSpan I3dl2ReverbIncrement_LateReverbDelayTime = nn::TimeSpan::FromMilliSeconds( 1 );
    const float        I3dl2ReverbIncrement_ReverbGain = 10.0f;
    const float        I3dl2ReverbIncrement_LateReverbDiffusion = 0.1f;
    const float        I3dl2ReverbIncrement_LateReverbDensity = 0.1f;
    const float        I3dl2ReverbIncrement_HfReference = 10.0f;
    const float        I3dl2ReverbIncrement_DryGain = 0.01f;

    //  I3dl2Reverb パラメータの L キーによる加速された増加値 (減少値)
    const float        I3dl2ReverbIncrementBoost_RoomGain = 100.0f;
    const float        I3dl2ReverbIncrementBoost_RoomHfGain = 100.0f;
    const nn::TimeSpan I3dl2ReverbIncrementBoost_ReflectionDelayTime = nn::TimeSpan::FromMilliSeconds( 10 );
    const float        I3dl2ReverbIncrementBoost_ReflectionsGain = 100.0f;
    const nn::TimeSpan I3dl2ReverbIncrementBoost_LateReverbDecayTime = nn::TimeSpan::FromMilliSeconds( 100 );
    const float        I3dl2ReverbIncrementBoost_LateReverbHfDecayRatio = 0.1f;
    const nn::TimeSpan I3dl2ReverbIncrementBoost_LateReverbDelayTime = nn::TimeSpan::FromMilliSeconds( 10 );
    const float        I3dl2ReverbIncrementBoost_ReverbGain = 100.0f;
    const float        I3dl2ReverbIncrementBoost_LateReverbDiffusion = 1.0f;
    const float        I3dl2ReverbIncrementBoost_LateReverbDensity = 1.0f;
    const float        I3dl2ReverbIncrementBoost_HfReference = 100.0f;
    const float        I3dl2ReverbIncrementBoost_DryGain = 0.1f;

    //  I3dl2Reverb パラメータの最小値
    const float        I3dl2ReverbMin_RoomGain = nn::atk::EffectI3dl2Reverb::RoomGainMin;
    const float        I3dl2ReverbMin_RoomHfGain = nn::atk::EffectI3dl2Reverb::RoomHfGainMin;
    const nn::TimeSpan I3dl2ReverbMin_ReflectionDelayTime = nn::atk::EffectI3dl2Reverb::GetReflectionDelayTimeMin();
    const float        I3dl2ReverbMin_ReflectionsGain = nn::atk::EffectI3dl2Reverb::ReflectionsGainMin;
    const nn::TimeSpan I3dl2ReverbMin_LateReverbDecayTime = nn::atk::EffectI3dl2Reverb::GetLateReverbDecayTimeMin();
    const float        I3dl2ReverbMin_LateReverbHfDecayRatio = nn::atk::EffectI3dl2Reverb::LateReverbHfDecayRatioMin;
    const nn::TimeSpan I3dl2ReverbMin_LateReverbDelayTime = nn::atk::EffectI3dl2Reverb::GetLateReverbDelayTimeMin();
    const float        I3dl2ReverbMin_ReverbGain = nn::atk::EffectI3dl2Reverb::ReverbGainMin;
    const float        I3dl2ReverbMin_LateReverbDiffusion = nn::atk::EffectI3dl2Reverb::LateReverbDiffusionMin;
    const float        I3dl2ReverbMin_LateReverbDensity = nn::atk::EffectI3dl2Reverb::LateReverbDensityMin;
    const float        I3dl2ReverbMin_HfReference = nn::atk::EffectI3dl2Reverb::HfReferenceMin;
    const float        I3dl2ReverbMin_DryGain = nn::atk::EffectI3dl2Reverb::DryGainMin;

    //  I3dl2Reverb パラメータの最大値
    const float        I3dl2ReverbMax_RoomGain = nn::atk::EffectI3dl2Reverb::RoomGainMax;
    const float        I3dl2ReverbMax_RoomHfGain = nn::atk::EffectI3dl2Reverb::RoomHfGainMax;
    const nn::TimeSpan I3dl2ReverbMax_ReflectionDelayTime = nn::atk::EffectI3dl2Reverb::GetReflectionDelayTimeMax();
    const float        I3dl2ReverbMax_ReflectionsGain = nn::atk::EffectI3dl2Reverb::ReflectionsGainMax;
    const nn::TimeSpan I3dl2ReverbMax_LateReverbDecayTime = nn::atk::EffectI3dl2Reverb::GetLateReverbDecayTimeMax();
    const float        I3dl2ReverbMax_LateReverbHfDecayRatio = nn::atk::EffectI3dl2Reverb::LateReverbHfDecayRatioMax;
    const nn::TimeSpan I3dl2ReverbMax_LateReverbDelayTime = nn::atk::EffectI3dl2Reverb::GetLateReverbDelayTimeMax();
    const float        I3dl2ReverbMax_ReverbGain = nn::atk::EffectI3dl2Reverb::ReverbGainMax;
    const float        I3dl2ReverbMax_LateReverbDiffusion = nn::atk::EffectI3dl2Reverb::LateReverbDiffusionMax;
    const float        I3dl2ReverbMax_LateReverbDensity = nn::atk::EffectI3dl2Reverb::LateReverbDensityMax;
    const float        I3dl2ReverbMax_HfReference = nn::atk::EffectI3dl2Reverb::HfReferenceMax;
    const float        I3dl2ReverbMax_DryGain = nn::atk::EffectI3dl2Reverb::DryGainMax;
}
namespace
{
    //  フォーカスがあるときとないときの項目の背景色
    const nn::util::Uint8x4 FocusedItemBackColor  = GetUint8x4( 28, 32, 54, 255 );
    const nn::util::Uint8x4 SelectedItemBackColor = GetUint8x4( 14, 16, 27, 255 );
    const nn::util::Uint8x4 ItemBackColor         = GetUint8x4(  8,  8,  8, 255 );

    //  エフェクト効果があるときとないときの項目の文字色
    const nn::util::Unorm8x4 EffectTextColor   = GetUnorm8x4( 64, 192, 96, 255 );
    const nn::util::Unorm8x4 NoEffectTextColor = GetUnorm8x4( 192, 192, 192, 255 );

    //  項目の枠の色, サイズ
    const nn::util::Uint8x4 SelectedItemEdgeColor = GetUint8x4( 192, 128, 64, 255 );
    const nn::util::Uint8x4 ItemEdgeColor         = GetUint8x4(  64,  64, 64, 255 );
    const float ItemEdgeSize = 1.0f;

    //  項目の文字のスケール
    const float ItemTextScaleX = 0.85f;
    const float ItemTextScaleY = 0.9f;

    //  パラメータの文字のスケール
    const float ParameterTextScaleX = 0.8f;
    const float ParameterTextScaleY = 0.85f;
}
namespace
{
    //  パラメータを増減させ、クランプした値を返します
    template <typename T> T ChangeAndClampParameterValue(T value, T increment, T boost, T min, T max, bool isBoost, bool isIncrement, const char* logText)
    {
        return nn::atk::detail::fnd::Clamp( ChangeParameterValue( value, increment, boost, isBoost, isIncrement, "Effect", logText ), min, max );
    }

    //  EarlyMode の表示用テキストを取得します
    const char* GetEarlyModeText(nn::atk::EffectReverb::EarlyMode mode)
    {
        switch( mode )
        {
        case nn::atk::EffectReverb::EarlyMode_Cavern:
            return "Cavern";
        case nn::atk::EffectReverb::EarlyMode_Hall:
            return "Hall";
        case nn::atk::EffectReverb::EarlyMode_LargeRoom:
            return "LargeRoom";
        case nn::atk::EffectReverb::EarlyMode_NoEarlyReflection:
            return "None";
        case nn::atk::EffectReverb::EarlyMode_SmallRoom:
            return "SmallRoom";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
    //  LateMode の表示用テキストを取得します
    const char* GetLateModeText(nn::atk::EffectReverb::LateMode mode)
    {
        switch( mode )
        {
        case nn::atk::EffectReverb::LateMode_Cavern:
            return "Cavern";
        case nn::atk::EffectReverb::LateMode_Hall:
            return "Hall";
        case nn::atk::EffectReverb::LateMode_MaximumDelay:
            return "MaximumDelay";
        case nn::atk::EffectReverb::LateMode_MetalCorridor:
            return "MetalCorridor";
        case nn::atk::EffectReverb::LateMode_Room:
            return "Room";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

//  ------------------------------------------------------------
//
//                          EffectItem
//
//  ------------------------------------------------------------
//  初期化します
void EffectItem::Initialize(float parameterClientAreaSizeX, float labelWidth) NN_NOEXCEPT
{
    m_CurrentEffectType = EffectType_NoEffect;
    m_NextEffectType = EffectType_NoEffect;
    m_IsNeedToAppend = false;

    m_Buffer = nullptr;
    m_BufferSize = 0;

    m_pEffect[EffectType_NoEffect]    = &m_NoEffect;
    m_pEffect[EffectType_Reverb]      = &m_Reverb;
    m_pEffect[EffectType_Delay]       = &m_Delay;
    m_pEffect[EffectType_I3dl2Reverb] = &m_I3dl2Reverb;

    for( int i = 0; i< EffectType_Count; i++ )
    {
        m_pEffect[i]->Initialize( parameterClientAreaSizeX );
        m_pEffect[i]->SetLabelWidth( labelWidth );
    }
}
//  終了します
void EffectItem::Finalize(nn::mem::StandardAllocator* pAllocator) NN_NOEXCEPT
{
    for( int i = 0; i< EffectType_Count; i++ )
    {
        NN_ABORT_UNLESS_NOT_NULL( m_pEffect[i] );
        m_pEffect[i]->Finalize();
    }

    if( m_Buffer != nullptr )
    {
        DetachMemoryPool( &m_MemoryPool );
        NN_ABORT_UNLESS_NOT_NULL( pAllocator );
        pAllocator->Free( m_Buffer );
    }
}

//  更新します
void EffectItem::Update() NN_NOEXCEPT
{
    EffectBase& effect = GetCurrentEffect();
    effect.Update();
    m_IsNeedToAppend = effect.IsNeedToAppend() || m_IsNeedToAppend;
}
//  パラメータを更新します
void EffectItem::UpdateParameters(const HidPad& hidPad) NN_NOEXCEPT
{
    GetCurrentEffect().UpdateParameters( hidPad );
}
//  項目を描画します
void EffectItem::DrawItem(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT
{
    GetCurrentEffect().DrawItem( gfxContext, positionX, positionY, isFocused, isSelected );
}
//  パラメータを描画します
void EffectItem::DrawParameters(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT
{
    GetCurrentEffect().DrawParameters( gfxContext, positionX, positionY, isFocused, isSelected );
}

//  パラメータがあるかどうかを取得します
bool EffectItem::IsHasParameters() const NN_NOEXCEPT
{
    return m_CurrentEffectType != EffectType_NoEffect;
}
//  使用メモリサイズを取得します
size_t EffectItem::GetUsingMemorySize() const NN_NOEXCEPT
{
    return m_BufferSize;
}
//  エフェクトを適用する必要があるかどうかを取得します
bool EffectItem::IsNeedToAppend() const NN_NOEXCEPT
{
    return m_IsNeedToAppend;
}
//  エフェクトを変更します
void EffectItem::ChangeEffect() NN_NOEXCEPT
{
    m_NextEffectType = GetNextValueOnLoopSelection( m_NextEffectType + 1, EffectType_Count );
    NN_ABORT_UNLESS_RANGE( m_NextEffectType, 0, EffectType_Count );

    m_IsNeedToAppend = true;
}
//  エフェクトを適用します
void EffectItem::Append(nn::atk::AuxBus bus, nn::mem::StandardAllocator* pAllocator) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RANGE( bus, 0, nn::atk::AuxBus_Count );
    NN_ABORT_UNLESS_NOT_NULL( pAllocator );

    if( m_IsNeedToAppend )
    {
        m_CurrentEffectType = m_NextEffectType;
        m_IsNeedToAppend = false;

        const size_t requiredMemSize = GetCurrentEffect().GetRequiredMemSize();
        if( requiredMemSize != m_BufferSize )
        {
            if( m_Buffer != nullptr )
            {
                DetachMemoryPool( &m_MemoryPool );
                pAllocator->Free( m_Buffer );
            }

            if( requiredMemSize == 0 )
            {
                m_Buffer = nullptr;
                m_BufferSize = 0;
            }
            else
            {
                //  バッファを要求されていれば取得します
                const size_t memoryPoolMemSize = nn::util::align_up( requiredMemSize, nn::audio::MemoryPoolType::SizeGranularity );
                void* buffer = pAllocator->Allocate( memoryPoolMemSize, nn::audio::MemoryPoolType::AddressAlignment );
                NN_ABORT_UNLESS_NOT_NULL( buffer );

                AttachMemoryPool( &m_MemoryPool, buffer, memoryPoolMemSize );

                m_Buffer = buffer;
                m_BufferSize = requiredMemSize;
            }
        }
    }

    GetCurrentEffect().Append( bus, m_Buffer, m_BufferSize );
}
//  エフェクトを消す準備をします。完了したとき true を返します
bool EffectItem::PrepareToClear() NN_NOEXCEPT
{
    return GetCurrentEffect().PrepareToClear();
}
//  今選択されている EffectBase の参照を取得します
EffectItem::EffectBase& EffectItem::GetCurrentEffect() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RANGE( m_CurrentEffectType, 0, EffectType_Count );
    NN_ABORT_UNLESS_NOT_NULL( m_pEffect[m_CurrentEffectType] );
    return *m_pEffect[m_CurrentEffectType];
}


//  ------------------------------------------------------------
//
//                          EffectBase
//
//  ------------------------------------------------------------
//  コンストラクタ
EffectItem::EffectBase::EffectBase(const char* itemText, const nn::util::Unorm8x4& itemColor) NN_NOEXCEPT
{
    m_ItemLabl.SetText( itemText );
    m_ItemLabl.SetColor( itemColor );
    m_ItemLabl.SetEdgeSize( ItemEdgeSize );
    m_ItemLabl.SetEdgeColor( ItemEdgeColor );
    m_ItemLabl.SetScale( ItemTextScaleX, ItemTextScaleY );
}
//  描画します
void EffectItem::EffectBase::DrawItem(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT
{
    if( isSelected )
    {
        m_ItemLabl.SetEdgeColor( SelectedItemEdgeColor );
        if( isFocused )
        {
            m_ItemLabl.SetBackColor( FocusedItemBackColor );
        }
        else
        {
            m_ItemLabl.SetBackColor( SelectedItemBackColor );
        }
    }
    else
    {
        m_ItemLabl.SetEdgeColor( ItemEdgeColor );
        m_ItemLabl.SetBackColor( ItemBackColor );
    }

    m_ItemLabl.SetPosition( positionX, positionY );
    m_ItemLabl.Draw( gfxContext );
}
//  ラベルの幅を設定します
void EffectItem::EffectBase::SetLabelWidth(float width)
{
    m_ItemLabl.SetDrawWidth( width );
}


//  ------------------------------------------------------------
//
//                          NoEffect
//
//  ------------------------------------------------------------
//  コンストラクタ
EffectItem::NoEffect::NoEffect() NN_NOEXCEPT
    : EffectBase( "None", NoEffectTextColor )
{
    m_ParamLabel.SetText( "No Parameters" );
    m_ParamLabel.SetScale( ParameterTextScaleX, ParameterTextScaleY );
}
//  初期化します
void EffectItem::NoEffect::Initialize(float parameterClientAreaSizeX) NN_NOEXCEPT
{
    NN_UNUSED( parameterClientAreaSizeX );
}
//  終了します
void EffectItem::NoEffect::Finalize() NN_NOEXCEPT
{
}

//  エフェクトを適用します
void EffectItem::NoEffect::Append(nn::atk::AuxBus bus, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    FlightRecorder::GetInstance().WriteLog( "[Effect] Append NoEffect (bus, buf, size)=(%d, %p, %d)", bus, buffer, bufferSize );
    NN_UNUSED( bus );
    NN_UNUSED( buffer );
    NN_UNUSED( bufferSize );
}
//  更新します
void EffectItem::NoEffect::Update() NN_NOEXCEPT
{
}
//  パラメータを更新します
void EffectItem::NoEffect::UpdateParameters(const HidPad& hidPad) NN_NOEXCEPT
{
    NN_UNUSED( hidPad );
}
//  パラメータを描画します
void EffectItem::NoEffect::DrawParameters(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT
{
    NN_UNUSED( isFocused );
    NN_UNUSED( isSelected );

    m_ParamLabel.SetPosition( positionX, positionY );
    m_ParamLabel.Draw( gfxContext );
}

//  エフェクトを無効化します。完了したとき true を返します
bool EffectItem::NoEffect::PrepareToClear() NN_NOEXCEPT
{
    return true;
}
//  使用するメモリ量を取得します
size_t EffectItem::NoEffect::GetRequiredMemSize() NN_NOEXCEPT
{
    return 0;
}
//  エフェクトを適用する必要があるかどうかを取得します
bool EffectItem::NoEffect::IsNeedToAppend() const NN_NOEXCEPT
{
    return false;
}


//  ------------------------------------------------------------
//
//                          Reverb
//
//  ------------------------------------------------------------
//  コンストラクタ
EffectItem::Reverb::Reverb() NN_NOEXCEPT
    : EffectBase( "Reverb", EffectTextColor )
{
    m_Reverb.SetEnabled( false );
}
//  初期化します
void EffectItem::Reverb::Initialize(float parameterClientAreaSizeX) NN_NOEXCEPT
{
    m_ParamList.Initialize( parameterClientAreaSizeX );
    for( int i = 0; i < ParamType_Count; i++ )
    {
        m_ParamList.GetNameLabel( i ).SetScale( ParameterTextScaleX, ParameterTextScaleY );
        m_ParamList.GetValueLabel( i ).SetScale( ParameterTextScaleX, ParameterTextScaleY );
    }

    m_ParamList.GetNameLabel( ParamType_OutGain ).SetText( "OutGain" );
    m_ParamList.GetNameLabel( ParamType_ReverbGain ).SetText( "ReverbGain" );
    m_ParamList.GetNameLabel( ParamType_EarlyGain ).SetText( "EarlyGain" );
    m_ParamList.GetNameLabel( ParamType_DryGain ).SetText( "DryGain" );
    m_ParamList.GetNameLabel( ParamType_LateGain ).SetText( "LateGain" );
    m_ParamList.GetNameLabel( ParamType_EarlyMode ).SetText( "EarlyMode" );
    m_ParamList.GetNameLabel( ParamType_LateMode ).SetText( "LateMode" );
    m_ParamList.GetNameLabel( ParamType_DecayTime ).SetText( "DecayTime" );
    m_ParamList.GetNameLabel( ParamType_PredelayTime ).SetText( "PredelayTime" );
    m_ParamList.GetNameLabel( ParamType_Coloration ).SetText( "Coloration" );
    m_ParamList.GetNameLabel( ParamType_HighFrequencyDecayRatio ).SetText( "HighFrequencyDecayRatio" );
}
//  終了します
void EffectItem::Reverb::Finalize() NN_NOEXCEPT
{
    if( m_Reverb.IsEnabled() )
    {
        m_Reverb.SetEnabled( false );
        while( !m_Reverb.IsRemovable() )
        {
            //  エフェクトを消すことができるようになるまで待ちます
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 16 ) );
        }
    }
}

//  エフェクトを適用します
void EffectItem::Reverb::Append(nn::atk::AuxBus bus, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    FlightRecorder::GetInstance().WriteLog( "[Effect] Append Reverb : (bus, buf, size)=(%d, %p, %d)", bus, buffer, bufferSize );
    NN_ABORT_UNLESS_NOT_NULL( buffer );
    NN_ABORT_UNLESS_RANGE( bus, 0, nn::atk::AuxBus_Count );

    m_Reverb.SetEnabled( true );
    const bool isSuccess = nn::atk::SoundSystem::AppendEffect( bus, &m_Reverb, buffer, bufferSize );
    NN_ABORT_UNLESS( isSuccess );
}
//  更新します
void EffectItem::Reverb::Update() NN_NOEXCEPT
{
    m_ParamList.GetValueLabel( ParamType_OutGain ).SetText( "%.2f", m_Reverb.GetOutGain() );
    m_ParamList.GetValueLabel( ParamType_ReverbGain ).SetText( "%.2f", m_Reverb.GetReverbGain() );
    m_ParamList.GetValueLabel( ParamType_EarlyGain ).SetText( "%.2f", m_Reverb.GetEarlyGain() );
    m_ParamList.GetValueLabel( ParamType_DryGain ).SetText( "%.2f", m_Reverb.GetDryGain() );
    m_ParamList.GetValueLabel( ParamType_LateGain ).SetText( "%.2f", m_Reverb.GetLateGain() );
    m_ParamList.GetValueLabel( ParamType_EarlyMode ).SetText( GetEarlyModeText( m_Reverb.GetEarlyMode() ) );
    m_ParamList.GetValueLabel( ParamType_LateMode ).SetText( GetLateModeText( m_Reverb.GetLateMode() ) );
    m_ParamList.GetValueLabel( ParamType_DecayTime ).SetText( "%dms", m_Reverb.GetDecayTime().GetMilliSeconds() );
    m_ParamList.GetValueLabel( ParamType_PredelayTime ).SetText( "%dms", m_Reverb.GetPredelayTime().GetMilliSeconds() );
    m_ParamList.GetValueLabel( ParamType_Coloration ).SetText( "%.2f", m_Reverb.GetColoration() );
    m_ParamList.GetValueLabel( ParamType_HighFrequencyDecayRatio ).SetText( "%.2f", m_Reverb.GetHighFrequencyDecayRatio() );
}
//  パラメータを更新します
void EffectItem::Reverb::UpdateParameters(const HidPad& hidPad) NN_NOEXCEPT
{
    m_ParamList.UpdateByHid( hidPad );

    const bool isLeftContinue = hidPad.IsContinue( HidPad::Button_Left );
    const bool isRightContinue = hidPad.IsContinue( HidPad::Button_Right );
    if( isLeftContinue || isRightContinue )
    {
        const bool isBoost = hidPad.IsHold( HidPad::Button_L );
        switch( m_ParamList.GetSelectedItemIndex() )
        {
        case ParamType_OutGain:
            m_Reverb.SetOutGain( ChangeAndClampParameterValue( m_Reverb.GetOutGain(), ReverbIncrement_OutGain, ReverbIncrementBoost_OutGain, ReverbMin_OutGain, ReverbMax_OutGain, isBoost, isRightContinue, "Reverb.OutGain" ) );
            break;
        case ParamType_ReverbGain:
            m_Reverb.SetReverbGain( ChangeAndClampParameterValue( m_Reverb.GetReverbGain(), ReverbIncrement_ReverbGain, ReverbIncrementBoost_ReverbGain, ReverbMin_ReverbGain, ReverbMax_ReverbGain, isBoost, isRightContinue, "Reverb.ReverbGain" ) );
            break;
        case ParamType_EarlyGain:
            m_Reverb.SetEarlyGain( ChangeAndClampParameterValue( m_Reverb.GetEarlyGain(), ReverbIncrement_EarlyGain, ReverbIncrementBoost_EarlyGain, ReverbMin_EarlyGain, ReverbMax_EarlyGain, isBoost, isRightContinue, "Reverb.EarlyGain" ) );
            break;
        case ParamType_DryGain:
            m_Reverb.SetDryGain( ChangeAndClampParameterValue( m_Reverb.GetDryGain(), ReverbIncrement_DryGain, ReverbIncrementBoost_DryGain, ReverbMin_DryGain, ReverbMax_DryGain, isBoost, isRightContinue, "Reverb.DryGain" ) );
            break;
        case ParamType_LateGain:
            m_Reverb.SetLateGain( ChangeAndClampParameterValue( m_Reverb.GetLateGain(), ReverbIncrement_LateGain, ReverbIncrementBoost_LateGain, ReverbMin_LateGain, ReverbMax_LateGain, isBoost, isRightContinue, "Reverb.LateGain" ) );
            break;
        case ParamType_EarlyMode:
            m_Reverb.SetEarlyMode( static_cast<nn::atk::EffectReverb::EarlyMode>( ChangeAndClampParameterValue<int>( m_Reverb.GetEarlyMode(), ReverbIncrement_EarlyMode, ReverbIncrementBoost_EarlyMode, ReverbMin_EarlyMode, ReverbMax_EarlyMode, isBoost, isRightContinue, "Reverb.EarlyMode" ) ) );
            break;
        case ParamType_LateMode:
            m_Reverb.SetLateMode( static_cast<nn::atk::EffectReverb::LateMode>( ChangeAndClampParameterValue<int>( m_Reverb.GetLateMode(), ReverbIncrement_LateMode, ReverbIncrementBoost_LateMode, ReverbMin_LateMode, ReverbMax_LateMode, isBoost, isRightContinue, "Reverb.LateMode" ) ) );
            break;
        case ParamType_DecayTime:
            m_Reverb.SetDecayTime( ChangeAndClampParameterValue( m_Reverb.GetDecayTime(), ReverbIncrement_DecayTime, ReverbIncrementBoost_DecayTime, ReverbMin_DecayTime, ReverbMax_DecayTime, isBoost, isRightContinue, "Reverb.DecayTime" ) );
            break;
        case ParamType_PredelayTime:
            m_Reverb.SetPredelayTime( ChangeAndClampParameterValue( m_Reverb.GetPredelayTime(), ReverbIncrement_PredelayTime, ReverbIncrementBoost_PredelayTime, ReverbMin_PredelayTime, ReverbMax_PredelayTime, isBoost, isRightContinue, "Reverb.PredelayTime" ) );
            break;
        case ParamType_Coloration:
            m_Reverb.SetColoration( ChangeAndClampParameterValue( m_Reverb.GetColoration(), ReverbIncrement_Coloration, ReverbIncrementBoost_Coloration, ReverbMin_Coloration, ReverbMax_Coloration, isBoost, isRightContinue, "Reverb.Coloration" ) );
            break;
        case ParamType_HighFrequencyDecayRatio:
            m_Reverb.SetHighFrequencyDecayRatio( ChangeAndClampParameterValue( m_Reverb.GetHighFrequencyDecayRatio(), ReverbIncrement_HighFrequencyDecayRatio, ReverbIncrementBoost_HighFrequencyDecayRatio, ReverbMin_HighFrequencyDecayRatio, ReverbMax_HighFrequencyDecayRatio, isBoost, isRightContinue, "Reverb.HighFrequencyDecayRatio" ) );
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}
//  パラメータを描画します
void EffectItem::Reverb::DrawParameters(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT
{
    m_ParamList.SetFocused( isFocused && isSelected );
    m_ParamList.Draw( gfxContext, positionX, positionY );
}

//  エフェクトを無効化します。完了したとき true を返します
bool EffectItem::Reverb::PrepareToClear() NN_NOEXCEPT
{
    if( m_Reverb.IsEnabled() )
    {
        m_Reverb.SetEnabled( false );
    }

    return m_Reverb.IsRemovable();
}
//  エフェクトを適用する必要があるかどうかを取得します
bool EffectItem::Reverb::IsNeedToAppend() const NN_NOEXCEPT
{
    return false;
}
//  使用するメモリ量を取得します
size_t EffectItem::Reverb::GetRequiredMemSize() NN_NOEXCEPT
{
    return m_Reverb.GetRequiredMemSize();
}


//  ------------------------------------------------------------
//
//                          Delay
//
//  ------------------------------------------------------------
//  コンストラクタ
EffectItem::Delay::Delay() NN_NOEXCEPT
    : EffectBase( "Delay", EffectTextColor )
{
    m_Delay.SetEnabled( false );
    m_DelayTimeMax = m_Delay.GetDelayTimeMax();
}
//  初期化します
void EffectItem::Delay::Initialize(float parameterClientAreaSizeX) NN_NOEXCEPT
{
    m_ParamList.Initialize( parameterClientAreaSizeX );
    for( int i = 0; i < ParamType_Count; i++ )
    {
        m_ParamList.GetNameLabel( i ).SetScale( ParameterTextScaleX, ParameterTextScaleY );
        m_ParamList.GetValueLabel( i ).SetScale( ParameterTextScaleX, ParameterTextScaleY );
    }

    m_ParamList.GetNameLabel( ParamType_DelayTime ).SetText( "DelayTime" );
    m_ParamList.GetNameLabel( ParamType_DelayTimeMax ).SetText( "DelayTimeMax" );
    m_ParamList.GetNameLabel( ParamType_InGain ).SetText( "InGain" );
    m_ParamList.GetNameLabel( ParamType_DryGain ).SetText( "DryGain" );
    m_ParamList.GetNameLabel( ParamType_FeedbackGain ).SetText( "FeedbackGain" );
    m_ParamList.GetNameLabel( ParamType_LowPassAmount ).SetText( "LowPassAmount" );
}
//  終了します
void EffectItem::Delay::Finalize() NN_NOEXCEPT
{
    if( m_Delay.IsEnabled() )
    {
        m_Delay.SetEnabled( false );
        while( !m_Delay.IsRemovable() )
        {
            //  エフェクトを消すことができるようになるまで待ちます
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 16 ) );
        }
    }
}

//  エフェクトを適用します
void EffectItem::Delay::Append(nn::atk::AuxBus bus, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    FlightRecorder::GetInstance().WriteLog( "[Effect] Append Delay : (bus, buf, size)=(%d, %p, %d)", bus, buffer, bufferSize );

    //  DelayTimeMax が 0 のとき、要求バッファは 0 になります。
    //  そのとき、buffer には nullptr が設定されるため分岐を行います。
    if( buffer == nullptr )
    {
        NN_ABORT_UNLESS_EQUAL( bufferSize, 0u );
        NN_ABORT_UNLESS_EQUAL( m_Delay.GetDelayTimeMax(), nn::TimeSpan::FromMilliSeconds( 0 ) );
    }
    else
    {
        NN_ABORT_UNLESS_RANGE( bus, 0, nn::atk::AuxBus_Count );
        m_Delay.SetEnabled( true );
        const bool isSuccess = nn::atk::SoundSystem::AppendEffect( bus, &m_Delay, buffer, bufferSize );
        NN_ABORT_UNLESS( isSuccess );
    }
}
//  更新します
void EffectItem::Delay::Update() NN_NOEXCEPT
{
    m_ParamList.GetValueLabel( ParamType_DelayTime ).SetText( "%dms", m_Delay.GetDelayTime().GetMilliSeconds() );
    m_ParamList.GetValueLabel( ParamType_DelayTimeMax ).SetText( "%dms", m_Delay.GetDelayTimeMax().GetMilliSeconds() );
    m_ParamList.GetValueLabel( ParamType_InGain ).SetText( "%.2f", m_Delay.GetInGain() );
    m_ParamList.GetValueLabel( ParamType_DryGain ).SetText( "%.2f", m_Delay.GetDryGain() );
    m_ParamList.GetValueLabel( ParamType_FeedbackGain ).SetText( "%.2f", m_Delay.GetFeedbackGain() );
    m_ParamList.GetValueLabel( ParamType_LowPassAmount ).SetText( "%.2f", m_Delay.GetLowPassAmount() );
}

//  パラメータを更新します
void EffectItem::Delay::UpdateParameters(const HidPad& hidPad) NN_NOEXCEPT
{
    m_ParamList.UpdateByHid( hidPad );

    const bool isLeftContinue = hidPad.IsContinue( HidPad::Button_Left );
    const bool isRightContinue = hidPad.IsContinue( HidPad::Button_Right );
    if( isLeftContinue || isRightContinue )
    {
        const bool isBoost = hidPad.IsHold( HidPad::Button_L );
        switch( m_ParamList.GetSelectedItemIndex() )
        {
        case ParamType_DelayTime:
            m_Delay.SetDelayTime( ChangeAndClampParameterValue( m_Delay.GetDelayTime(), DelayIncrement_DelayTime, DelayIncrementBoost_DelayTime, DelayMin_DelayTime, m_Delay.GetDelayTimeMax(), isBoost, isRightContinue, "Delay.DelayTime" ) );
            break;
        case ParamType_DelayTimeMax:
            m_DelayTimeMax = ChangeAndClampParameterValue( m_Delay.GetDelayTimeMax(), DelayIncrement_DelayTimeMax, DelayIncrementBoost_DelayTimeMax, DelayMin_DelayTimeMax, DelayMax_DelayTimeMax, isBoost, isRightContinue, "Delay.DelayTimeMax" );

            if( m_DelayTimeMax < m_Delay.GetDelayTime() )
            {
                m_Delay.SetDelayTime( m_DelayTimeMax );
            }
            break;
        case ParamType_InGain:
            m_Delay.SetInGain( ChangeAndClampParameterValue( m_Delay.GetInGain(), DelayIncrement_InGain, DelayIncrementBoost_InGain, DelayMin_InGain, DelayMax_InGain, isBoost, isRightContinue, "Delay.InGain" ) );
            break;
        case ParamType_DryGain:
            m_Delay.SetDryGain( ChangeAndClampParameterValue( m_Delay.GetDryGain(), DelayIncrement_DryGain, DelayIncrementBoost_DryGain, DelayMin_DryGain, DelayMax_DryGain, isBoost, isRightContinue, "Delay.DryGain" ) );
            break;
        case ParamType_FeedbackGain:
            m_Delay.SetFeedbackGain( ChangeAndClampParameterValue( m_Delay.GetFeedbackGain(), DelayIncrement_FeedbackGain, DelayIncrementBoost_FeedbackGain, DelayMin_FeedbackGain, DelayMax_FeedbackGain, isBoost, isRightContinue, "Delay.FeedbackGain" ) );
            break;
        case ParamType_LowPassAmount:
            m_Delay.SetLowPassAmount( ChangeAndClampParameterValue( m_Delay.GetLowPassAmount(), DelayIncrement_LowPassAmount, DelayIncrementBoost_LowPassAmount, DelayMin_LowPassAmount, DelayMax_LowPassAmount, isBoost, isRightContinue, "Delay.LowPassAmount" ) );
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}
//  パラメータを描画します
void EffectItem::Delay::DrawParameters(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT
{
    m_ParamList.SetFocused( isFocused && isSelected );
    m_ParamList.Draw( gfxContext, positionX, positionY );
}

//  エフェクトを無効化します。完了したとき true を返します
bool EffectItem::Delay::PrepareToClear() NN_NOEXCEPT
{
    //  以下の if 文の条件を満たすとき、EffectItem::Delay::Append() で
    //  エフェクトは Append されていないため、すぐに true を返すことができます
    if( m_Delay.GetDelayTimeMax() == nn::TimeSpan::FromMilliSeconds( 0 ) )
    {
        return true;
    }

    if( m_Delay.IsEnabled() )
    {
        m_Delay.SetEnabled( false );
    }

    return m_Delay.IsRemovable();
}
//  エフェクトを適用する必要があるかどうかを取得します
bool EffectItem::Delay::IsNeedToAppend() const NN_NOEXCEPT
{
    return m_DelayTimeMax != m_Delay.GetDelayTimeMax();
}
//  使用するメモリ量を取得します
size_t EffectItem::Delay::GetRequiredMemSize() NN_NOEXCEPT
{
    m_Delay.SetDelayTimeMax( m_DelayTimeMax );
    return m_Delay.GetRequiredMemSize();
}



//  ------------------------------------------------------------
//
//                          I3dl2Reverb
//
//  ------------------------------------------------------------
//  コンストラクタ
EffectItem::I3dl2Reverb::I3dl2Reverb() NN_NOEXCEPT
    : EffectBase( "I3dl2", EffectTextColor )
{
    m_I3dl2Reverb.SetEnabled( false );
}
//  初期化します
void EffectItem::I3dl2Reverb::Initialize(float parameterClientAreaSizeX) NN_NOEXCEPT
{
    m_ParamList.Initialize( parameterClientAreaSizeX );
    for( int i = 0; i < ParamType_Count; i++ )
    {
        m_ParamList.GetNameLabel( i ).SetScale( ParameterTextScaleX, ParameterTextScaleY );
        m_ParamList.GetValueLabel( i ).SetScale( ParameterTextScaleX, ParameterTextScaleY );
    }

    m_ParamList.GetNameLabel( ParamType_RoomGain ).SetText( "RoomGain" );
    m_ParamList.GetNameLabel( ParamType_RoomHfGain ).SetText( "RoomHfGain" );
    m_ParamList.GetNameLabel( ParamType_ReflectionDelayTime ).SetText( "ReflectionDelayTime" );
    m_ParamList.GetNameLabel( ParamType_ReflectionsGain ).SetText( "ReflectionsGain" );
    m_ParamList.GetNameLabel( ParamType_LateReverbDecayTime ).SetText( "LateReverbDecayTime" );
    m_ParamList.GetNameLabel( ParamType_LateReverbHfDecayRatio ).SetText( "LateReverbHfDecayRatio" );
    m_ParamList.GetNameLabel( ParamType_LateReverbDelayTime ).SetText( "LateReverbDelayTime" );
    m_ParamList.GetNameLabel( ParamType_ReverbGain ).SetText( "ReverbGain" );
    m_ParamList.GetNameLabel( ParamType_LateReverbDiffusion ).SetText( "LateReverbDiffusion" );
    m_ParamList.GetNameLabel( ParamType_LateReverbDensity ).SetText( "LateReverbDensity" );
    m_ParamList.GetNameLabel( ParamType_HfReference ).SetText( "HfReference" );
    m_ParamList.GetNameLabel( ParamType_DryGain ).SetText( "DryGain" );
}
//  終了します
void EffectItem::I3dl2Reverb::Finalize() NN_NOEXCEPT
{
    if( m_I3dl2Reverb.IsEnabled() )
    {
        m_I3dl2Reverb.SetEnabled( false );
        while( !m_I3dl2Reverb.IsRemovable() )
        {
            //  エフェクトを消すことができるようになるまで待ちます
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 16 ) );
        }
    }
}

//  エフェクトを適用します
void EffectItem::I3dl2Reverb::Append(nn::atk::AuxBus bus, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    FlightRecorder::GetInstance().WriteLog( "[Effect] Append I3dl2Reverb : (bus, buf, size)=(%d, %p, %d)", bus, buffer, bufferSize );
    NN_ABORT_UNLESS_NOT_NULL( buffer );
    NN_ABORT_UNLESS_RANGE( bus, 0, nn::atk::AuxBus_Count );

    m_I3dl2Reverb.SetEnabled( true );
    const bool isSuccess = nn::atk::SoundSystem::AppendEffect( bus, &m_I3dl2Reverb, buffer, bufferSize );
    NN_ABORT_UNLESS( isSuccess );
}
//  更新します
void EffectItem::I3dl2Reverb::Update() NN_NOEXCEPT
{
    m_ParamList.GetValueLabel( ParamType_RoomGain ).SetText( "%.2f", m_I3dl2Reverb.GetRoomGain() );
    m_ParamList.GetValueLabel( ParamType_RoomHfGain ).SetText( "%.2f", m_I3dl2Reverb.GetRoomHfGain() );
    m_ParamList.GetValueLabel( ParamType_ReflectionDelayTime ).SetText( "%dms", m_I3dl2Reverb.GetReflectionDelayTime().GetMilliSeconds() );
    m_ParamList.GetValueLabel( ParamType_ReflectionsGain ).SetText( "%.2f", m_I3dl2Reverb.GetReflectionsGain() );
    m_ParamList.GetValueLabel( ParamType_LateReverbDecayTime ).SetText( "%dms", m_I3dl2Reverb.GetLateReverbDecayTime().GetMilliSeconds() );
    m_ParamList.GetValueLabel( ParamType_LateReverbHfDecayRatio ).SetText( "%.2f", m_I3dl2Reverb.GetLateReverbHfDecayRatio() );
    m_ParamList.GetValueLabel( ParamType_LateReverbDelayTime ).SetText( "%dms", m_I3dl2Reverb.GetLateReverbDelayTime().GetMilliSeconds() );
    m_ParamList.GetValueLabel( ParamType_ReverbGain ).SetText( "%.2f", m_I3dl2Reverb.GetReverbGain() );
    m_ParamList.GetValueLabel( ParamType_LateReverbDiffusion ).SetText( "%.2f", m_I3dl2Reverb.GetLateReverbDiffusion() );
    m_ParamList.GetValueLabel( ParamType_LateReverbDensity ).SetText( "%.2f", m_I3dl2Reverb.GetLateReverbDensity() );
    m_ParamList.GetValueLabel( ParamType_HfReference ).SetText( "%.2f", m_I3dl2Reverb.GetHfReference() );
    m_ParamList.GetValueLabel( ParamType_DryGain ).SetText( "%.2f", m_I3dl2Reverb.GetDryGain() );
}
//  パラメータを更新します
void EffectItem::I3dl2Reverb::UpdateParameters(const HidPad& hidPad) NN_NOEXCEPT
{
    m_ParamList.UpdateByHid( hidPad );

    const bool isLeftContinue = hidPad.IsContinue( HidPad::Button_Left );
    const bool isRightContinue = hidPad.IsContinue( HidPad::Button_Right );
    if( isLeftContinue || isRightContinue )
    {
        const bool isBoost = hidPad.IsHold( HidPad::Button_L );
        switch( m_ParamList.GetSelectedItemIndex() )
        {
        case ParamType_RoomGain:
            m_I3dl2Reverb.SetRoomGain( ChangeAndClampParameterValue( m_I3dl2Reverb.GetRoomGain(), I3dl2ReverbIncrement_RoomGain, I3dl2ReverbIncrementBoost_RoomGain, I3dl2ReverbMin_RoomGain, I3dl2ReverbMax_RoomGain, isBoost, isRightContinue, "I3dl2Reverb.RoomGain" ) );
            break;
        case ParamType_RoomHfGain:
            m_I3dl2Reverb.SetRoomHfGain( ChangeAndClampParameterValue( m_I3dl2Reverb.GetRoomHfGain(), I3dl2ReverbIncrement_RoomHfGain, I3dl2ReverbIncrementBoost_RoomHfGain, I3dl2ReverbMin_RoomHfGain, I3dl2ReverbMax_RoomHfGain, isBoost, isRightContinue, "I3dl2Reverb.RoomHfGain" ) );
            break;
        case ParamType_ReflectionDelayTime:
            m_I3dl2Reverb.SetReflectionDelayTime( ChangeAndClampParameterValue( m_I3dl2Reverb.GetReflectionDelayTime(), I3dl2ReverbIncrement_ReflectionDelayTime, I3dl2ReverbIncrementBoost_ReflectionDelayTime, I3dl2ReverbMin_ReflectionDelayTime, I3dl2ReverbMax_ReflectionDelayTime, isBoost, isRightContinue, "I3dl2Reverb.ReflectionDelayTime" ) );
            break;
        case ParamType_ReflectionsGain:
            m_I3dl2Reverb.SetReflectionsGain( ChangeAndClampParameterValue( m_I3dl2Reverb.GetReflectionsGain(), I3dl2ReverbIncrement_ReflectionsGain, I3dl2ReverbIncrementBoost_ReflectionsGain, I3dl2ReverbMin_ReflectionsGain, I3dl2ReverbMax_ReflectionsGain, isBoost, isRightContinue, "I3dl2Reverb.ReflectionsGain" ) );
            break;
        case ParamType_LateReverbDecayTime:
            m_I3dl2Reverb.SetLateReverbDecayTime( ChangeAndClampParameterValue( m_I3dl2Reverb.GetLateReverbDecayTime(), I3dl2ReverbIncrement_LateReverbDecayTime, I3dl2ReverbIncrementBoost_LateReverbDecayTime, I3dl2ReverbMin_LateReverbDecayTime, I3dl2ReverbMax_LateReverbDecayTime, isBoost, isRightContinue, "I3dl2Reverb.LateReverbDecayTime" ) );
            break;
        case ParamType_LateReverbHfDecayRatio:
            m_I3dl2Reverb.SetLateReverbHfDecayRatio( ChangeAndClampParameterValue( m_I3dl2Reverb.GetLateReverbHfDecayRatio(), I3dl2ReverbIncrement_LateReverbHfDecayRatio, I3dl2ReverbIncrementBoost_LateReverbHfDecayRatio, I3dl2ReverbMin_LateReverbHfDecayRatio, I3dl2ReverbMax_LateReverbHfDecayRatio, isBoost, isRightContinue, "I3dl2Reverb.LateReverbHfDecayRatio" ) );
            break;
        case ParamType_LateReverbDelayTime:
            m_I3dl2Reverb.SetLateReverbDelayTime( ChangeAndClampParameterValue( m_I3dl2Reverb.GetLateReverbDelayTime(), I3dl2ReverbIncrement_LateReverbDelayTime, I3dl2ReverbIncrementBoost_LateReverbDelayTime, I3dl2ReverbMin_LateReverbDelayTime, I3dl2ReverbMax_LateReverbDelayTime, isBoost, isRightContinue, "I3dl2Reverb.LateReverbDelayTime" ) );
            break;
        case ParamType_ReverbGain:
            m_I3dl2Reverb.SetReverbGain( ChangeAndClampParameterValue( m_I3dl2Reverb.GetReverbGain(), I3dl2ReverbIncrement_ReverbGain, I3dl2ReverbIncrementBoost_ReverbGain, I3dl2ReverbMin_ReverbGain, I3dl2ReverbMax_ReverbGain, isBoost, isRightContinue, "I3dl2Reverb.ReverbGain" ) );
            break;
        case ParamType_LateReverbDiffusion:
            m_I3dl2Reverb.SetLateReverbDiffusion( ChangeAndClampParameterValue( m_I3dl2Reverb.GetLateReverbDiffusion(), I3dl2ReverbIncrement_LateReverbDiffusion, I3dl2ReverbIncrementBoost_LateReverbDiffusion, I3dl2ReverbMin_LateReverbDiffusion, I3dl2ReverbMax_LateReverbDiffusion, isBoost, isRightContinue, "I3dl2Reverb.LateReverbDiffusion" ) );
            break;
        case ParamType_LateReverbDensity:
            m_I3dl2Reverb.SetLateReverbDensity( ChangeAndClampParameterValue( m_I3dl2Reverb.GetLateReverbDensity(), I3dl2ReverbIncrement_LateReverbDensity, I3dl2ReverbIncrementBoost_LateReverbDensity, I3dl2ReverbMin_LateReverbDensity, I3dl2ReverbMax_LateReverbDensity, isBoost, isRightContinue, "I3dl2Reverb.LateReverbDensity" ) );
            break;
        case ParamType_HfReference:
            m_I3dl2Reverb.SetHfReference( ChangeAndClampParameterValue( m_I3dl2Reverb.GetHfReference(), I3dl2ReverbIncrement_HfReference, I3dl2ReverbIncrementBoost_HfReference, I3dl2ReverbMin_HfReference, I3dl2ReverbMax_HfReference, isBoost, isRightContinue, "I3dl2Reverb.HfReference" ) );
            break;
        case ParamType_DryGain:
            m_I3dl2Reverb.SetDryGain( ChangeAndClampParameterValue( m_I3dl2Reverb.GetDryGain(), I3dl2ReverbIncrement_DryGain, I3dl2ReverbIncrementBoost_DryGain, I3dl2ReverbMin_DryGain, I3dl2ReverbMax_DryGain, isBoost, isRightContinue, "I3dl2Reverb.DryGain" ) );
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}
//  パラメータを描画します
void EffectItem::I3dl2Reverb::DrawParameters(gfxutil::GfxContext& gfxContext, float positionX, float positionY, bool isFocused, bool isSelected) NN_NOEXCEPT
{
    m_ParamList.SetFocused( isFocused && isSelected );
    m_ParamList.Draw( gfxContext, positionX, positionY );
}

//  エフェクトを無効化します。完了したとき true を返します
bool EffectItem::I3dl2Reverb::PrepareToClear() NN_NOEXCEPT
{
    if( m_I3dl2Reverb.IsEnabled() )
    {
        m_I3dl2Reverb.SetEnabled( false );
    }

    return m_I3dl2Reverb.IsRemovable();
}
//  エフェクトを適用する必要があるかどうかを取得します
bool EffectItem::I3dl2Reverb::IsNeedToAppend() const NN_NOEXCEPT
{
    return false;
}
//  使用するメモリ量を取得します
size_t EffectItem::I3dl2Reverb::GetRequiredMemSize() NN_NOEXCEPT
{
    return m_I3dl2Reverb.GetRequiredMemSize();
}

