﻿/*---------------------------------------------------------------------------*
  Project:  NintendoWare

  Copyright (C)Nintendo/HAL Laboratory, Inc.  All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *---------------------------------------------------------------------------*/

uint GetBits(
    const in uint bit,
    const in uint pos,
    const in uint len)
{
   uint mask = ~(0xFFFFFFFFU << len);
   return (bit >> pos) & mask;
}

uint GetDetailedCombinerRgb(const in ivec4 bit)
{
    return GetBits(bit.x, 24, 4);
}

uint GetDetailedCombinerAlpha(const in ivec4 bit)
{
    return GetBits(bit.y, 24, 4);
}

uint GetDetailedCombinerSrcRgb0(const in ivec4 bit)
{
    return GetBits(bit.x, 0, 4);
}

uint GetDetailedCombinerSrcRgb1(const in ivec4 bit)
{
    return GetBits(bit.x, 4, 4);
}

uint GetDetailedCombinerSrcRgb2(const in ivec4 bit)
{
    return GetBits(bit.x, 8, 4);
}

uint GetDetailedCombinerSrcAlpha0(const in ivec4 bit)
{
    return GetBits(bit.y, 0, 4);
}

uint GetDetailedCombinerSrcAlpha1(const in ivec4 bit)
{
    return GetBits(bit.y, 4, 4);
}

uint GetDetailedCombinerSrcAlpha2(const in ivec4 bit)
{
    return GetBits(bit.y, 8, 4);
}

uint GetDetailedCombinerOperandRgb0(const in ivec4 bit)
{
    return GetBits(bit.x, 12, 4);
}

uint GetDetailedCombinerOperandRgb1(const in ivec4 bit)
{
    return GetBits(bit.x, 16, 4);
}

uint GetDetailedCombinerOperandRgb2(const in ivec4 bit)
{
    return GetBits(bit.x, 20, 4);
}

uint GetDetailedCombinerOperandAlpha0(const in ivec4 bit)
{
    return GetBits(bit.y, 12, 4);
}

uint GetDetailedCombinerOperandAlpha1(const in ivec4 bit)
{
    return GetBits(bit.y, 16, 4);
}

uint GetDetailedCombinerOperandAlpha2(const in ivec4 bit)
{
    return GetBits(bit.y, 20, 4);
}

uint GetDetailedCombinerScaleRgb(const in ivec4 bit)
{
    return GetBits(bit.x, 28, 2);
}

uint GetDetailedCombinerScaleAlpha(const in ivec4 bit)
{
    return GetBits(bit.y, 28, 2);
}

uint GetDetailedCombinerConstantRgb(const in ivec4 bit)
{
    return GetBits(bit.z, 0, 4);
}

uint GetDetailedCombinerConstantAlpha(const in ivec4 bit)
{
    return GetBits(bit.z, 4, 4);
}

uint GetDetailedCombinerSavePrevRgb(const in ivec4 bit)
{
    return GetBits(bit.x, 30, 1);
}

uint GetDetailedCombinerSavePrevAlpha(const in ivec4 bit)
{
    return GetBits(bit.y, 30, 1);
}

uint GetDetailedCombinerRgbSourceCount(const in ivec4 bit)
{
    return GetBits(bit.w, 0, 4);
}

uint GetDetailedCombinerAlphaSourceCount(const in ivec4 bit)
{
    return GetBits(bit.w, 4, 4);
}


//---------------------------------------------------------------------------
#define DETAILED_COMBINER_OPRGB_RGB       (0x0)
#define DETAILED_COMBINER_OPRGB_INV_RGB   (0x1)
#define DETAILED_COMBINER_OPRGB_ALPHA     (0x2)
#define DETAILED_COMBINER_OPRGB_INV_ALPHA (0x3)
#define DETAILED_COMBINER_OPRGB_RRR       (0x4)
#define DETAILED_COMBINER_OPRGB_INV_RRR   (0x5)
#define DETAILED_COMBINER_OPRGB_GGG       (0x8)
#define DETAILED_COMBINER_OPRGB_INV_GGG   (0x9)
#define DETAILED_COMBINER_OPRGB_BBB       (0xc)
#define DETAILED_COMBINER_OPRGB_INV_BBB   (0xd)

vec3 GetDetailedCombinerSrcOpRgb(
    const in uint srcRgbOp,
    const in vec4 rgbColor)
{
    vec3 color;
    switch(srcRgbOp)
    {
    case DETAILED_COMBINER_OPRGB_RGB:
        color = rgbColor.rgb;
        break;
    case DETAILED_COMBINER_OPRGB_INV_RGB:
        color = 1.0f - rgbColor.rgb;
        break;
    case DETAILED_COMBINER_OPRGB_ALPHA:
        color = vec3(rgbColor.a);
        break;
    case DETAILED_COMBINER_OPRGB_INV_ALPHA:
        color = vec3(1.0f - rgbColor.a);
        break;
    case DETAILED_COMBINER_OPRGB_RRR:
        color = vec3(rgbColor.r);
        break;
    case DETAILED_COMBINER_OPRGB_INV_RRR:
        color = vec3(1.0f - rgbColor.r);
        break;
    case DETAILED_COMBINER_OPRGB_GGG:
        color = vec3(rgbColor.g);
        break;
    case DETAILED_COMBINER_OPRGB_INV_GGG:
        color = vec3(1.0f - rgbColor.g);
        break;
    case DETAILED_COMBINER_OPRGB_BBB:
        color = vec3(rgbColor.b);
        break;
    case DETAILED_COMBINER_OPRGB_INV_BBB:
        color = vec3(1.0f - rgbColor.b);
        break;
    default:
        color = vec3(0.0f);
        break;
    }
    return color;
}

//---------------------------------------------------------------------------
#define DETAILED_COMBINER_OPALPHA_ALPHA     (0x0)
#define DETAILED_COMBINER_OPALPHA_INV_ALPHA (0x1)
#define DETAILED_COMBINER_OPALPHA_R         (0x2)
#define DETAILED_COMBINER_OPALPHA_INV_R     (0x3)
#define DETAILED_COMBINER_OPALPHA_G         (0x4)
#define DETAILED_COMBINER_OPALPHA_INV_G     (0x5)
#define DETAILED_COMBINER_OPALPHA_B         (0x6)
#define DETAILED_COMBINER_OPALPHA_INV_B     (0x7)

float GetDetailedCombinerSrcOpAlpha(
    const in uint srcAlphaOp,
    const in vec4 color)
{
    float alpha = 0.0f;
    switch(srcAlphaOp)
    {
    case DETAILED_COMBINER_OPALPHA_ALPHA:
        alpha = color.a;
        break;
    case DETAILED_COMBINER_OPALPHA_INV_ALPHA:
        alpha = 1.0f - color.a;
        break;
    case DETAILED_COMBINER_OPALPHA_R:
        alpha = color.r;
        break;
    case DETAILED_COMBINER_OPALPHA_INV_R:
        alpha = 1.0f - color.r;
        break;
    case DETAILED_COMBINER_OPALPHA_G:
        alpha = color.g;
        break;
    case DETAILED_COMBINER_OPALPHA_INV_G:
        alpha = 1.0f - color.g;
        break;
    case DETAILED_COMBINER_OPALPHA_B:
        alpha = color.b;
        break;
    case DETAILED_COMBINER_OPALPHA_INV_B:
        alpha = 1.0f - color.b;
        break;
    default:
        break;
    }
    return alpha;
}

//---------------------------------------------------------------------------
#define DETAILED_COMBINER_SRC_TEXTURE0        (0x3)
#define DETAILED_COMBINER_SRC_TEXTURE1        (0x4)
#define DETAILED_COMBINER_SRC_TEXTURE2        (0x5)
#define DETAILED_COMBINER_SRC_TEXTURE3        (0x6)
#define DETAILED_COMBINER_SRC_CONSTANT        (0xe)
#define DETAILED_COMBINER_SRC_PRIMARY         (0x0)
#define DETAILED_COMBINER_SRC_PREVIOUS        (0xf)
#define DETAILED_COMBINER_SRC_PREVIOUS_BUFFER (0xd)

vec3 GetDetailedCombinerSrcRgb(
    const in uint srcRgb,
    const in uint srcRgbOp,
    const in vec4 constantColor,
    const in sampler2D texture0,
    const in sampler2D texture1,
    const in sampler2D texture2,
    const in vec4 texCoord[3],
    const in vec4 primary,
    const in vec4 previous,
    const in vec4 previousBuffer
)
{
    vec4 color;
    switch(srcRgb)
    {
    case DETAILED_COMBINER_SRC_TEXTURE0:
        color = texture(texture0, texCoord[0].st);
        break;
    case DETAILED_COMBINER_SRC_TEXTURE1:
        color = texture(texture1, texCoord[1].st);
        break;
    case DETAILED_COMBINER_SRC_TEXTURE2:
        color = texture(texture2, texCoord[2].st);
        break;
    case DETAILED_COMBINER_SRC_TEXTURE3:
        color = vec4(0.0f);
        break;
    case DETAILED_COMBINER_SRC_CONSTANT:
        color = constantColor;
        break;
    case DETAILED_COMBINER_SRC_PRIMARY:
        color = primary;
        break;
    case DETAILED_COMBINER_SRC_PREVIOUS:
        color = previous;
        break;
    case DETAILED_COMBINER_SRC_PREVIOUS_BUFFER:
        color = previousBuffer;
        break;
    default:
        color = vec4(0.0f);
        break;
    }

    return GetDetailedCombinerSrcOpRgb(srcRgbOp, color);
}

float GetDetailedCombinerSrcAlpha(
    const in uint srcAlpha,
    const in uint srcAlphaOp,
    const in vec4 constantColor,
    const in sampler2D texture0,
    const in sampler2D texture1,
    const in sampler2D texture2,
    const in vec4 texCoord[3],
    const in vec4 primary,
    const in vec4 previous,
    const in vec4 previousBuffer
)
{
    vec4 color;
    switch(srcAlpha)
    {
    case DETAILED_COMBINER_SRC_TEXTURE0:
        color = texture(texture0, texCoord[0].st);
        break;
    case DETAILED_COMBINER_SRC_TEXTURE1:
        color = texture(texture1, texCoord[1].st);
        break;
    case DETAILED_COMBINER_SRC_TEXTURE2:
        color = texture(texture2, texCoord[2].st);
        break;
    case DETAILED_COMBINER_SRC_TEXTURE3:
        color = vec4(0.0f);
        break;
    case DETAILED_COMBINER_SRC_CONSTANT:
        color = constantColor;
        break;
    case DETAILED_COMBINER_SRC_PRIMARY:
        color = primary;
        break;
    case DETAILED_COMBINER_SRC_PREVIOUS:
        color = previous;
        break;
    case DETAILED_COMBINER_SRC_PREVIOUS_BUFFER:
        color = previousBuffer;
        break;
    default:
        color = vec4(0.0f);
        break;
    }
    return GetDetailedCombinerSrcOpAlpha(srcAlphaOp, color);
}

//---------------------------------------------------------------------------
#define DETAILED_COMBINER_COMBINE_REPLACE       (0x0)
#define DETAILED_COMBINER_COMBINE_MODULATE      (0x1)
#define DETAILED_COMBINER_COMBINE_ADD           (0x2)
#define DETAILED_COMBINER_COMBINE_ADD_SIGNED    (0x3)
#define DETAILED_COMBINER_COMBINE_INTERPOLATE   (0x4)
#define DETAILED_COMBINER_COMBINE_SUBTRACT      (0x5)
#define DETAILED_COMBINER_COMBINE_DOT3_RGB      (0x6)
#define DETAILED_COMBINER_COMBINE_DOT3_RGBA     (0x7)
#define DETAILED_COMBINER_COMBINE_ADD_MULT      (0x8)
#define DETAILED_COMBINER_COMBINE_MULT_ADD      (0x9)

vec3 GetDetailedCombinerModeRgb(
    const in uint combinerRgbMode,
    const in vec3 sourceRgb[3])
{
    vec3 color;
    switch(combinerRgbMode)
    {
    case DETAILED_COMBINER_COMBINE_REPLACE:
        color = sourceRgb[0];
        break;
    case DETAILED_COMBINER_COMBINE_MODULATE:
        color = sourceRgb[0] * sourceRgb[1];
        break;
    case DETAILED_COMBINER_COMBINE_ADD:
        color = sourceRgb[0] + sourceRgb[1];
        break;
    case DETAILED_COMBINER_COMBINE_ADD_SIGNED:
        color = sourceRgb[0] + sourceRgb[1] - 0.5f;
        break;
    case DETAILED_COMBINER_COMBINE_INTERPOLATE:
        color = (sourceRgb[0] * sourceRgb[2]) + (sourceRgb[1] * (1.0f - sourceRgb[2]));
        break;
    case DETAILED_COMBINER_COMBINE_SUBTRACT:
        color = sourceRgb[0] - sourceRgb[1];
        break;
    case DETAILED_COMBINER_COMBINE_MULT_ADD:
        color = (sourceRgb[0] * sourceRgb[1]) + sourceRgb[2];
        break;
    case DETAILED_COMBINER_COMBINE_ADD_MULT:
        color = (sourceRgb[0] + sourceRgb[1]) * sourceRgb[2];
        break;

    case DETAILED_COMBINER_COMBINE_DOT3_RGB:
    case DETAILED_COMBINER_COMBINE_DOT3_RGBA:
    default:
        color = vec3(0.0f);
        break;
    }
    return color;
}

float GetDetailedCombinerModeAlpha(
    const in uint combinerAlphaMode,
    const in float sourceAlpha[3])
{
    float alpha;
    switch(combinerAlphaMode)
    {
    case DETAILED_COMBINER_COMBINE_REPLACE:
        alpha = sourceAlpha[0];
        break;
    case DETAILED_COMBINER_COMBINE_MODULATE:
        alpha = sourceAlpha[0] * sourceAlpha[1];
        break;
    case DETAILED_COMBINER_COMBINE_ADD:
        alpha = sourceAlpha[0] + sourceAlpha[1];
        break;
    case DETAILED_COMBINER_COMBINE_ADD_SIGNED:
        alpha = sourceAlpha[0] + sourceAlpha[1] - 0.5f;
        break;
    case DETAILED_COMBINER_COMBINE_INTERPOLATE:
        alpha = (sourceAlpha[0] * sourceAlpha[2]) + (sourceAlpha[1] * (1.0f - sourceAlpha[2]));
        break;
    case DETAILED_COMBINER_COMBINE_SUBTRACT:
        alpha = sourceAlpha[0] - sourceAlpha[1];
        break;
    case DETAILED_COMBINER_COMBINE_MULT_ADD:
        alpha = (sourceAlpha[0] * sourceAlpha[1]) + sourceAlpha[2];
        break;
    case DETAILED_COMBINER_COMBINE_ADD_MULT:
        alpha = (sourceAlpha[0] + sourceAlpha[1]) * sourceAlpha[2];
        break;

    case DETAILED_COMBINER_COMBINE_DOT3_RGB:
    case DETAILED_COMBINER_COMBINE_DOT3_RGBA:
    default:
        alpha = 0.0f;
        break;
    }
    return alpha;
}

//---------------------------------------------------------------------------
#define DETAILED_COMBINER_SCALE_1 (0)
#define DETAILED_COMBINER_SCALE_2 (1)
#define DETAILED_COMBINER_SCALE_4 (2)

float GetDetailedCombinerRgbScale(const in uint scaleRgb)
{
    float scale;
    switch(scaleRgb)
    {
    case DETAILED_COMBINER_SCALE_1:
        scale = 1.0f;
        break;
    case DETAILED_COMBINER_SCALE_2:
        scale = 2.0f;
        break;
    case DETAILED_COMBINER_SCALE_4:
        scale = 4.0f;
        break;
    default:
        scale = 0.0f;
        break;
    }
    return scale;
}

float GetDetailedCombinerAlphaScale(const in uint scaleAlpha)
{
    float scale;
    switch(scaleAlpha)
    {
    case DETAILED_COMBINER_SCALE_1:
        scale = 1.0f;
        break;
    case DETAILED_COMBINER_SCALE_2:
        scale = 2.0f;
        break;
    case DETAILED_COMBINER_SCALE_4:
        scale = 4.0f;
        break;
    default:
        scale = 0.0f;
        break;
    }
    return scale;
}

//---------------------------------------------------------------------------
void DetailedCombinerStage(
    const in ivec4 stageBit,
    const in vec4 constantColors[7],
    const in sampler2D texture0,
    const in sampler2D texture1,
    const in sampler2D texture2,
    const in vec4 texCoord[3],
    const in vec4 primary,
    inout vec4 colorOutput,
    inout vec4 colorBuffer
    )
{
    // コンスタントカラー
    uint constantRgbReg = GetDetailedCombinerConstantRgb(stageBit);
    uint constantAlphaReg = GetDetailedCombinerConstantAlpha(stageBit);

    vec4 constantColorReg = vec4(
        constantColors[constantRgbReg].r,
        constantColors[constantRgbReg].g,
        constantColors[constantRgbReg].b,
        constantColors[constantAlphaReg].a);

    // バッファの入力元
    if (GetDetailedCombinerSavePrevRgb(stageBit) == 1)
    {
        colorBuffer.rgb = colorOutput.rgb;
    }
    if (GetDetailedCombinerSavePrevAlpha(stageBit) == 1)
    {
        colorBuffer.a = colorOutput.a;
    }

    // カラー ------------------------------
    vec3 rgbColor = vec3(0.0f);
    float alphaColor = 0.0f;
    const uint combineRgbMode = GetDetailedCombinerRgb(stageBit);
    const uint combineRgbCount = GetDetailedCombinerRgbSourceCount(stageBit);

    {
        // RGB ------------------------------
        vec3 rgbColors[3] = vec3[](
            vec3(0.0f),
            vec3(0.0f),
            vec3(0.0f)
        );
        // 動的に分岐を行っている箇所はバリエーション定数で静的な値となる為、最適化の際に無くなります。
        if (combineRgbCount >= 1)
        {
            rgbColors[0] = GetDetailedCombinerSrcRgb(
                GetDetailedCombinerSrcRgb0(stageBit),
                GetDetailedCombinerOperandRgb0(stageBit),
                constantColorReg,
                texture0,
                texture1,
                texture2,
                texCoord,
                primary,
                colorOutput,
                colorBuffer);
        }
        if (combineRgbCount >= 2)
        {
            rgbColors[1] = GetDetailedCombinerSrcRgb(
                GetDetailedCombinerSrcRgb1(stageBit),
                GetDetailedCombinerOperandRgb1(stageBit),
                constantColorReg,
                texture0,
                texture1,
                texture2,
                texCoord,
                primary,
                colorOutput,
                colorBuffer);
        }
        if (combineRgbCount >= 3)
        {
            rgbColors[2] = GetDetailedCombinerSrcRgb(
                GetDetailedCombinerSrcRgb2(stageBit),
                GetDetailedCombinerOperandRgb2(stageBit),
                constantColorReg,
                texture0,
                texture1,
                texture2,
                texCoord,
                primary,
                colorOutput,
                colorBuffer);
        }
        rgbColor = GetDetailedCombinerModeRgb(combineRgbMode, rgbColors) * GetDetailedCombinerRgbScale(GetDetailedCombinerScaleRgb(stageBit));

    }
    const uint combineAlphaMode = GetDetailedCombinerAlpha(stageBit);
    const uint combineAlphaCount = GetDetailedCombinerAlphaSourceCount(stageBit);
    {
        // アルファ ------------------------------
        float alphaColors[3] = float[]( 0.0f, 0.0f, 0.0f );
        // 動的に分岐を行っている箇所はバリエーション定数で静的な値となる為、最適化の際に無くなります。
        if (combineAlphaCount >= 1)
        {
            alphaColors[0] = GetDetailedCombinerSrcAlpha(
                GetDetailedCombinerSrcAlpha0(stageBit),
                GetDetailedCombinerOperandAlpha0(stageBit),
                constantColorReg,
                texture0,
                texture1,
                texture2,
                texCoord,
                primary,
                colorOutput,
                colorBuffer);
        }
        if (combineAlphaCount >= 2)
        {
            alphaColors[1] = GetDetailedCombinerSrcAlpha(
                GetDetailedCombinerSrcAlpha1(stageBit),
                GetDetailedCombinerOperandAlpha1(stageBit),
                constantColorReg,
                texture0,
                texture1,
                texture2,
                texCoord,
                primary,
                colorOutput,
                colorBuffer);
        }
        if (combineAlphaCount >= 3)
        {
            alphaColors[2] = GetDetailedCombinerSrcAlpha(
                GetDetailedCombinerSrcAlpha2(stageBit),
                GetDetailedCombinerOperandAlpha2(stageBit),
                constantColorReg,
                texture0,
                texture1,
                texture2,
                texCoord,
                primary,
                colorOutput,
                colorBuffer);
        }
        alphaColor = GetDetailedCombinerModeAlpha(combineAlphaMode, alphaColors) * GetDetailedCombinerAlphaScale(GetDetailedCombinerScaleAlpha(stageBit));
    }

    colorOutput = vec4(rgbColor.r, rgbColor.g, rgbColor.b, alphaColor);
}

//---------------------------------------------------------------------------
vec4 SampleColorWithDetailedCombiner(
    const in int count,
    const in ivec4 stageBits[6],
    const in vec4 constantColors[7],
    const in sampler2D texture0,
    const in sampler2D texture1,
    const in sampler2D texture2,
    const in vec4 texCoord[3],
    const in vec4 primary
    )
{
    vec4 colorBuffer = constantColors[0];
    vec4 colorOutput = vec4(0.0f);

    for (int i = 0; i < count; i++)
    {
        DetailedCombinerStage(
            stageBits[i],
            constantColors,
            texture0,
            texture1,
            texture2,
            texCoord,
            primary,
            colorOutput,
            colorBuffer);
    }
    return colorOutput;
}

//---------------------------------------------------------------------------
vec4 SampleColorWithVariationVariableDetailedCombiner(
    const in vec4 constantColors[7],
    const in sampler2D texture0,
    const in sampler2D texture1,
    const in sampler2D texture2,
    const in vec4 texCoord[3],
    const in vec4 primary
    )
{
    vec4 colorBuffer = constantColors[0];
    vec4 colorOutput = vec4(0.0f);

    // 動的に分岐を行っている箇所はバリエーション定数で静的な値となる為、最適化の際に無くなります。
    if (stageCount > 0)
    {
        DetailedCombinerStage(
            bit0,
            constantColors,
            texture0,
            texture1,
            texture2,
            texCoord,
            primary,
            colorOutput,
            colorBuffer);
    }

    if (stageCount > 1)
    {
        DetailedCombinerStage(
            bit1,
            constantColors,
            texture0,
            texture1,
            texture2,
            texCoord,
            primary,
            colorOutput,
            colorBuffer);
    }

    if (stageCount > 2)
    {
        DetailedCombinerStage(
            bit2,
            constantColors,
            texture0,
            texture1,
            texture2,
            texCoord,
            primary,
            colorOutput,
            colorBuffer);
    }

    if (stageCount > 3)
    {
        DetailedCombinerStage(
            bit3,
            constantColors,
            texture0,
            texture1,
            texture2,
            texCoord,
            primary,
            colorOutput,
            colorBuffer);
    }

    if (stageCount > 4)
    {
        DetailedCombinerStage(
            bit4,
            constantColors,
            texture0,
            texture1,
            texture2,
            texCoord,
            primary,
            colorOutput,
            colorBuffer);
    }

    if (stageCount > 5)
    {
        DetailedCombinerStage(
            bit5,
            constantColors,
            texture0,
            texture1,
            texture2,
            texCoord,
            primary,
            colorOutput,
            colorBuffer);
    }

    return colorOutput;
}
