﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/viewer/vfx_CmdSender.h>
#include <nn/vfx/viewer/vfx_Message.h>
#include <nn/vfx/viewer/vfx_Protocol.h>
#include <nn/vfx/viewer/vfx_ToolConnector.h>
#include <nn/vfx/vfx_Misc.h>

namespace nn {
namespace vfx {
namespace viewer {
namespace detail {

CommandSender*  CommandSender::g_pCommandSender = NULL;
nn::os::Mutex   CommandSender::m_Mutex( true );

//------------------------------------------------------------------------------
//  コマンド送信クラスを生成する
//------------------------------------------------------------------------------
CommandSender* CreateCommandSender( nn::vfx::Heap* pHeap, size_t bufferSize ) NN_NOEXCEPT
{
    if( CommandSender::g_pCommandSender )
    {
        return CommandSender::g_pCommandSender;
    }

    void* buffer = pHeap->Alloc( sizeof( CommandSender ) );
    CommandSender::g_pCommandSender = new ( buffer )CommandSender( pHeap, bufferSize );
    return CommandSender::g_pCommandSender;
}

//------------------------------------------------------------------------------
//  コマンド受信クラスを取得する
//------------------------------------------------------------------------------
CommandSender* GetCommandSender() NN_NOEXCEPT
{
    return CommandSender::g_pCommandSender;
}

//------------------------------------------------------------------------------
//  コマンド受信クラスを破棄する
//------------------------------------------------------------------------------
void DestroyCommandSender( nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    CommandSender::g_pCommandSender->~CommandSender();
    pHeap->Free( CommandSender::g_pCommandSender );
    CommandSender::g_pCommandSender = NULL;
}

//------------------------------------------------------------------------------
//  コントラクタ
//------------------------------------------------------------------------------
CommandSender::CommandSender( nn::vfx::Heap* pHeap, size_t bufferSize ) NN_NOEXCEPT
{
    m_pHeap = pHeap;
    m_SharedBufferSize = bufferSize;



    // プロセス内のアドレス空間にファイルのビューをマップ
    m_SharedBuffer = reinterpret_cast< uint8_t* >( m_pHeap->Alloc( m_SharedBufferSize ) );
    NN_SDK_ASSERT_NOT_NULL( m_SharedBuffer );
}

//------------------------------------------------------------------------------
//  デストラクタ
//------------------------------------------------------------------------------
CommandSender::~CommandSender() NN_NOEXCEPT
{
    while( GetCommand() )
    {
        PopCommand();
    }

    m_pHeap->Free( m_SharedBuffer );
}

//------------------------------------------------------------------------------
//  EffectMakerにコマンドを送信する
//------------------------------------------------------------------------------
void CommandSender::SendCommand( MessageType type, void* buffer, size_t size ) NN_NOEXCEPT
{
    NN_UNUSED( type );
    NN_UNUSED( buffer );
    NN_UNUSED( size );

    // TODO: 送信データフォーマットを作成すること
    Message header;
    header.type = type;
    header.bufferSize = static_cast<uint32_t>(size);

    size_t headerSize = sizeof( Message );

    if( ( size + headerSize ) > m_SharedBufferSize )
    {
        nn::vfx::detail::OutputWarning( "Send Buffer is Over.\n" );
        return;
    }

    // データ
    memcpy( m_SharedBuffer, &header, headerSize );
    memcpy( ( m_SharedBuffer + headerSize ), buffer, size );

    AddCommand( m_SharedBuffer, size + headerSize );

    if ( m_ToolConnector )
    {
        m_ToolConnector->SendPacket();
    }
}

//------------------------------------------------------------------------------
//  送信コマンドを追加する
//------------------------------------------------------------------------------
void CommandSender::AddCommand( const void* buffer, size_t size ) NN_NOEXCEPT
{
    if( m_Mutex.TryLock() )
    {
        Command* pCommand = static_cast< Command* >( m_pHeap->Alloc( sizeof( Command ) ) );
        NN_SDK_ASSERT_NOT_NULL( pCommand );

        pCommand->Initialize();
        pCommand->SetCommand( m_pHeap, static_cast< uint32_t >( CommandReceiverSettings_CommandId ), buffer, size );
        m_CommandQueue.PushBack( pCommand );
        m_Mutex.Unlock();
    }
}

//------------------------------------------------------------------------------
//  送信コマンドをpopする
//------------------------------------------------------------------------------
void CommandSender::PopCommand() NN_NOEXCEPT
{
    if( m_CommandQueue.Empty() )
    {
        return;
    }
    if( m_Mutex.TryLock() )
    {
        Command* pCommand = m_CommandQueue.Back();
        if( pCommand )
        {
            pCommand->~Command();
            m_pHeap->Free( pCommand );
        }
        m_CommandQueue.PopBack();
        m_Mutex.Unlock();
    }
}

//------------------------------------------------------------------------------
//  送信コマンドを取得する
//------------------------------------------------------------------------------
Command* CommandSender::GetCommand() NN_NOEXCEPT
{
    if( m_CommandQueue.Empty() )
    {
        return NULL;
    }
    return m_CommandQueue.Back();
}

} // namespace detail
} // namespace viewer
} // namespace vfx
} // namespace nn
