﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/vfx/vfx_UniformBlock.h>
#include <nn/vfx/vfx_Heap.h>


namespace nn {
namespace vfx {
namespace detail {


//---------------------------------------------------------------------------
//  定数バッファ名から初期化します。
//---------------------------------------------------------------------------
void ConstantBufferSlot::Initialize( nn::gfx::Shader* pShader, const char* constantBufferName ) NN_NOEXCEPT
{
    m_VertexShaderSlot = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, constantBufferName );
    m_PixelShaderSlot  = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_ConstantBuffer, constantBufferName );
}

//---------------------------------------------------------------------------
//  バッファを初期化する
//---------------------------------------------------------------------------
bool Buffer::Initialize( nn::gfx::Device*   pGfxDevice,
                        nn::vfx::Heap*     pHeap,
                        int                gpuAccess,
                        size_t             bufferSize ) NN_NOEXCEPT
{
    if ( m_IsInitialized ) return true;

    m_pBuffer           = NULL;
    m_BufferSize        = bufferSize;
    m_MemoryPoolSize  = 0;

    nn::gfx::Buffer::InfoType bfInfo;
    bfInfo.SetDefault();
    bfInfo.SetGpuAccessFlags( gpuAccess );

    nn::gfx::MemoryPool::InfoType mpInfo;
    mpInfo.SetDefault();
    mpInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );

    m_BufferAlign       = nn::gfx::Buffer::GetBufferAlignment( pGfxDevice, bfInfo );
    size_t poolAlign    = nn::gfx::MemoryPool::GetPoolMemoryAlignment( pGfxDevice, mpInfo );

    m_MemoryPoolSize  = m_BufferSize;
    m_MemoryPoolSize  = nn::util::align_up( m_MemoryPoolSize, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( pGfxDevice, mpInfo ) );

    m_pBuffer = pHeap->Alloc( m_MemoryPoolSize, poolAlign );
    if ( !m_pBuffer ) return false;

    mpInfo.SetPoolMemory( m_pBuffer, m_MemoryPoolSize );
    m_GfxMemoryPool.Initialize( pGfxDevice, mpInfo );

    bfInfo.SetSize( m_BufferSize );
    m_GfxBuffer.Initialize( pGfxDevice, bfInfo, &m_GfxMemoryPool, 0, m_MemoryPoolSize );

    m_IsInitialized = true;
    return true;
}


//---------------------------------------------------------------------------
//  バッファを終了する
//---------------------------------------------------------------------------
void Buffer::Finalize( nn::gfx::Device* pGfxDevice, nn::vfx::Heap* heap ) NN_NOEXCEPT
{
    if ( !m_IsInitialized ) return;

    m_GfxBuffer.Finalize( pGfxDevice );
    m_GfxMemoryPool.Finalize( pGfxDevice );

    if ( m_pBuffer )
    {
        heap->Free( m_pBuffer );
        m_pBuffer = NULL;
    }

    m_IsInitialized = false;
}

}

} // namespace vfx
} // namespace nn

