﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_StripeCommon.h>
#include <nn/vfx/vfx_Emitter.h>
#include <nn/vfx/vfx_System.h>

namespace nn {
namespace vfx {
namespace detail {

//---------------------------------------------------------------------------
//  遅延ストライプの最大本数を計算します。
//  基本的に Emitter::CalculateRequiredParticleAsignmentCount() を寿命の代わりに履歴数で計算したもの
//---------------------------------------------------------------------------
int StripeSystemUtility::CalculateDelayStripeCount( const Emitter* pEmitter, int numHistory ) NN_NOEXCEPT
{
    const detail::ResEmitter* pData = pEmitter->GetResEmitter();
    int requiredParticleCount = 0;

    float maxEmitRatio = 0; // （アニメーションを含む）最大の放出レート
    int   maxEmitCountUntilPeak = 0; // 最初に生んだパーティクルが消えるまでに最大何回放出が行われるか

    // TODO: この計算、もうちょっと数を詰められそう
    if( pEmitter->GetEmitterResource()->m_EmitterAnimationArray[ detail::EmitterAnimationType_EmissionRate ] )
    {
        maxEmitRatio = pEmitter->GetEmissionRateMax();
        if( maxEmitRatio < 1.0f )
        {
            maxEmitRatio = 1.0f;
        }
    }
    else
    {
        maxEmitRatio = pData->emission.rate;
    }

    // 放出の回数の推定
    const bool isOneTime = ( pData->emission.isOneTime > 0 );    // ワンタイムかどうか
    const bool isLifeInfinity = ( pData->ptcl.isLifeInfinity > 0 );   // 寿命無限かどうか
    const int  emitDuration = ( pData->emission.emitDuration );     // 放出時間
    const int  emitInterval = ( pData->emission.interval );         // 放出間隔

    // 放出期間が終わるまでに何回放出機会があったか
    const float emitTime = ( emitDuration == 0 ) ? 1.0f : static_cast< float >( emitDuration );
    const int maxEmitCountUntilEmitEnd = static_cast< int >( ::std::ceilf( emitTime / ( emitInterval + 1 ) ) );

    // 履歴が無くなるまで（≒遅延ストライプが履歴数分更新するまで）に何回放出機会があったか
    const int maxEmitCountUntilHistoryEnd = static_cast< int >( ::std::ceilf( static_cast< float >( numHistory ) / ( emitInterval + 1 ) ) );

    if( isLifeInfinity && isOneTime )
    {
        // 寿命無限（※ワンタイム時限定）
        // 放出回数：( 放出時間 / (放出間隔＋１) ) の切り上げ。
        maxEmitCountUntilPeak = maxEmitCountUntilEmitEnd;
    }
    else
    {
        if( isOneTime && ( maxEmitCountUntilEmitEnd < maxEmitCountUntilHistoryEnd ) )
        {
            // ワンタイムの時だけ、放出時間による放出回数と比べて、小さい方を採用
            // 放出回数：( 放出時間 / (放出間隔＋１) ) の切り上げ。
            maxEmitCountUntilPeak = maxEmitCountUntilEmitEnd;
        }
        else
        {
            // それ以外のケースでは、パーティクルの寿命が尽きるまでに放出された回数を採用
            // 放出回数：(寿命÷ (放出間隔＋１) ) の切り上げ。
            maxEmitCountUntilPeak = maxEmitCountUntilHistoryEnd;
        }
    }

    // 放出レートが小数の場合、多めに切り上げる
    requiredParticleCount = static_cast< int >( ::std::ceilf( maxEmitCountUntilPeak * maxEmitRatio ) + ::std::ceilf( maxEmitRatio ) );

    // 等分割系エミッタ x 一斉放出 の場合の補正
    if( pData->volume.primEmitType == detail::PrimitiveEmissionType_Unison )
    {
        // 等分割系の一斉放出の場合は、放出レートを分割数倍にする
        if( pData->volume.volumeType == detail::EmitterVolumeType_CircleEquallyDivided )
        {
            requiredParticleCount *= pData->volume.numDivideCircle;
        }
        else if( pData->volume.volumeType == detail::EmitterVolumeType_LineEquallyDivided )
        {
            requiredParticleCount *= pData->volume.numDivideLine;
        }
    }

    //---------------------------------------------------------------------------
    //  「軽量版チャイルド」の場合、エミッタは共通で持つので、
    //  最大パーティクル数を、親エミッタが出すパーティクル数（※子エミッタ数）分乗算する。
    //---------------------------------------------------------------------------
    if( pEmitter->IsChildEmitter() && !pData->inherit.enableEmitterParticle )
    {
        requiredParticleCount *= pEmitter->GetParentEmitter()->GetParticleAttrFillMax();
    }

    return requiredParticleCount;
}

//---------------------------------------------------------------------------
//  デフォルトのシェーダ設定処理を行います。
//---------------------------------------------------------------------------
bool StripeSystemUtility::SetupDefaultShaderSetting( nn::gfx::CommandBuffer* pCommandBuffer, Emitter* pEmitter, ShaderType shaderType, Shader* pShader, void* pUserParam, DrawParameterArg* pDrawParameterArg ) NN_NOEXCEPT
{
    pCommandBuffer->SetShader( pShader->GetShader(), nn::gfx::ShaderStageBit_All );

    EmitterCalculator* const pCalculator = pEmitter->GetEmitterCalculator();

    // ブレンドステート
    pCommandBuffer->SetBlendState( &pEmitter->GetEmitterResource()->m_RenderState.GetGfxBelndState() );

    // デプスステート
    pCommandBuffer->SetDepthStencilState( &pEmitter->GetEmitterResource()->m_RenderState.GetGfxDepthState() );

    // ラスタライザステート
    pCommandBuffer->SetRasterizerState( &pEmitter->GetEmitterResource()->m_RenderState.GetGfxRasterizerState() );

    // 頂点ステート
    pCommandBuffer->SetVertexState( &pEmitter->GetEmitterResource()->m_VertexState );

    // エミッタ管理のテクスチャを設定
    pCalculator->SetDefaultEmitterTextures( pCommandBuffer, pEmitter, pShader, pDrawParameterArg );

    // エミッタ管理の定数バッファを設定
    pCalculator->SetDefaultEmitterConstantBuffers( pCommandBuffer, pEmitter, pShader, pDrawParameterArg );

    // 描画前コールバックの呼び出し
    if( !pEmitter->GetEmitterSet()->GetSystem()->InvokeBeforeRenderCallbacks( pCommandBuffer, pEmitter, shaderType, pUserParam, pDrawParameterArg ) )
    {
        return false;
    }

    return true;
}

} // namespace detail
} // namespace vfx
} // namespace nn
