﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_Stripe.h>
#include <nn/vfx/vfx_Heap.h>
#include <nn/vfx/vfx_Emitter.h>
#include <nn/vfx/vfx_System.h>
#include <nn/vfx/vfx_Misc.h>

namespace nn {
namespace vfx {
namespace detail {

static const CustomShaderConstantBufferIndex StripeConstantBufferIndex = CustomShaderConstantBufferIndex_4;                 //!< カスタム定数バッファIndex
static const EmitterPluginCallbackIndex StripeCallbackIndex = EmitterPluginCallbackIndex_2;   //!< エミッタプラグイン用コールバックID

StripeSystem* StripeSystem::g_pStripeSystem = NULL;

//---------------------------------------------------------------------------
//  コンストラクタ
//---------------------------------------------------------------------------
StripeSystem::StripeSystem( nn::vfx::Heap* pHeap, nn::vfx::System* pSystem, BufferingMode bufferMode, int stripeCount ) NN_NOEXCEPT
    : m_pSystem( pSystem )
    , m_pHeap( pHeap )
    , m_BufferMode( bufferMode )
    , m_StripeArrayCount( stripeCount )
    , m_StripeWorkSize( sizeof( StripeInstance ) * stripeCount )
    , m_StripeArrayIndex( 0 )
    , m_StripeCount( 0 )
    , m_StripeArray( NULL )
{
    m_StripeArray = reinterpret_cast< StripeInstance* >( m_pHeap->Alloc( sizeof( StripeInstance ) * m_StripeArrayCount ) );
    memset( m_StripeArray, 0, sizeof( StripeInstance ) * m_StripeArrayCount );

    // ストライプコールバックを設定する
    CallbackSet callbackSet;
    callbackSet.emitterInitialize = StripeSystem::InitializeStripeEmitter;
    callbackSet.particleEmit = StripeSystem::EmitStripe;
    callbackSet.particleRemove = StripeSystem::KillStripe;
    callbackSet.particleCalculate = StripeSystem::ParticleCalculateCallback;
    callbackSet.emitterPreCalculate = StripeSystem::EmitterPreCalculateCallback;
    callbackSet.emitterPostCalculate = StripeSystem::EmitterPostCalculateCallback;
    callbackSet.emitterDraw = StripeSystem::EmitterDrawCallback;
    callbackSet.emitterFinalize = StripeSystem::FinalizeStripeEmitter;
    callbackSet.renderStateSet = detail::DummyRenderStateSetCallback;
    m_pSystem->SetEmitterPluginCallbackSet( detail::StripeCallbackIndex, callbackSet );
}

//---------------------------------------------------------------------------
//  デストラクタ
//---------------------------------------------------------------------------
StripeSystem::~StripeSystem() NN_NOEXCEPT
{
    m_pHeap->Free( m_StripeArray );
    m_StripeArray = NULL;
}

//------------------------------------------------------------------------------
//  ストライプシステムの初期化処理を行います。
//------------------------------------------------------------------------------
void StripeSystem::InitializeSystem( nn::vfx::Heap* pHeap, nn::vfx::System* pSystem, BufferingMode bufferMode, int stripeCount ) NN_NOEXCEPT
{
    if( g_pStripeSystem )
    {
        OutputError( "StripeSystem is already initialized.\n" );
    }

    // プレビュー生成用のメモリを確保します。
    void* buffer = pHeap->Alloc( sizeof( StripeSystem ) );
    if( buffer == NULL )
    {
        nn::vfx::detail::OutputWarning( "[Stripe] Memory Allocate Error!! : %d\n", sizeof( StripeSystem ) );
        return;
    }

    // システムのコンストラクタを走らせる
    g_pStripeSystem = new ( buffer )StripeSystem( pHeap, pSystem, bufferMode, stripeCount );
}

//------------------------------------------------------------------------------
//  ストライプシステムの終了処理を行います。
//------------------------------------------------------------------------------
void StripeSystem::FinalizeSystem( nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    g_pStripeSystem->~StripeSystem();
    pHeap->Free( g_pStripeSystem );
    g_pStripeSystem = NULL;
}

//------------------------------------------------------------------------------
//  エミッタ生成後コールバック
//------------------------------------------------------------------------------
bool StripeSystem::InitializeStripeEmitter( EmitterInitializeArg& arg ) NN_NOEXCEPT
{
    //------------------------------------------------------------------------------
    // シェーダアトリビュートの方は ShaderManager 側で初期化済みという前提。
    // （代わりにアトリビュート名が決め打ちかつ VEC4 縛りになっている）
    //------------------------------------------------------------------------------
    if( arg.pEmitter->GetCalculationType() != EmitterCalculationMode_Cpu )
    {
        return false;
    }

    // エミッタ単位での記憶領域を確保
    EmitterPluginUserData* pEPUserData = reinterpret_cast< EmitterPluginUserData* >( arg.pEmitter->GetDynamicHeap().Alloc( sizeof( EmitterPluginUserData ) ) );
    if( pEPUserData == NULL )
    {
        return false;
    }
    MemUtil::FillZero( pEPUserData, sizeof( EmitterPluginUserData ) ); // ゼロ埋めで初期化
    arg.pEmitter->SetEmitterPluginUserData( pEPUserData );

    //------------------------------------------------------------------------------
    // エミッタ単位で記憶する情報を保存
    //------------------------------------------------------------------------------
    {
        pEPUserData->maxEmitRatio = static_cast< uint32_t >( ::std::ceilf( arg.pEmitter->GetEmissionRateMax() ) );
        pEPUserData->maxParticleLife = arg.pEmitter->GetParticleLifeMax();
        pEPUserData->bufferSide = BufferSide_FrontBuffer;
        {
            //------------------------------------------------------------------------------
            // 静的な定数バッファの内容をあらかじめ作っておく
            //------------------------------------------------------------------------------
            ResStripeHistory* pStripeData = GetStripeData( arg.pEmitter );
            pEPUserData->staticParam0.x = pStripeData->headAlpha;
            pEPUserData->staticParam0.y = pStripeData->tailAlpha;
            pEPUserData->staticParam0.z = 0.0f;
            pEPUserData->staticParam0.w = 0.0f;    // パラメータ無し
        }
    }

    bool ret = false;

    // 定数バッファ を初期化
    ret = arg.pEmitter->InitializeCustomConstantBuffer( detail::StripeConstantBufferIndex, sizeof( ConstantBufferObject ) );
    if ( !ret )
    {
        return ret;
    }

    // ポリゴン用バッファを確保
    ret = g_pStripeSystem->AllocStripeSystemVertexBuffer( arg.pEmitter );
    if ( !ret )
    {
        Warning( arg.pEmitter, RuntimeWarningId_StripeGpuBufferAllocationFailed );
        nn::vfx::detail::OutputWarning( "[Stripe] Memory Allocate Error!! : %d\n", sizeof( StripeSystem ) );
    }

    return ret;
}

//---------------------------------------------------------------------------
//  ポリゴン用ワークを確保します。
//---------------------------------------------------------------------------
bool StripeSystem::AllocStripeSystemVertexBuffer( Emitter* const pEmitter ) NN_NOEXCEPT
{
    //---------------------------------------------------------------------------
    // ここでエミッタ全体のストライプ用頂点バッファを初期化。
    // 描画はストライプ1本ごとにオフセットと範囲を指定して描画していく。
    //---------------------------------------------------------------------------
    ResStripeHistory* pStripeData = GetStripeData( pEmitter );
    EmitterPluginUserData* pEPUserData = StripeSystem::GetEmitterUserData( pEmitter );

    const int historyCount = static_cast< int >( pStripeData->numHistory );
    const int maxDelayedStripeCount = StripeSystemUtility::CalculateDelayStripeCount( pEmitter, historyCount );

    // 最大のパーティクル存在数＋遅延ストライプの本数（上記）が最大数
    pEPUserData->maxStripeCount = pEmitter->GetParticleAttrFillMax() + maxDelayedStripeCount;

    const int divCount = static_cast< int >( pStripeData->numDivide );
    pEPUserData->vertexCountPerStripe = ( historyCount + ( ( historyCount - 1 ) * divCount ) ) * 2;

    const int vertexBufferCount = pEPUserData->vertexCountPerStripe * pEPUserData->maxStripeCount;  // 背骨の数×左右（2個）×ストライプ本数
    pEPUserData->vertexBufferCount = vertexBufferCount * m_BufferMode;                       // バッファリングの分倍加する

    // 頂点バッファサイズ（シングルバッファの分）
    const size_t bufferSize = vertexBufferCount * sizeof( VertexAttribute );

    //---------------------------------------------------------------------------
    // Attribute の初期化
    // 確保した領域は後に自分で開放する必要があるので記憶する
    //---------------------------------------------------------------------------
    pEPUserData->vertexBufferInitialized = false;
    if( bufferSize != 0 )
    {
        if ( pEmitter->InitializeCustomAttribute( m_BufferMode, bufferSize ) )
        {
            pEPUserData->vertexBufferInitialized = true;
        }
    }

    return pEPUserData->vertexBufferInitialized;
}

//------------------------------------------------------------------------------
//  エミッタ破棄後コールバック
//------------------------------------------------------------------------------
void StripeSystem::FinalizeStripeEmitter( EmitterFinalizeArg& arg ) NN_NOEXCEPT
{
    if( arg.pEmitter->GetCalculationType() != EmitterCalculationMode_Cpu )
    {
        return;
    }

    EmitterPluginUserData* pEPUserData = StripeSystem::GetEmitterUserData( arg.pEmitter );
    if ( pEPUserData )
    {
        pEPUserData->vertexBufferInitialized = false;

        // 遅延ストライプの解放
        StripeSystemUtility::FreeAllDelayedStripe( g_pStripeSystem, &pEPUserData, arg.pEmitter );

        // エミッタユーザーデータの領域を解放
        detail::FreeFromDynamicHeapSafely( reinterpret_cast< void** >( &pEPUserData ), arg.pEmitter );
        arg.pEmitter->SetEmitterPluginUserData( NULL );
    }
}

//------------------------------------------------------------------------------
//  パーティクル生成コールバック
//------------------------------------------------------------------------------
bool StripeSystem::EmitStripe( ParticleEmitArg& arg ) NN_NOEXCEPT
{
    if( arg.pEmitter->GetCalculationType() != EmitterCalculationMode_Cpu )
    {
        return false;
    }

    StripeInstance* pStripe = StripeSystemUtility::AllocStripe( g_pStripeSystem, arg.pEmitter );
    if( pStripe == NULL )
    {
        return false;
    }

    // ユーザーデータの割り当て
    arg.pEmitterPluginData = pStripe;

    // 頂点バッファの開始位置を設定
    // 最大履歴数でアラインしてラウンドロビンで割り当て。
    // かならず隙間ができるので、確定で詰められるならその方がベター。
    EmitterPluginUserData* pEPUserData = StripeSystem::GetEmitterUserData( arg.pEmitter );
    pStripe->indexOffset = pEPUserData->attributeIndex % pEPUserData->maxStripeCount;
    if( ++pEPUserData->attributeIndex >= pEPUserData->maxStripeCount )
    {
        pEPUserData->attributeIndex = 0;
    }

    // ランダム値の割り当て
    arg.GetRandom( &pStripe->random );

    return true;
}

//------------------------------------------------------------------------------
// ストライプのカラーを更新（ストライプ一本で一つのカラーを共有）
//------------------------------------------------------------------------------
void StripeSystem::UpdateStripeColor( ParticleCalculateArgImpl& arg, StripeInstance* pStripe ) NN_NOEXCEPT
{
    nn::util::Vector4fType random;
    arg.GetRandom( &random );
    arg.pEmitter->GetEmitterCalculator()->CalculateParticleColor0RawValue(
        &pStripe->color0,
        arg.pEmitter->GetEmitterResource(),
        random,
        arg.GetLife(),
        arg.GetTime() );
    arg.pEmitter->GetEmitterCalculator()->CalculateParticleColor1RawValue(
        &pStripe->color1,
        arg.pEmitter->GetEmitterResource(),
        random,
        arg.GetLife(),
        arg.GetTime() );
    // カラースケールを乗算
    // MEMO: 本来はシェーダ側で乗算するべきだが、シェーダー側がまだ追いついていないのでこちらで乗算しておく
    // エミッタカラー / カラースケール
    const float colorScale = arg.pEmitter->GetEmitterResource()->m_ResEmitterStaticConstantBuffer->colorScale;
    nn::util::VectorSetX( &pStripe->color0, nn::util::VectorGetX( pStripe->color0 ) * colorScale );
    nn::util::VectorSetY( &pStripe->color0, nn::util::VectorGetY( pStripe->color0 ) * colorScale );
    nn::util::VectorSetZ( &pStripe->color0, nn::util::VectorGetZ( pStripe->color0 ) * colorScale );

    nn::util::VectorSetX( &pStripe->color1, nn::util::VectorGetX( pStripe->color1 ) * colorScale );
    nn::util::VectorSetY( &pStripe->color1, nn::util::VectorGetY( pStripe->color1 ) * colorScale );
    nn::util::VectorSetZ( &pStripe->color1, nn::util::VectorGetZ( pStripe->color1 ) * colorScale );
}

//------------------------------------------------------------------------------
//  パーティクル削除コールバック
//------------------------------------------------------------------------------
bool StripeSystem::KillStripe( ParticleRemoveArg& arg ) NN_NOEXCEPT
{
    if( arg.pEmitter->GetCalculationType() != EmitterCalculationMode_Cpu )
    {
        return false;
    }

    if( arg.pEmitterPluginData )
    {
        StripeInstance* pStripe = reinterpret_cast< StripeInstance* >( arg.pEmitterPluginData );
        NN_SDK_ASSERT( pStripe );

        // ストライプのカラーを更新（ストライプ一本で一つのカラーを共有）
#ifdef _VFX_NSDK01215_COMPATIBLE
        arg.CalculateLocalColor0( &pStripe->color0 );
        arg.CalculateLocalColor1( &pStripe->color1 );
#else
        UpdateStripeColor( arg, pStripe );
#endif
        EmitterPluginUserData* pEPUserData = StripeSystem::GetEmitterUserData( arg.pEmitter );
        StripeSystemUtility::MoveStripeToDelayList< StripeSystem >( &pEPUserData, pStripe );
    }
    return true;
}

//---------------------------------------------------------------------------
//  ストライプ計算処理を行います。
//---------------------------------------------------------------------------
inline void StripeSystem::CalculateStripe( ParticleCalculateArg& arg, StripeInstance* pStripe ) NN_NOEXCEPT
{
    //---------------------------------------------------------------------------
    // 事前条件チェックは呼び出し元のコールバック側で確認済み。
    //---------------------------------------------------------------------------
    Emitter* pEmitter = arg.pEmitter;
    ResStripeHistory* pStripeData = GetStripeData( pEmitter );
    EmitterPluginUserData* pEPUserData = StripeSystem::GetEmitterUserData( pEmitter );
    VertexAttribute* pVertexBufferHead = reinterpret_cast< VertexAttribute* >( arg.pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.Map( pEPUserData->bufferSide ) );

    // 現在時刻をコピー
    pStripe->time = arg.GetTime();
    pStripe->life = arg.GetLife();
    pStripe->renderVertexCount = 0;

    // ストライプのカラーを更新（ストライプ一本で一つのカラーを共有）
#ifdef _VFX_NSDK01215_COMPATIBLE
    arg.CalculateLocalColor0( &pStripe->color0 );
    arg.CalculateLocalColor1( &pStripe->color1 );
#else
    UpdateStripeColor( arg, pStripe );
#endif

    // 履歴を積む
    History* pNowHistory = pStripe->pHistoryHead;          // 新しい履歴（最後尾）
    pStripe->pHistoryHead = pStripe->pHistoryHead->pPrev;    // 最後尾を指すポインタを更新
    {
        nn::util::Vector3fType scale;
        nn::util::Vector3fType stripePos, stripeVec;

        if( pStripeData->emitterFollow )
        {
            // エミッタに完全追従：ローカル系で位置を覚える
            arg.CalculateWorldScale( &scale );
            arg.GetLocalPos( &stripePos );
            arg.GetLocalVec( &stripeVec );
        }
        else
        {
            // エミッタに完全追従しない：ワールド系で位置を覚える
            arg.CalculateWorldScale( &scale );
            arg.GetWorldPos( &stripePos );
            arg.GetWorldVec( &stripeVec );
        }

#ifndef _VFX_NSDK01215_COMPATIBLE
        // 先頭位置を記憶
        pStripe->headPos = stripePos;
#endif

        detail::Vector3fCopy( &pNowHistory->pos, stripePos );
        nn::util::VectorSetW( &pNowHistory->pos ,nn::util::VectorGetX( scale ) );
        pNowHistory->dir = stripeVec;   // 履歴ポイントの向き

        if( pStripeData->emitterFollow )
        {
            // エミッタに完全追従：ローカル系で位置を覚えるので上方向はあくまでY軸方向
            nn::util::VectorSet( &pNowHistory->emat, 0, 1, 0 );
        }
        else
        {
            // エミッタに完全追従しない：ワールド系で位置を覚えるので上方向はエミッタ行列のY軸方向
            const nn::util::Matrix4x3fType& matrixSrt = arg.pEmitter->GetMatrixSrt();
            nn::util::Vector3f ay;
            nn::util::MatrixGetAxisY( &ay, matrixSrt );
            nn::util::VectorSetX( &pNowHistory->emat, nn::util::VectorGetX( ay ) );
            nn::util::VectorSetY( &pNowHistory->emat, nn::util::VectorGetY( ay ) );
            nn::util::VectorSetZ( &pNowHistory->emat, nn::util::VectorGetZ( ay ) );
        }

#ifndef _VFX_NSDK01215_COMPATIBLE
        // ストライプ Z ロール: ematを使うビルボードタイプ（エミッタ行列ストライプ＆エミッタ上下に伸ばす）の時、emat を回転させてストライプを曲げる
        if( pStripeData->calcType == StripeOrientationType_EmitterMatrix || pStripeData->calcType == StripeOrientationType_EmitterUpright )
        {
            if( arg.pEmitter->GetResEmitter()->ptcl.isRotateZ )
            {
                const float Epsilon = 0.00001f; // 「ほぼ」0とみなす値

                nn::util::Vector3fType rotateAxis = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
                if( pStripe->historyCount >= 1 )
                {
                    // 2つ目以降の履歴点は、移動差分も進行方向とみなす
                    nn::util::Vector4fType temp;
                    nn::util::VectorSubtract( &temp, pNowHistory->pos, pNowHistory->pNext->pos );
                    nn::util::Vector3fType diff;
                    detail::Vector3fCopy( &diff, temp );
                    nn::util::VectorMultiply( &diff, diff, 1.0f / arg.pEmitter->GetFrameRate() ); // フレームレートから速度換算
                    nn::util::VectorAdd( &rotateAxis, rotateAxis, diff );
                }
                else
                {
                    // 一旦速度で代用（次の更新で2回目の値で上書きされる）
                    rotateAxis = pNowHistory->dir;
                }

                if( nn::util::VectorLengthSquared( rotateAxis ) < Epsilon )
                {
                    // 進行方向＋移動差分が 0 に近い場合、エミッタ行列を採用
                    const nn::util::Matrix4x3fType& matrixSrt = arg.pEmitter->GetMatrixSrt();
                    nn::util::MatrixGetAxisZ( &rotateAxis, matrixSrt );
                }
                nn::util::VectorNormalize( &rotateAxis, rotateAxis );

                // 進行方向について、 emat を rotateZ だけ回転させる
                nn::util::Vector3fType rotate;
                arg.CalculateCurrentRotateRadian( &rotate );    // Z 要素だけ使う
                const float rotateZ = nn::util::VectorGetZ( rotate );
                detail::VectorRotateArbitraryAxis( &pNowHistory->emat, pNowHistory->emat, rotateAxis, rotateZ );
                nn::util::VectorNormalize( &pNowHistory->emat, pNowHistory->emat );

                if( pStripe->historyCount == 1 )
                {
                    // 2つ目の履歴点ができる瞬間、初回の履歴点を2回目と揃えておく。
                    pNowHistory->pNext->emat = pNowHistory->emat;
                }
            }
        }
#endif
    }
    if( pStripe->historyCount < pStripeData->numHistory )
    {
        pStripe->historyCount++;
    }

    if( pStripe->historyCount >= 2 )
    {
        // 方向補間率による補正

        History* pPrevHistory = pNowHistory->pNext;    // 一つ前の履歴（奥ほど古い）
        nn::util::Vector4fType tempDir;
        nn::util::VectorSubtract( &tempDir, pNowHistory->pos, pPrevHistory->pos );
        nn::util::Vector3fType direction;
        detail::Vector3fCopy( &direction, tempDir );
        if( pStripeData->dirInterpolate > 0 )
        {
            // 方向補間処理（テスト）
            if( pStripe->time < 2 )
            {
                pStripe->vInterpolate = direction;
                if( nn::util::VectorLengthSquared( pStripe->vInterpolate ) > 0.0f )
                {
                    nn::util::VectorNormalize( &pStripe->vInterpolate, pStripe->vInterpolate );
                }
                // ストライプの進行方向
                pNowHistory->dir = pStripe->vInterpolate;
            }
            else
            {
                if( nn::util::VectorLengthSquared( direction ) > 0.0f )
                {
                    nn::util::VectorNormalize( &direction, direction );
                }

                nn::util::Vector3fType temp;
                nn::util::VectorSubtract( &temp, direction, pStripe->vInterpolate );
                nn::util::VectorMultiply( &temp, temp, pStripeData->dirInterpolate );
                nn::util::VectorAdd( &pStripe->vInterpolate, pStripe->vInterpolate, temp );

                if( nn::util::VectorLengthSquared( pStripe->vInterpolate ) > 0.0f )
                {
                    nn::util::VectorNormalize( &pStripe->vInterpolate, pStripe->vInterpolate );
                }
                // ストライプの進行方向
                pNowHistory->dir = pStripe->vInterpolate;
            }
        }
        else
        {
            pNowHistory->dir = direction;
        }

        if( pStripe->historyCount == 2 )
        {
            // ヒストリが2つの時、（≒一番最初に描画されるとき）
            // dirはhistNowの方向に合わせておく！
            // ※以降、同じ方向のまま後ろに移動していくので面倒を見るのはこのタイミングだけでOK。
            pPrevHistory->dir = pNowHistory->dir;
        }

        //---------------------------------------------------------------------------
        // ポリゴンバッファを生成
        //---------------------------------------------------------------------------

        // 今更新中のストライプが書くべき頂点バッファの先頭を取得
        VertexAttribute* pVertexAttribute = &pVertexBufferHead[ pStripe->indexOffset * pEPUserData->vertexCountPerStripe ];
        {
            float textureBaseValue = 1;             //テクスチャ座標を決めるための分母
            if( pStripeData->texturing == 0 )
            {
                // 「全て貼る」
                textureBaseValue = static_cast< float >( pStripe->historyCount ); // 今あるストライプの数
            }
            else if( pStripeData->texturing == 1 )
            {
                // 「描画範囲に合わせて貼る」
                textureBaseValue = pStripeData->numHistory;    // ストライプの総数
            }

            const float scaleFadeValue = pEmitter->CalculateCurrentScaleFadeValue();

            if( pStripeData->numDivide == 0 )
            {
                // 分割無し
                MakeStripePolygon( pStripe, pStripeData, pVertexAttribute, textureBaseValue, scaleFadeValue );
            }
            else
            {
                // 分割あり
                MakeStripePolygonWithDivision( pStripe, pStripeData, pVertexAttribute, textureBaseValue, scaleFadeValue );
            }
        }
    }
    arg.pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.Unmap();

} // NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  遅延ストライプ計算処理を行います。
//---------------------------------------------------------------------------
inline void StripeSystem::CalculateDelayedStripe( Emitter* pEmitter ) NN_NOEXCEPT
{
    EmitterPluginUserData* pEPUserData = StripeSystem::GetEmitterUserData( pEmitter );
    if( pEPUserData->pDelayedStripeListHead == NULL )
    {
        return;
    }
    if( pEPUserData->vertexBufferInitialized == false )
    {
        return;
    }

    StripeInstance* pPreStripe = NULL;
    StripeInstance* pStripe = pEPUserData->pDelayedStripeListHead;
    ResStripeHistory* pStripeData = GetStripeData( pEmitter );
    VertexAttribute* pVertexBufferHead = reinterpret_cast< VertexAttribute* >( pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.Map( pEPUserData->bufferSide ) );

    while( pStripe )
    {
        VertexAttribute* pVertexAttribute = &pVertexBufferHead[ pStripe->indexOffset * pEPUserData->vertexCountPerStripe ];
        StripeInstance* pNextStripe = pStripe->pNextDelayedStripe;

        //---------------------------------------------------------------------------
        // 死亡判定
        //---------------------------------------------------------------------------
        if( pStripe->historyCount > 1 )
        {
            pStripe->historyCount--;

            const float scaleFadeValue = pEmitter->CalculateCurrentScaleFadeValue();

            // 遅延中はストライプタイムを進める
            pStripe->time += pEmitter->GetFrameRate();
            {
                const float textureBaseValue = static_cast< float >( pStripe->historyCount ); // 今あるストライプの数
                if( pStripeData->numDivide == 0 )
                {
                    // 分割無し
                    MakeStripePolygon( pStripe, pStripeData, pVertexAttribute, textureBaseValue, scaleFadeValue );
                }
                else
                {
                    // 分割あり
                    MakeStripePolygonWithDivision( pStripe, pStripeData, pVertexAttribute, textureBaseValue, scaleFadeValue );
                }
            }
            pPreStripe = pStripe;
            pStripe = pNextStripe;
        }
        else
        {
            if( pPreStripe )
            {
                pPreStripe->pNextDelayedStripe = pNextStripe;
            }
            else
            {
                pEPUserData->pDelayedStripeListHead = pNextStripe;
            }
            StripeSystemUtility::FreeStripe( g_pStripeSystem, pStripe, pEmitter );
            pStripe = pNextStripe;
        }
    }

    pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.Unmap();
}

//---------------------------------------------------------------------------
//  ストライプのポリゴン生成を行います。
//  vertexBuffer には、書き込むべき領域の先頭アドレスが収まっている前提。
//---------------------------------------------------------------------------
void StripeSystem::MakeStripePolygon( StripeInstance* pStripe, ResStripeHistory* pStripeData, VertexAttribute* pVertexBuffer, float textureBaseValue, float scaleFadeValue ) NN_NOEXCEPT
{
    //---------------------------------------------------------------------------
    // 履歴ポイントが2つは無いとポリゴンにならないので処理を返す。
    //---------------------------------------------------------------------------
    const int historyCount = static_cast< int >( pStripe->historyCount );
    if( historyCount <= 2 )
    {
        pStripe->renderVertexCount = 0;
        return;
    }

    //---------------------------------------------------------------------------
    // 頂点バッファの必須の要素を埋める。
    //---------------------------------------------------------------------------
    const History* pHistory = pStripe->pHistoryHead->pNext;     // 先頭は空なのでその次から。
    const float invTexBase = 1.0f / ( textureBaseValue - 1 );   // ストライプ末尾でv=1.0になってほしいので-1する

    int index;
    for( index = 0; index < historyCount; ++index, pHistory = pHistory->pNext )
    {
#define _VFX_STRIPE_MOD
#ifndef _VFX_STRIPE_MOD
        // 先頭をインデックス0にしたいので降順でループさせる
        const int iLeft = index * 2;
        const int iRight = iLeft + 1;

        nn::util::VectorStore( &pVertexBuffer[ iLeft ].pos, pHistory->pos );    // 翼の長さのコピーも一緒に行う！
        nn::util::VectorStore( &pVertexBuffer[ iRight ].pos, pHistory->pos );   // 要素 w に値がコピーされているので次で上書きする。
        pVertexBuffer[ iRight ].pos.w = -nn::util::VectorGetW( pHistory->pos ); // 翼の長さ（※マイナスにするので別途行う）

        // スケールフェード
        pVertexBuffer[ iLeft ].pos.w  *= scaleFadeValue;
        pVertexBuffer[ iRight ].pos.w *= scaleFadeValue;

        nn::util::VectorStore( pVertexBuffer[ iLeft ].dir.v, pHistory->dir );
        nn::util::VectorStore( pVertexBuffer[ iRight ].dir.v, pHistory->dir );

        // テクスチャ座標
        pVertexBuffer[ iRight ].dir.w = pVertexBuffer[ iLeft ].dir.w = ( static_cast< float >( index ) * invTexBase );

        // 頂点インデックス
        pVertexBuffer[ iRight ].emat.w = pVertexBuffer[ iLeft ].emat.w = static_cast< float >( index );
#else
        NN_UNUSED( pStripeData );
        VertexAttribute attr[ 2 ] = {};
        attr[ 0 ].pos.x = attr[ 1 ].pos.x = nn::util::VectorGetX( pHistory->pos );
        attr[ 0 ].pos.y = attr[ 1 ].pos.y = nn::util::VectorGetY( pHistory->pos );
        attr[ 0 ].pos.z = attr[ 1 ].pos.z = nn::util::VectorGetZ( pHistory->pos );
        attr[ 0 ].pos.w = attr[ 1 ].pos.w = nn::util::VectorGetW( pHistory->pos );
        attr[ 1 ].pos.w *= -1.0f;
        attr[ 0 ].pos.w *= scaleFadeValue;
        attr[ 1 ].pos.w *= scaleFadeValue;

        attr[ 0 ].dir.y = attr[ 1 ].dir.y = nn::util::VectorGetY( pHistory->dir );
        attr[ 0 ].dir.z = attr[ 1 ].dir.z = nn::util::VectorGetZ( pHistory->dir );
        attr[ 0 ].dir.x = attr[ 1 ].dir.x = nn::util::VectorGetX( pHistory->dir );
        attr[ 0 ].dir.w = attr[ 1 ].dir.w = ( static_cast< float >( index ) * invTexBase );

        attr[ 0 ].emat.x = attr[ 1 ].emat.x = nn::util::VectorGetX( pHistory->emat );
        attr[ 0 ].emat.y = attr[ 1 ].emat.y = nn::util::VectorGetY( pHistory->emat );
        attr[ 0 ].emat.z = attr[ 1 ].emat.z = nn::util::VectorGetZ( pHistory->emat );
        attr[ 0 ].emat.w = attr[ 1 ].emat.w = static_cast< float >( index );
        memcpy( &pVertexBuffer[ index * 2 ].pos, &attr, sizeof( VertexAttribute ) * 2 );
#endif
    }
    pStripe->renderVertexCount = index * 2;

#ifndef _VFX_STRIPE_MOD
    //---------------------------------------------------------------------------
    // Ematが必要なストライプタイプの場合、emat要素も埋める。
    //---------------------------------------------------------------------------
    if( pStripeData->calcType != StripeOrientationType_Billboard )
    {
        pHistory = pStripe->pHistoryHead->pNext;
        for( index = 0; index < historyCount; ++index, pHistory = pHistory->pNext )
        {
            const int iLeft = index * 2;
            const int iRight = iLeft + 1;
            nn::util::VectorStore( pVertexBuffer[ iLeft ].emat.v, pHistory->emat );
            nn::util::VectorStore( pVertexBuffer[ iRight ].emat.v, pHistory->emat );
        }
    }
#endif

    return;
}

//---------------------------------------------------------------------------
//  ストライプのポリゴン生成を行います。（ストライプ分割版）
//---------------------------------------------------------------------------
void StripeSystem::MakeStripePolygonWithDivision( StripeInstance* pStripe, ResStripeHistory* pStripeData, VertexAttribute* pVertexBuffer, float textureBaseValue, float scaleFadeValue ) NN_NOEXCEPT
{
    //---------------------------------------------------------------------------
    // 履歴ポイントが2つは無いとポリゴンにならないので処理を返す。
    //---------------------------------------------------------------------------
    const int historyCount = static_cast< int >( pStripe->historyCount );
    if( historyCount <= 2 )
    {
        pStripe->renderVertexCount = 0;
        return;
    }

    const History* pHistory = pStripe->pHistoryHead->pNext;             // 先頭は空なのでその次から。
    const int divCount = static_cast< int >( pStripeData->numDivide );  // 分割数。
    const int divPointCount = ( ( historyCount - 1 ) * divCount );      // 分割により発生する履歴ポイントの数
    const int drawHistoryCount = historyCount + divPointCount;          // 分割分の頂点を追加。
    const float texBase = ( textureBaseValue - 1 ) + divPointCount;     // ストライプ末尾でv=1.0になってほしいので-1する
    const float fIndexDiv = 1.0f / ( 1 + divCount );
    const float invTexBase = 1.0f / texBase;

    //---------------------------------------------------------------------------
    // 頂点バッファの必須の要素を埋める。
    //---------------------------------------------------------------------------
    int index;
    int prevPos = 0;
    for( index = 0; index < drawHistoryCount; ++index )
    {
        const int iLeft = index * 2;
        const int iRight = iLeft + 1;

        const float fIndex = index * fIndexDiv;                             // ストライプ状の位置（履歴ポイントが整数になる）
        const int histPos = static_cast< int >( ::std::floorf( fIndex ) );  // 所属する履歴ポイント
        if( histPos > prevPos )
        {
            pHistory = pHistory->pNext;
            prevPos = histPos;
        }
        const float innerPos = fIndex - histPos;                            // 履歴ポイント内の位置 [0.0f, 1.0f]

        //-----------------------------------------------------------------
        // 履歴点の間をエルミート補間
        //-----------------------------------------------------------------
        nn::util::Vector3fType curvePos;    // 履歴ポイントの位置（エルミート補間後）
        nn::util::Vector3fType curveDir;    // 履歴ポイントの向き（線形補間後）
        StripeSystemUtility::CalculateHermiteInterpolatedCurveVec<StripeSystem>( &curvePos, &curveDir, pHistory, histPos, historyCount, innerPos );

        nn::util::VectorStore( pVertexBuffer[ iLeft ].pos.v, curvePos );        // 背骨の位置
        nn::util::VectorStore( pVertexBuffer[ iRight ].pos.v, curvePos );       // 背骨の位置
        pVertexBuffer[ iLeft ].pos.w = nn::util::VectorGetW( pHistory->pos );   // 翼の長さ
        pVertexBuffer[ iRight ].pos.w = -nn::util::VectorGetW( pHistory->pos ); // 翼の長さ（※マイナスにするので別途行う）

        // スケールフェード
        pVertexBuffer[ iLeft ].pos.w  *= scaleFadeValue;
        pVertexBuffer[ iRight ].pos.w *= scaleFadeValue;

        nn::util::VectorStore( pVertexBuffer[ iLeft ].dir.v, curveDir );
        nn::util::VectorStore( pVertexBuffer[ iRight ].dir.v, curveDir );

        // テクスチャ座標
        pVertexBuffer[ iRight ].dir.w = pVertexBuffer[ iLeft ].dir.w = ( static_cast< float >( index ) * invTexBase );

        // 頂点インデックス
        pVertexBuffer[ iRight ].emat.w = pVertexBuffer[ iLeft ].emat.w = static_cast< float >( index );
    }
    pStripe->renderVertexCount = index * 2;

    //---------------------------------------------------------------------------
    // Ematが必要なストライプタイプの場合、emat要素も埋める。
    //---------------------------------------------------------------------------
    if( pStripeData->calcType != StripeOrientationType_Billboard )
    {
        pHistory = pStripe->pHistoryHead->pNext;
        prevPos = 0;
        for( index = 0; index < drawHistoryCount; ++index )
        {
            const int iLeft = index * 2;
            const int iRight = iLeft + 1;

            const float fIndex = index * fIndexDiv;                             // ストライプ状の位置（履歴ポイント場が整数になる）
            const int histPos = static_cast< int >( ::std::floorf( fIndex ) );  // 所属する履歴ポイント
            if( histPos > prevPos )
            {
                pHistory = pHistory->pNext;
                prevPos = histPos;
            }

#ifndef _VFX_NSDK01215_COMPATIBLE
            // emat 要素は線形補間＋ Normalize する
            const float innerPos = fIndex - histPos;
            nn::util::Vector3fType emat;
            nn::util::VectorLerp( &emat, pHistory->emat, pHistory->pNext->emat, innerPos );
            nn::util::VectorNormalize( &emat, emat );

            nn::util::VectorStore( pVertexBuffer[ iLeft ].emat.v, emat );
            nn::util::VectorStore( pVertexBuffer[ iRight ].emat.v, emat );
#else
            nn::util::VectorStore( pVertexBuffer[ iLeft ].emat.v, pHistory->emat );
            nn::util::VectorStore( pVertexBuffer[ iRight ].emat.v, pHistory->emat );
#endif
        }
    }

    return;
}

//------------------------------------------------------------------------------
//  ワンタイムエミッタ用の追加のエミッタ寿命を取得
//------------------------------------------------------------------------------
int StripeSystem::GetExtendedEndTimeForOneTimeEmitter( nn::vfx::Emitter* pEmitter ) NN_NOEXCEPT
{
    ResStripeHistory* pStripeData = GetStripeData( pEmitter );
    float extendLife = pStripeData->numHistory * pEmitter->GetFrameRate();
    return static_cast< int >( extendLife );
}

//------------------------------------------------------------------------------
//  ストライプの正確な再生本数を取得
//------------------------------------------------------------------------------
int StripeSystem::GetActualStripeCalclationCount( const nn::vfx::Emitter* pEmitter ) NN_NOEXCEPT
{
    EmitterPluginUserData* pEPUserData = StripeSystem::GetEmitterUserData( pEmitter );
    if ( pEPUserData->vertexBufferInitialized == false )
    {
        return 0;
    }

    int actualStripeCount = pEmitter->GetProcessingParticleCount();

    StripeInstance* pStripe = pEPUserData->pDelayedStripeListHead;
    while ( pStripe )
    {
        actualStripeCount++;
        pStripe = pStripe->pNextDelayedStripe;
    }

    return actualStripeCount;
}



//---------------------------------------------------------------------------
//  ConstantBuffer を作成します。
//---------------------------------------------------------------------------
void StripeSystem::MakeConstantBufferObject(
    ConstantBufferObject* const pConstantBuffer,
    const ResStripeHistory* pStripeData,
    const EmitterPluginUserData* pEPUserData,
    const StripeInstance* pStripe,
    const Emitter* pEmitter,
    StripeMeshType meshType ) NN_NOEXCEPT
{
    NN_UNUSED( pEmitter );

    // 静的な要素
    nn::util::VectorStore( &pConstantBuffer->random, pStripe->random );
    pConstantBuffer->param0 = pEPUserData->staticParam0;

    // 動的な要素
    nn::util::VectorStore( &pConstantBuffer->color0, pStripe->color0 );
    nn::util::VectorStore( &pConstantBuffer->color1, pStripe->color1 );
    pConstantBuffer->param1.x = pStripe->time;
    pConstantBuffer->param1.y = static_cast< float >( pStripe->historyCount + pStripeData->numDivide * ( pStripe->historyCount - 1 ) );
    pConstantBuffer->param1.z = static_cast< float >( meshType );
    pConstantBuffer->param1.w = pStripe->life;
#ifndef _VFX_NSDK01215_COMPATIBLE
    pConstantBuffer->param2.x = nn::util::VectorGetX( pStripe->headPos );
    pConstantBuffer->param2.y = nn::util::VectorGetY( pStripe->headPos );
    pConstantBuffer->param2.z = nn::util::VectorGetZ( pStripe->headPos );
    pConstantBuffer->param2.w = 0.0f;
#else
    pConstantBuffer->param2.x = 0.0f;
    pConstantBuffer->param2.y = 0.0f;
    pConstantBuffer->param2.z = 0.0f;
    pConstantBuffer->param2.w = 0.0f;
#endif
}


} // namespace detail
} // namespace vfx
} // namespace nn

