﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_Enum.h>
#include <nn/vfx/vfx_Shader.h>

namespace nn {
namespace vfx {
namespace detail {


//---------------------------------------------------------------------------
//  コンピュートシェーダを初期化します。
//---------------------------------------------------------------------------
bool ComputeShader::Initialize( nn::gfx::Shader* computeShader ) NN_NOEXCEPT
{
    m_pComputeShader = computeShader;

    m_EmitterDynamicBufferSlot      = m_pComputeShader->GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_ConstantBuffer, "sysEmitterDynamicUniformBlock" );
    m_EmitterStaticBufferSlot       = m_pComputeShader->GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_ConstantBuffer, "sysEmitterStaticUniformBlock" );
    m_EmitterFieldBufferSlot        = m_pComputeShader->GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_ConstantBuffer, "sysEmitterFieldUniformBlock" );
    m_CurlNoiseTexSamplerSlot       = m_pComputeShader->GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_Sampler, "sysCurlNoiseTextureArray" );

    m_BufferSlot[ComputeShaderAttributeBufferIndex_Pos]         = m_pComputeShader->GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_UnorderedAccessBuffer, "sysPosBuff" );
    m_BufferSlot[ComputeShaderAttributeBufferIndex_Vec]         = m_pComputeShader->GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_UnorderedAccessBuffer, "sysVecBuff" );
    m_BufferSlot[ComputeShaderAttributeBufferIndex_Diff]        = m_pComputeShader->GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_UnorderedAccessBuffer, "sysPosDeltaBuff" );
    m_BufferSlot[ComputeShaderAttributeBufferIndex_Scale]       = m_pComputeShader->GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_UnorderedAccessBuffer, "sysScaleBuff" );
    m_BufferSlot[ComputeShaderAttributeBufferIndex_Random]      = m_pComputeShader->GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_UnorderedAccessBuffer, "sysRandomBuff" );
    m_BufferSlot[ComputeShaderAttributeBufferIndex_MatrixSrt0]  = m_pComputeShader->GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_UnorderedAccessBuffer, "sysEmtMat0Buff" );
    m_BufferSlot[ComputeShaderAttributeBufferIndex_MatrixSrt1]  = m_pComputeShader->GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_UnorderedAccessBuffer, "sysEmtMat1Buff" );
    m_BufferSlot[ComputeShaderAttributeBufferIndex_MatrixSrt2]  = m_pComputeShader->GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_UnorderedAccessBuffer, "sysEmtMat2Buff" );
    return true;
}

//---------------------------------------------------------------------------
//  シェーダの終了処理を行います。
//---------------------------------------------------------------------------
void Shader::Finalize() NN_NOEXCEPT
{
    for ( int i = 0; i < ConstantBufferType_MaxType; i++ )
    {
        m_ConstantBufferSlot[ i ].Finalize();
    }
    for ( int i = 0; i < CustomShaderConstantBufferIndex_MaxIndexCount; i++ )
    {
        m_CustomShaderConstantBufferSlot[ i ].Finalize();
    }
}

//---------------------------------------------------------------------------
//  無効化
//---------------------------------------------------------------------------
void Shader::Invalidate() NN_NOEXCEPT
{
    m_IsInitialized = false;
}

//---------------------------------------------------------------------------
//  シェーダをシェーダバイナリから初期化します。
//---------------------------------------------------------------------------
bool Shader::InitializeShader( nn::gfx::ResShaderVariation* shaderVariation, nn::gfx::ShaderCodeType codeType ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( !m_IsInitialized );

    m_pShader = shaderVariation->GetResShaderProgram( codeType )->GetShader();
    NN_SDK_ASSERT_NOT_NULL( m_pShader );
    m_IsInitialized = true;
    return true;
}

//---------------------------------------------------------------------------
//  サンプラスロットを初期化します。
//---------------------------------------------------------------------------
void Shader::InitializeSamplerSlot() NN_NOEXCEPT
{
    m_TexSmpLoc[ TextureSlotId_0 ].Initialize( m_pShader, "sysTextureSampler0" );
    m_TexSmpLoc[ TextureSlotId_1 ].Initialize( m_pShader, "sysTextureSampler1" );
    m_TexSmpLoc[ TextureSlotId_2 ].Initialize( m_pShader, "sysTextureSampler2" );
    m_FrameBuferTexSmpLoc.Initialize( m_pShader, "sysFrameBufferTexture" );
    m_DepthBuferTexSmpLoc.Initialize( m_pShader, "sysDepthBufferTexture" );
    m_CurlNoiseTexSmpLoc.Initialize( m_pShader, "sysCurlNoiseTextureArray" );
    m_CustomTextureLoc[ CustomShaderTextureType_2d0 ].Initialize( m_pShader, "sysCustomShaderTextureSampler0" );
    m_CustomTextureLoc[ CustomShaderTextureType_2d1 ].Initialize( m_pShader, "sysCustomShaderTextureSampler1" );
    m_CustomTextureLoc[ CustomShaderTextureType_2d2 ].Initialize( m_pShader, "sysCustomShaderTextureSampler2" );
    m_CustomTextureLoc[ CustomShaderTextureType_2d3 ].Initialize( m_pShader, "sysCustomShaderTextureSampler3" );
    m_CustomTextureLoc[ CustomShaderTextureType_2dArray0 ].Initialize( m_pShader, "sysCustomShaderTextureArraySampler0" );
    m_CustomTextureLoc[ CustomShaderTextureType_2dArray1 ].Initialize( m_pShader, "sysCustomShaderTextureArraySampler1" );
    m_CustomTextureLoc[ CustomShaderTextureType_2dArray2 ].Initialize( m_pShader, "sysCustomShaderTextureArraySampler2" );
    m_CustomTextureLoc[ CustomShaderTextureType_2dArray3 ].Initialize( m_pShader, "sysCustomShaderTextureArraySampler3" );
    m_CustomTextureLoc[ CustomShaderTextureType_3d0 ].Initialize( m_pShader, "sysCustomShaderTexture3DSampler0" );
    m_CustomTextureLoc[ CustomShaderTextureType_3d1 ].Initialize( m_pShader, "sysCustomShaderTexture3DSampler1" );
    m_CustomTextureLoc[ CustomShaderTextureType_3d2 ].Initialize( m_pShader, "sysCustomShaderTexture3DSampler2" );
    m_CustomTextureLoc[ CustomShaderTextureType_3d3 ].Initialize( m_pShader, "sysCustomShaderTexture3DSampler3" );
    m_CustomTextureLoc[ CustomShaderTextureType_CubeMap0 ].Initialize( m_pShader, "sysCustomShaderCubeSampler0" );
    m_CustomTextureLoc[ CustomShaderTextureType_CubeMap1 ].Initialize( m_pShader, "sysCustomShaderCubeSampler1" );
    m_CustomTextureLoc[ CustomShaderTextureType_CubeMap2 ].Initialize( m_pShader, "sysCustomShaderCubeSampler2" );
    m_CustomTextureLoc[ CustomShaderTextureType_CubeMap3 ].Initialize( m_pShader, "sysCustomShaderCubeSampler3" );
    m_CustomTextureLoc[ CustomShaderTextureType_CubeMapArray0 ].Initialize( m_pShader, "sysCustomShaderCubeArraySampler0" );
    m_CustomTextureLoc[ CustomShaderTextureType_CubeMapArray1 ].Initialize( m_pShader, "sysCustomShaderCubeArraySampler1" );
    m_CustomTextureLoc[ CustomShaderTextureType_CubeMapArray2 ].Initialize( m_pShader, "sysCustomShaderCubeArraySampler2" );
    m_CustomTextureLoc[ CustomShaderTextureType_CubeMapArray3 ].Initialize( m_pShader, "sysCustomShaderCubeArraySampler3" );
    m_CustomTextureLoc[ CustomShaderTextureType_ShadowMap0 ].Initialize( m_pShader, "sysCustomShaderShadowSampler0" );
    m_CustomTextureLoc[ CustomShaderTextureType_ShadowMap1 ].Initialize( m_pShader, "sysCustomShaderShadowSampler1" );
    m_CustomTextureLoc[ CustomShaderTextureType_ShadowMap2 ].Initialize( m_pShader, "sysCustomShaderShadowSampler2" );
    m_CustomTextureLoc[ CustomShaderTextureType_ShadowMap3 ].Initialize( m_pShader, "sysCustomShaderShadowSampler3" );
    m_CustomTextureLoc[ CustomShaderTextureType_ShadowMapArray0 ].Initialize( m_pShader, "sysCustomShaderShadowArraySampler0" );
    m_CustomTextureLoc[ CustomShaderTextureType_ShadowMapArray1 ].Initialize( m_pShader, "sysCustomShaderShadowArraySampler1" );
    m_CustomTextureLoc[ CustomShaderTextureType_ShadowMapArray2 ].Initialize( m_pShader, "sysCustomShaderShadowArraySampler2" );
    m_CustomTextureLoc[ CustomShaderTextureType_ShadowMapArray3 ].Initialize( m_pShader, "sysCustomShaderShadowArraySampler3" );
}

//---------------------------------------------------------------------------
//  定数バッファスロットの初期化
//---------------------------------------------------------------------------
void Shader::InitializeConstantBufferSlot() NN_NOEXCEPT
{
    for( int i = 0; i < ConstantBufferType_MaxType; i++ )
    {
        m_ConstantBufferSlot[ i ].Invalidate();
    }
    for( int i = 0; i < CustomShaderConstantBufferIndex_MaxIndexCount; i++ )
    {
        m_CustomShaderConstantBufferSlot[ i ].Invalidate();
    }

    m_ConstantBufferSlot[ ConstantBufferType_View ].Initialize( m_pShader, "sysViewUniformBlock" );
    m_ConstantBufferSlot[ ConstantBufferType_EmitterStatic ].Initialize( m_pShader, "sysEmitterStaticUniformBlock" );
    m_ConstantBufferSlot[ ConstantBufferType_EmitterDynamic ].Initialize( m_pShader, "sysEmitterDynamicUniformBlock" );
    m_ConstantBufferSlot[ ConstantBufferType_EmitterField ].Initialize( m_pShader, "sysEmitterFieldUniformBlock" );
    m_ConstantBufferSlot[ ConstantBufferType_EmitterPlugin ].Initialize( m_pShader, "sysEmitterPluginUniformBlock" );
    m_ConstantBufferSlot[ ConstantBufferType_ReservedParam ].Initialize( m_pShader, "sysCustomShaderReservedUniformBlockParam" );
    m_CustomShaderConstantBufferSlot[ CustomShaderConstantBufferIndex_0 ].Initialize( m_pShader, "sysCustomShaderUniformBlock0" );
    m_CustomShaderConstantBufferSlot[ CustomShaderConstantBufferIndex_1 ].Initialize( m_pShader, "sysCustomShaderUniformBlock1" );
    m_CustomShaderConstantBufferSlot[ CustomShaderConstantBufferIndex_2 ].Initialize( m_pShader, "sysCustomShaderUniformBlock2" );
    m_CustomShaderConstantBufferSlot[ CustomShaderConstantBufferIndex_3 ].Initialize( m_pShader, "sysCustomShaderUniformBlock3" );

    if( m_ConstantBufferSlot[ ConstantBufferType_View ].GetVertexShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_VertexView;
    }
    if( m_ConstantBufferSlot[ ConstantBufferType_View ].GetPixelShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_PixelView;
    }

    if( m_ConstantBufferSlot[ ConstantBufferType_EmitterStatic ].GetVertexShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_VertexEmitterStatic;
    }
    if( m_ConstantBufferSlot[ ConstantBufferType_EmitterStatic ].GetPixelShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_PixelEmitterStatic;
    }

    if( m_ConstantBufferSlot[ ConstantBufferType_EmitterDynamic ].GetVertexShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_VertexEmitterDynamic;
    }
    if( m_ConstantBufferSlot[ ConstantBufferType_EmitterDynamic ].GetPixelShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_PixelEmitterDynamic;
    }
    if( m_ConstantBufferSlot[ ConstantBufferType_EmitterField ].GetVertexShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_VertexEmitterField;
    }
    if( m_ConstantBufferSlot[ ConstantBufferType_EmitterField ].GetPixelShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_PixelField;
    }
    if( m_ConstantBufferSlot[ ConstantBufferType_EmitterPlugin ].GetVertexShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_VertexEmitterPlugin;
    }
    if( m_ConstantBufferSlot[ ConstantBufferType_EmitterPlugin ].GetPixelShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_PixelEmitterPlugin;
    }
    if( m_ConstantBufferSlot[ ConstantBufferType_ReservedParam ].GetVertexShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_VertexReservedParam;
    }
    if( m_ConstantBufferSlot[ ConstantBufferType_ReservedParam ].GetPixelShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_PixelReservedParam;
    }
    if( m_ConstantBufferSlot[ CustomShaderConstantBufferIndex_0 ].GetVertexShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_VertexCustomConstantBuffer0;
    }
    if( m_ConstantBufferSlot[ CustomShaderConstantBufferIndex_0 ].GetPixelShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_PixelCustomConstantBuffer0;
    }
    if( m_ConstantBufferSlot[ CustomShaderConstantBufferIndex_1 ].GetVertexShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_VertexCustomConstantBuffer1;
    }
    if( m_ConstantBufferSlot[ CustomShaderConstantBufferIndex_1 ].GetPixelShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_PixelCustomConstantBuffer1;
    }
    if( m_ConstantBufferSlot[ CustomShaderConstantBufferIndex_2 ].GetVertexShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_VertexCustomConstantBuffer2;
    }
    if( m_ConstantBufferSlot[ CustomShaderConstantBufferIndex_2 ].GetPixelShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_PixelCustomConstantBuffer2;
    }
    if( m_ConstantBufferSlot[ CustomShaderConstantBufferIndex_3 ].GetVertexShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_VertexCustomConstantBuffer3;
    }
    if( m_ConstantBufferSlot[ CustomShaderConstantBufferIndex_3 ].GetPixelShaderSlot() != InvalidValueId_ConstantBufferSlotId )
    {
        m_ConstantBufferEnableFlag |= ConstantBufferEnabledBitFlag_PixelCustomConstantBuffer3;
    }
}

//---------------------------------------------------------------------------
//  カスタムシェーダユニフォームブロックをバインドします。
//---------------------------------------------------------------------------
bool Shader::BindCustomShaderUniformBlock( nn::gfx::CommandBuffer* pCommandBuffer, CustomShaderConstantBufferIndex index, nn::gfx::GpuAddress* address, size_t uniformBlockSize ) NN_NOEXCEPT
{
    bool ret = false;

    int staticConstantBufferSlotV = GetCustomShaderVertexConstantBufferSlot( index );
    int staticConstantBufferSlotP = GetCustomShaderPixelConstantBufferSlot( index );

    if( staticConstantBufferSlotV != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( staticConstantBufferSlotV, nn::gfx::ShaderStage_Vertex, *address, uniformBlockSize );
        ret = true;
    }
    if( staticConstantBufferSlotP != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( staticConstantBufferSlotP, nn::gfx::ShaderStage_Pixel, *address, uniformBlockSize );
        ret = true;
    }

    return ret;
}


} // namespace detail
} // namespace vfx
} // namespace nn

