﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_System.h>
#include <nn/vfx/vfx_Misc.h>
#include <nn/vfx/vfx_Resource.h>
#include <nn/vfx/vfx_Macro.h>
#include <nn/vfx/vfx_Callback.h>
#include <nn/vfx/vfx_MemUtil.h>
#include <nn/vfx/vfx_RenderContext.h>
#include <nn/vfx/vfx_ShaderFlag.h>

namespace nn {
namespace vfx {

//------------------------------------------------------------------------------
//  コンストラクタ
//------------------------------------------------------------------------------
Resource::Resource( nn::gfx::Device* pDevice, Heap* pHeap, void* pBinary, nn::gfx::MemoryPool* pMemoryPool, size_t memoryPoolSize, size_t memoryPoolOffset, int resourceId, System* pSystem, bool shaderDelaySetup, Resource* residentResource ) NN_NOEXCEPT
{
    m_pDevice                           = pDevice;
    m_pBinaryTop                        = pBinary;
    m_pSystem                           = pSystem;
    m_pResidentResource                 = residentResource;
    m_pMemoryPool                       = pMemoryPool;
    m_MemoryPoolSize                    = memoryPoolSize;
    m_MemoryPoolOffset                  = memoryPoolOffset;
    m_pEmitterResArray                  = nullptr;
    m_PrimitiveAllocatedSize              = 0;
    m_G3dPrimitiveAllocatedSize           = 0;
    m_EmitterSetAllocatedSize             = 0;
    m_EmitterAllocatedSize                = 0;
    m_ShaderAllocatedSize                 = 0;
    m_TextureAllocatedSize                = 0;
    m_ConstantBufferAllocatedSize       = 0;
    m_ResourceId                        = resourceId;
    m_EmitterSetCount                   = 0;
    m_EmitterCount                      = 0;
    m_EmitterSetResArray                = NULL;
    m_BindEmitterSetResArray            = NULL;
    m_PrimitiveCount                    = 0;
    m_PrimitiveArray                    = NULL;
    m_G3dPrimitiveCount                 = 0;
    m_G3dPrimitiveArrayPtr              = NULL;
    m_G3dPrimitiveArray                 = NULL;
    m_pG3dResFile                       = NULL;
    m_TextureCount                      = 0;
    m_ShaderDelaySetup                  = shaderDelaySetup;
    m_EmitterResourceSize               = 0;
    m_TextureResourceSize               = 0;
    m_ShaderResourceSize                = 0;
    m_PrimitiveResourceSize             = 0;
    m_G3dPrimitiveResourceSize          = 0;
    m_EmitterCount                      = 0;
    m_IsAlive                           = true;
    m_PrimitiveArrayPtr                 = NULL;
    m_ResTextureFile                    = NULL;
    m_pTextureTable                     = NULL;
    m_pComputeResShaderFile             = NULL;
    m_pExtComputeResShaderFile          = nullptr;
    m_ComputeShaderResourceSize         = 0;
    m_pResShaderFile                    = nullptr;
    m_ResHeap.SetHeap( pHeap );
    m_pExtResShaderFile                 = nullptr;
    m_pExtComputeResShaderFile          = nullptr;
    m_pVertexStateBuffer                = nullptr;
    nn::util::BinaryFileHeader* pHeader = reinterpret_cast< nn::util::BinaryFileHeader* >( m_pBinaryTop );
    nn::util::BinFileSignature tag;
    tag.Set( "VFXB    " );

    if( pHeader->IsSignatureValid( tag.GetPacked() ) == false )
    {
        detail::OutputError( "Binary Tag Error.\n" );
    }

#ifdef _VFX_NSDK01215_COMPATIBLE
    if ( !( SystemParameters_BinaryVersionId == pHeader->version.major ||
            SystemParameters_CompatibleBinaryVersionId == pHeader->version.major ) )
    {
        // メジャーバージョンにバイナリバージョンが入る
        detail::OutputLog( "Data               Version:%d\n", pHeader->version.major );
        detail::OutputLog( "Runtime            Version:%d\n", SystemParameters_BinaryVersionId );
        detail::OutputLog( "Runtime Compatible Version:%d\n", SystemParameters_CompatibleBinaryVersionId );
        detail::OutputError( "Binary Version Error!!! Please check your binary file.\n" );
    }
#else
    if( SystemParameters_BinaryVersionId != pHeader->version.major )
    {
        // メジャーバージョンにバイナリバージョンが入る
        detail::OutputLog( "Data    Version:%d\n", pHeader->version.major );
        detail::OutputLog( "Runtime Version:%d\n", SystemParameters_BinaryVersionId );
        detail::OutputError( "Binary Version Error!!! Please check your binary file.\n" );
    }
#endif

    if( pHeader->version.minor != nn::gfx::TargetConfig::Type::value )
    {
        // マイナーバージョンにApiTypeが入る
        detail::OutputLog( "Data    API:%d\n", pHeader->version.minor );
        detail::OutputLog( "Runtime API:%d\n", nn::gfx::TargetConfig::Type::value );
        detail::OutputError( "Graphic API Unmatched Error!!! Please check your binary file.\n" );
    }
    if( pHeader->IsAlignmentValid() == false )
    {
        detail::OutputLog( "Required Alignment:%d\n", pHeader->GetAlignment() );
        detail::OutputLog( "               Gap:%d\n", reinterpret_cast< uint64_t >( m_pBinaryTop ) % pHeader->GetAlignment() );
        detail::OutputError( "Alignment Unmatched Error!!! Please check your binary file.\n" );
    }

#if defined( NN_BUILD_CONFIG_OS_COS )
    NN_SDK_ASSERT( !( reinterpret_cast< uint32_t >( pBinary ) % SystemParameters_ParticleBinaryDataAlignment ), "[VFX] Binary Alignment Error!  VFX_PTCL_BINARY_ALIGNMENT!!" );
#endif

    // offsetToFirstBlockに最初のルートブロックが入る
    void* pBinaryTreeTop = ( reinterpret_cast< uint8_t* >( m_pBinaryTop ) + pHeader->_offsetToFirstBlock );

    // バイナリ内をトレースする
    detail::BinaryData* top = reinterpret_cast< detail::BinaryData* >( pBinaryTreeTop );
    Trace( top );
}

//---------------------------------------------------------------------------
//  エフェクトリソースの破棄を行います。
//---------------------------------------------------------------------------
Resource::~Resource() NN_NOEXCEPT
{
    Finalize( NULL );
}

//------------------------------------------------------------------------------
//  バイナリをトレースする
//------------------------------------------------------------------------------
void Resource::Trace( detail::BinaryData* pDataTop ) NN_NOEXCEPT
{
    detail::BinaryData* pData = pDataTop;

    m_EmitterCount      = 0;

    detail::BufferSizeCalculator constantBufferSizeCalculator( m_pDevice, nn::gfx::GpuAccess_ConstantBuffer );

    detail::ResEmitterSetFilePathTableRecord* pEmitterSetFilePathTableRecord = nullptr;

    // リソースのセットアップ
    while( pData )
    {
        // GfxTextureArray
        if (pData->GetBinaryTag() == VFX_MAKE_TAG( 'G', 'R', 'T', 'F' ) )
        {
            size_t tempSize = m_ResHeap.GetAllocatedSize();
            TraceGfxResTextureFile( pData );
            detail::BinaryData *table = pData->GetDirectChildData( VFX_MAKE_TAG( 'G', 'T', 'N', 'T' ) );
            m_pTextureTable = table != NULL ? table->GetBinaryDataWithFlip<detail::ResNameTableRecord>() : NULL;
            m_TextureAllocatedSize = m_ResHeap.GetAllocatedSize() - tempSize;
        }

        // PrimitiveArray
        if( pData->GetBinaryTag() == VFX_MAKE_TAG( 'P', 'R', 'M', 'A' ) )
        {
            size_t tempSize = m_ResHeap.GetAllocatedSize();
            TracePrimitiveArray( pData );
            m_PrimitiveAllocatedSize = m_ResHeap.GetAllocatedSize() - tempSize;
        }

        // G3dPrimitiveArray
        if ( pData->GetBinaryTag() == VFX_MAKE_TAG( 'G', '3', 'P', 'R' ) )
        {
            size_t tempSize = m_ResHeap.GetAllocatedSize();
            detail::BinaryData *table = pData->GetDirectChildData( VFX_MAKE_TAG( 'G', '3', 'N', 'T' ) );
            m_G3dPrimitiveTable = table != NULL ? table->GetBinaryDataWithFlip<detail::ResG3dPrimitiveTableRecord>() : NULL;
            TraceG3dPrimitiveArray( pData );
            m_G3dPrimitiveAllocatedSize = m_ResHeap.GetAllocatedSize() - tempSize;
        }

        // GfxResShader
        if( pData->GetBinaryTag() == VFX_MAKE_TAG( 'G', 'R', 'S', 'N' ) )
        {
            size_t tempSize = m_ResHeap.GetAllocatedSize();
            TraceShaderBinaryArray( pData );
            m_ShaderAllocatedSize = m_ResHeap.GetAllocatedSize() - tempSize;
        }

        // エミッタリソースの初期化＆バッファサイズ計算を行う
        if( pData->GetBinaryTag() == VFX_MAKE_TAG( 'E', 'S', 'T', 'A' ) )
        {
            size_t tempSize = m_ResHeap.GetAllocatedSize();
            TraceEmitterSetArray( pData, &constantBufferSizeCalculator );
            m_EmitterSetAllocatedSize = m_ResHeap.GetAllocatedSize() - tempSize;
        }

        // エミッタセットのファイルパステーブル
        if (pData->GetBinaryTag() == VFX_MAKE_TAG( 'E', 'S', 'F', 'T' ) )
        {
            pEmitterSetFilePathTableRecord = pData->GetBinaryDataWithFlip<detail::ResEmitterSetFilePathTableRecord>();
        }

        pData = pData->GetNextData();
    }

    // エミッタ静的コンスタントバッファ用ワークを確保
    size_t tempSize = m_ResHeap.GetAllocatedSize();
    NN_SDK_ASSERT( m_EmitterCount != 0 );
    m_ResourceConstantBuffer.Initialize( m_pDevice, &m_ResHeap, nn::gfx::GpuAccess_ConstantBuffer, constantBufferSizeCalculator.GetBufferSize() );
    m_ConstantBufferAllocatedSize = m_ResHeap.GetAllocatedSize() - tempSize;

    // 頂点ステートバッファ用ワークを確保
    // ここでは、利用可能な最大ワークサイズで確保を行う
    const int maxVertexAttributeStateInfo = 16;
    const int maxVertexBufferStateInfo = 16;

    nn::gfx::VertexAttributeStateInfo vertexAttribs[ maxVertexAttributeStateInfo ];
    nn::gfx::VertexBufferStateInfo bufferStateInfo[ maxVertexBufferStateInfo ];

    nn::gfx::VertexState::InfoType info;
    info.SetDefault();
    info.SetVertexAttributeStateInfoArray( vertexAttribs, maxVertexAttributeStateInfo );
    info.SetVertexBufferStateInfoArray( bufferStateInfo, maxVertexBufferStateInfo );
    vertexAttribs[0].SetShaderSlot( 15 );   // アトリビュートを最大16本利用することを宣言
    size_t memorySize = nn::gfx::VertexState::GetRequiredMemorySize( info );
    memorySize = nn::util::align_up( memorySize, nn::gfx::VertexState::RequiredMemoryInfo_Alignment );

    size_t vertexStateBufferSize = m_EmitterCount * memorySize;
    m_pVertexStateBuffer = m_ResHeap.Alloc( vertexStateBufferSize, nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
    NN_SDK_ASSERT_NOT_NULL( m_pVertexStateBuffer );

    BufferCutter cutter;
    cutter.Initialize( m_pVertexStateBuffer, vertexStateBufferSize );

    // エミッタリソース内 コンスタントバッファやグラフィックスリソースの割り当て
    tempSize = m_ResHeap.GetAllocatedSize();
    detail::ResEmitterSetFilePathTableRecord* pCurrentEmitterSetFilePathRecord = pEmitterSetFilePathTableRecord;
    m_ResourceConstantBuffer.Begin();
    for( int i = 0; i < m_EmitterSetCount; i++ )
    {
        for( int j = 0; j < m_EmitterSetResArray[ i ].emitterAllCount; j++ )
        {
            // 頂点ステートバッファを割り当て
            m_EmitterSetResArray[ i ].pEmitterResource[ j ].m_pVertexStateBuffer = cutter.Cut( memorySize, nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
            m_EmitterSetResArray[ i ].pEmitterResource[ j ].m_VertexStateBufferSize = memorySize;

            InitializeEmitterGraphicsResource( &m_EmitterSetResArray[ i ].pEmitterResource[ j ] );
        }

        //  エミッタセットのファイルパステーブルバイナリをトレースする
        if (pCurrentEmitterSetFilePathRecord != nullptr)
        {
            m_EmitterSetResArray[ i ].pFilePath = pCurrentEmitterSetFilePathRecord->filePath;
            pCurrentEmitterSetFilePathRecord = pCurrentEmitterSetFilePathRecord->nextOffset == 0 ? NULL :
                reinterpret_cast<detail::ResEmitterSetFilePathTableRecord*>
                (reinterpret_cast<uint8_t*>(pCurrentEmitterSetFilePathRecord) + pCurrentEmitterSetFilePathRecord->nextOffset);
        }
    }
    m_ResourceConstantBuffer.End();
    m_EmitterAllocatedSize = m_ResHeap.GetAllocatedSize() - tempSize;
}

//------------------------------------------------------------------------------
//  終了処理
//------------------------------------------------------------------------------
void Resource::Finalize( Heap* pHeap ) NN_NOEXCEPT
{
    m_pResidentResource = NULL;

    if ( pHeap )
    {
        m_ResHeap.SetHeap( pHeap );
    }
    else
    {
        detail::OutputWarning( "Heap is not set in the argument of the resource destruction method." );
    }

    if ( m_pVertexStateBuffer )
    {
        m_ResHeap.Free( m_pVertexStateBuffer );
    }

    // m_pResShaderFile が存在する場合は、バイナリ内にシェーダバイナリを含む
    if ( m_pResShaderFile  )
    {
        m_pResShaderFile->GetShaderContainer()->Finalize( m_pDevice );
        m_pResShaderFile = NULL;
    }

    // m_pComputeResShaderFile が存在する場合は、バイナリ内にシェーダバイナリを含む
    if ( m_pComputeResShaderFile != NULL )
    {
        m_pComputeResShaderFile->GetShaderContainer()->Finalize( m_pDevice );
        m_pComputeResShaderFile = NULL;
    }

    m_ShaderManager.Finalize( m_pDevice, &m_ResHeap );
    m_ComputeShaderManager.Finalize( m_pDevice, &m_ResHeap );

    for( int i = 0; i < m_PrimitiveCount; i++ )
    {
        m_PrimitiveArray[ i ].Finalize();
    }
    m_PrimitiveArray = NULL;

    if( m_PrimitiveArrayPtr )
    {
        m_ResHeap.Free( m_PrimitiveArrayPtr );
        m_PrimitiveArrayPtr = NULL;
    }

    if ( m_G3dPrimitiveArrayPtr )
    {
        m_ResHeap.Free( m_G3dPrimitiveArrayPtr );
        m_G3dPrimitiveArrayPtr = NULL;
    }

    if ( m_pG3dResFile )
    {
        m_pG3dResFile->Cleanup( m_pDevice );
        m_pG3dResFile = NULL;
    }

    for ( int i = 0; i < m_TextureCount; i++ )
    {
        nn::gfx::ResTexture* resTexture = m_ResTextureFile->GetResTexture( i );
        resTexture->Finalize( m_pDevice );
    }

    if ( m_ResTextureFile )
    {
        m_ResTextureFile->Finalize( m_pDevice );
        m_ResTextureFile = NULL;
    }

    for( int i = 0; i < m_EmitterSetCount; i++ )
    {
        for( int j = 0; j < m_EmitterSetResArray[ i ].emitterAllCount; j++ )
        {
            FinalizeEmitterResource( &m_EmitterSetResArray[ i ].pEmitterResource[ j ] );
#if defined( NN_BUILD_CONFIG_OS_COS )
            if( m_EmitterSetResArray[ i ].pEmitterResource[ j ].m_pResEmitter->emitter.calcType == detail::EmitterCalculationMode_Cpu )
            {
                m_ResHeap.Free( m_EmitterSetResArray[ i ].pEmitterResource[ j ].m_ResEmitterStaticConstantBuffer );
            }
#endif
        }

        m_ResHeap.Free( m_EmitterSetResArray[i].pEmitterResourcePtr );
    }

    if( m_EmitterSetResArray )
    {
        m_ResHeap.Free( m_EmitterSetResArray );
        m_EmitterSetResArray = NULL;
    }

    if( m_BindEmitterSetResArray )
    {
        m_ResHeap.Free( m_BindEmitterSetResArray );
        m_BindEmitterSetResArray = NULL;
    }

    if ( m_ResourceConstantBuffer.IsValidate() )
    {
        m_ResourceConstantBuffer.Finalize( m_pDevice, &m_ResHeap );
    }

    if ( m_PrimitiveVertexBuffer.IsValidate() )
    {
        m_PrimitiveVertexBuffer.Finalize( m_pDevice, &m_ResHeap );
    }

    NN_SDK_ASSERT( m_ResHeap.GetAllocatedCount() == 0 );
}

//---------------------------------------------------------------------------
// リソースをアンリロケート
//---------------------------------------------------------------------------
void Resource::Unrelocate( void * pData ) NN_NOEXCEPT
{
    nn::util::BinaryFileHeader* pHeader = reinterpret_cast< nn::util::BinaryFileHeader* >( pData );
    detail::BinaryData* pDataTop = reinterpret_cast< detail::BinaryData* >( reinterpret_cast< uint8_t* >( pData ) + pHeader->_offsetToFirstBlock );

    // リソースの検索
    while( pDataTop )
    {
        // GfxTextureArray
        if (pDataTop->GetBinaryTag() == VFX_MAKE_TAG( 'G', 'R', 'T', 'F' ) )
        {
            if (pDataTop->GetBinarySize() > 32)
            {
                void* ptr = reinterpret_cast<void*>( pDataTop->GetBinaryData() );
                nn::gfx::ResTextureFileData* pResTextureFileData = static_cast<::nn::gfx::ResTextureFileData*>(ptr);
                if (pResTextureFileData != NULL) {
                    if (pResTextureFileData->fileHeader.IsRelocated())
                    {
                        pResTextureFileData->fileHeader.GetRelocationTable()->Unrelocate();
                        pResTextureFileData->fileHeader.SetRelocated(false);
                    }
                }
            }
        }

        // GfxResShader
        if( pDataTop->GetBinaryTag() == VFX_MAKE_TAG( 'G', 'R', 'S', 'N' ) )
        {
            detail::BinaryData* pCSData = pDataTop->GetDirectChildData( VFX_MAKE_TAG( 'G', 'R', 'S', 'C' ) );
            if ( pCSData )
            {
                nn::gfx::ResShaderFileData* pResShaderFileData = static_cast<::nn::gfx::ResShaderFileData*>(pCSData->GetBinaryData());
                if (pResShaderFileData != NULL)
                {
                    if ( pResShaderFileData->fileHeader.IsRelocated())
                    {
                        pResShaderFileData->fileHeader.GetRelocationTable()->Unrelocate();
                        pResShaderFileData->fileHeader.SetRelocated(false);
                    }
                }
            }

            void* ptr = reinterpret_cast<void*>( pDataTop->GetBinaryData() );
            nn::gfx::ResShaderFileData* pResShaderFileData = static_cast<::nn::gfx::ResShaderFileData*>(ptr);
            if (pResShaderFileData != NULL) {
                if ( pResShaderFileData->fileHeader.IsRelocated())
                {
                     pResShaderFileData->fileHeader.GetRelocationTable()->Unrelocate();
                     pResShaderFileData->fileHeader.SetRelocated(false);
                }
            }
        }

        // G3dPrimitiveArray
        if (pDataTop->GetBinaryTag() == VFX_MAKE_TAG('G', '3', 'P', 'R'))
        {
            void* ptr = reinterpret_cast<void*>(pDataTop->GetBinaryData());
            if ((pDataTop->GetBinarySize() > 32) && (pDataTop->GetBinaryData()))
            {
                nn::g3d::ResFile* pG3dResFile = static_cast<nn::g3d::ResFile *>(ptr);
                if (pG3dResFile != NULL)
                {
                    pG3dResFile->Unrelocate();
                }
            }
        }

        pDataTop = pDataTop->GetNextData();
    }
}

//------------------------------------------------------------------------------
//  GfxResTextureFile バイナリをトレースする
//------------------------------------------------------------------------------
void Resource::TraceGfxResTextureFile( detail::BinaryData* pData ) NN_NOEXCEPT
{
    if ( pData->GetBinarySize() <= 32 ) return;

    void* textureBinary = pData->GetBinaryData();

    m_TextureResourceSize = pData->GetBinarySize();

    nn::util::BytePtr binTop( m_pBinaryTop );

    NN_SDK_ASSERT( m_ResTextureFile == NULL ); // 既に NULL 以外が入っていると、Finalize() で解放できなくなる。
    m_ResTextureFile = nn::gfx::ResTextureFile::ResCast( textureBinary );
    NN_SDK_ASSERT_NOT_NULL( m_ResTextureFile );

    if ( m_pMemoryPool == nullptr )
    {
        m_ResTextureFile->Initialize( m_pDevice );
    }
    else
    {
        m_ResTextureFile->Initialize( m_pDevice, m_pMemoryPool, binTop.Distance( textureBinary ) + m_MemoryPoolOffset, m_MemoryPoolSize );
    }

    // テクスチャ数
    m_TextureCount = m_ResTextureFile->GetTextureDic()->GetCount();

    for ( int i = 0; i < m_TextureCount; i++ )
    {
        nn::gfx::ResTexture* resTexture = m_ResTextureFile->GetResTexture( i );
        resTexture->Initialize( m_pDevice );
    }

    return;
}

//------------------------------------------------------------------------------
//  テクスチャービューをディスクリプタプールに登録します。
//------------------------------------------------------------------------------
void Resource::RegisterTextureViewToDescriptorPool( nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    if (!pRegisterTextureSlotCallback) return;

    for (int i = 0; i < m_TextureCount; i++)
    {
        nn::gfx::ResTexture* resTexture = m_ResTextureFile->GetResTexture(i);
        nn::gfx::TextureView& textureView = *(static_cast<nn::gfx::TextureView*>(resTexture->GetTextureView()));

        nn::gfx::DescriptorSlot slot;
        pRegisterTextureSlotCallback(&slot, textureView, pUserData);

        resTexture->SetUserDescriptorSlot(slot);
    }

    // エミッタ内テクスチャディスクリプタの割り当て
    for( int i = 0; i < m_EmitterSetCount; i++ )
    {
        for( int j = 0; j < m_EmitterSetResArray[ i ].emitterAllCount; j++ )
        {
            EmitterResource* pEmitterResource = &m_EmitterSetResArray[ i ].pEmitterResource[ j ];
            // テクスチャリソースをguidで検索して設定
            pEmitterResource->m_TextureDescSlot[ TextureSlotId_0 ].Invalidate();
            pEmitterResource->m_TextureDescSlot[ TextureSlotId_1 ].Invalidate();
            pEmitterResource->m_TextureDescSlot[ TextureSlotId_2 ].Invalidate();

            GetTextureDescriptorSlot( &pEmitterResource->m_TextureDescSlot[ TextureSlotId_0 ], pEmitterResource->m_pResEmitter->textureSampler0.guid );
            GetTextureDescriptorSlot( &pEmitterResource->m_TextureDescSlot[ TextureSlotId_1 ], pEmitterResource->m_pResEmitter->textureSampler1.guid );
            GetTextureDescriptorSlot( &pEmitterResource->m_TextureDescSlot[ TextureSlotId_2 ], pEmitterResource->m_pResEmitter->textureSampler2.guid );
        }
    }
}


//------------------------------------------------------------------------------
//  ResNameTable から指定 guid のレコードを検索する
//------------------------------------------------------------------------------
detail::ResNameTableRecord* Resource::SearchRecordFromResNameTable( detail::ResNameTableRecord* tableTop, uint64_t guid ) NN_NOEXCEPT
{
    detail::ResNameTableRecord *currentRecord = tableTop;
    while ( currentRecord != NULL )
    {
        if ( currentRecord->guid == guid )
        {
            return currentRecord;
        }

        currentRecord = currentRecord->nextOffset == 0 ? NULL :
            reinterpret_cast<detail::ResNameTableRecord*>
            ( reinterpret_cast<uint8_t*>( currentRecord ) + currentRecord->nextOffset );
    }

    return NULL;
}

//---------------------------------------------------------------------------
//  外部で初期化された nn::gfx::ResTextureFile をバインドします。
//---------------------------------------------------------------------------
bool Resource::BindExternalResTextureFile( nn::gfx::ResTextureFile* pResTextureFile ) NN_NOEXCEPT
{
    if ( !pResTextureFile || !m_pTextureTable )
    {
        return false;
    }

    detail::ResNameTableRecord *currentRecord = NULL;

    for ( int i = 0; i < m_EmitterSetCount; i++ )
    {
        for ( int j = 0; j < m_EmitterSetResArray[ i ].emitterAllCount; j++ )
        {
            EmitterResource* pEmitterResource = &m_EmitterSetResArray[ i ].pEmitterResource[ j ];

            if ( !pEmitterResource->m_TextureDescSlot[ TextureSlotId_0 ].IsValid() )
            {
                currentRecord = SearchRecordFromResNameTable( m_pTextureTable, pEmitterResource->m_pResEmitter->textureSampler0.guid );
                if ( currentRecord )
                {
                    int index = pResTextureFile->GetTextureDic()->FindIndex( currentRecord->name );
                    nn::gfx::ResTexture* resTexture = pResTextureFile->GetResTexture( index );
                    resTexture->GetUserDescriptorSlot( &pEmitterResource->m_TextureDescSlot[ TextureSlotId_0 ] );
                }
            }

            if ( !pEmitterResource->m_TextureDescSlot[ TextureSlotId_1 ].IsValid() )
            {
                currentRecord = SearchRecordFromResNameTable( m_pTextureTable, pEmitterResource->m_pResEmitter->textureSampler1.guid );
                if ( currentRecord )
                {
                    int index = pResTextureFile->GetTextureDic()->FindIndex( currentRecord->name );
                    nn::gfx::ResTexture* resTexture = pResTextureFile->GetResTexture( index );
                    resTexture->GetUserDescriptorSlot( &pEmitterResource->m_TextureDescSlot[ TextureSlotId_1 ] );
                }
            }

            if ( !pEmitterResource->m_TextureDescSlot[ TextureSlotId_2 ].IsValid() )
            {
                currentRecord = SearchRecordFromResNameTable( m_pTextureTable, pEmitterResource->m_pResEmitter->textureSampler2.guid );
                if ( currentRecord )
                {
                    int index = pResTextureFile->GetTextureDic()->FindIndex( currentRecord->name );
                    nn::gfx::ResTexture* resTexture = pResTextureFile->GetResTexture( index );
                    resTexture->GetUserDescriptorSlot( &pEmitterResource->m_TextureDescSlot[ TextureSlotId_2 ] );
                }
            }
        }
    }

    return true;
}


//------------------------------------------------------------------------------
//  テクスチャービューをディスクリプタプールから解放します
//------------------------------------------------------------------------------
void Resource::UnregisterTextureViewFromDescriptorPool( nn::vfx::UnregisterTextureViewSlot pUnregisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    if (!pUnregisterTextureSlotCallback) return;

    for (int i = 0; i < m_TextureCount; i++)
    {
        nn::gfx::ResTexture* resTexture = m_ResTextureFile->GetResTexture(i);
        nn::gfx::TextureView& textureView = *(static_cast<nn::gfx::TextureView*>(resTexture->GetTextureView()));

        nn::gfx::DescriptorSlot slot;
        resTexture->GetUserDescriptorSlot(&slot);
        pUnregisterTextureSlotCallback(&slot, textureView, pUserData);

        slot.Invalidate();
        resTexture->SetUserDescriptorSlot(slot);
    }
}

//------------------------------------------------------------------------------
//  PrimitiveArray バイナリをトレースする
//------------------------------------------------------------------------------
void Resource::TracePrimitiveArray( detail::BinaryData* pBinaryData ) NN_NOEXCEPT
{
    m_PrimitiveCount = pBinaryData->GetDirectChildCount( VFX_MAKE_TAG( 'P', 'R', 'I', 'M' ) );
    if( m_PrimitiveCount == 0 )
    {
        m_PrimitiveArrayPtr = NULL;
        m_PrimitiveArray    = NULL;
        return;
    }

    size_t primitiveArraySize = sizeof( Primitive ) * m_PrimitiveCount + 32;   // +32は配列でのnewの為
    m_PrimitiveArrayPtr = m_ResHeap.Alloc( primitiveArraySize );
    NN_SDK_ASSERT_NOT_NULL( m_PrimitiveArrayPtr );
    m_PrimitiveArray = new ( m_PrimitiveArrayPtr )Primitive[ m_PrimitiveCount ];

    detail::BinaryData* primDataTop = pBinaryData->GetDirectChildData( VFX_MAKE_TAG( 'P', 'R', 'I', 'M' ) );
    NN_SDK_ASSERT_NOT_NULL( primDataTop );

    int index = 0;
    detail::BinaryData* primData = primDataTop;

    // バッファサイズを計算し、バッファを初期化する
    detail::BufferSizeCalculator vertexBufferSizeCalculator( m_pDevice, nn::gfx::GpuAccess_VertexBuffer );
    while( primData )
    {
        vertexBufferSizeCalculator.AddBufferSize( Primitive::CalculateBufferSize( primData ) );
        primData = primData->GetNextData();
    }

    m_PrimitiveVertexBuffer.Initialize( m_pDevice, &m_ResHeap, nn::gfx::GpuAccess_VertexBuffer, vertexBufferSizeCalculator.GetBufferSize() );

    // 各プリミティブをセットアップする
    primData = primDataTop;

    m_PrimitiveVertexBuffer.Begin();
    while( primData )
    {
        bool result = m_PrimitiveArray[ index ].Initialize( &m_PrimitiveVertexBuffer, primData );

        if( !result )
        {
            detail::OutputWarning( "Faild to initialize primitive: Index %d \n", index );
            primData = primData->GetNextData();
            index++;
            continue;
        }

        m_PrimitiveResourceSize += primData->GetBinarySize();
        primData = primData->GetNextData();
        index++;
    }
    m_PrimitiveVertexBuffer.End();

    NN_SDK_ASSERT( m_PrimitiveCount == index );
}


//------------------------------------------------------------------------------
//  G3d Primitiveバイナリをトレースする
//------------------------------------------------------------------------------
void Resource::TraceG3dPrimitiveArray( detail::BinaryData* pData ) NN_NOEXCEPT
{
    // 現状G3dプリミティブは、ファイル分割にしか対応できていません。
    // 常駐テクスチャのようなG3dプリミティブ部分が不完全なエフェクトバイナリとbfresの組み合わせで運用はできません。

    if ( pData->GetBinarySize() <= 32 ) return;
    if ( !pData->GetBinaryData() ) return;

    NN_SDK_ASSERT_NOT_NULL( m_G3dPrimitiveTable );

    void* ptr = pData->GetBinaryData();
    nn::util::BytePtr binTop( m_pBinaryTop );
    m_G3dPrimitiveResourceSize = pData->GetBinarySize();

    NN_SDK_ASSERT( nn::g3d::ResFile::IsValid( ptr ) );
    m_pG3dResFile = nn::g3d::ResFile::ResCast( ptr );
    NN_SDK_ASSERT_NOT_NULL( m_pG3dResFile );

    if ( m_pMemoryPool == nullptr )
    {
        m_pG3dResFile->Setup( m_pDevice );
    }
    else
    {
        m_pG3dResFile->Setup( m_pDevice, m_pMemoryPool, m_MemoryPoolOffset + binTop.Distance( ptr ), m_MemoryPoolSize );
    }

    BindExternalG3dResFile( m_pG3dResFile );
}

//---------------------------------------------------------------------------
//  外部で初期化された nn::g3d::ResFile をバインドします。
//---------------------------------------------------------------------------
bool Resource::BindExternalG3dResFile( nn::g3d::ResFile* pG3dResFile ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_G3dPrimitiveArrayPtr == nullptr );
    NN_SDK_ASSERT( m_G3dPrimitiveArray == nullptr );
    NN_SDK_ASSERT_NOT_NULL( m_G3dPrimitiveTable );

    m_G3dPrimitiveCount = pG3dResFile->GetModelCount();
    if ( m_G3dPrimitiveCount == 0 )
    {
        m_G3dPrimitiveArrayPtr = NULL;
        m_G3dPrimitiveArray = NULL;
        return false;
    }

    size_t primitiveArraySize = sizeof( G3dPrimitive ) * m_G3dPrimitiveCount + 32;   // +32は配列でのnewの為
    m_G3dPrimitiveArrayPtr = m_ResHeap.Alloc( primitiveArraySize );
    NN_SDK_ASSERT_NOT_NULL( m_G3dPrimitiveArrayPtr );
    m_G3dPrimitiveArray = new ( m_G3dPrimitiveArrayPtr )G3dPrimitive[ m_G3dPrimitiveCount ];

    detail::ResG3dPrimitiveTableRecord* currentRecord = m_G3dPrimitiveTable;
    for ( int i = 0; i < m_G3dPrimitiveCount; i++ )
    {
        NN_SDK_ASSERT_NOT_NULL( currentRecord );
        m_G3dPrimitiveArray[ i ].Initialize( pG3dResFile->GetModel( i ), currentRecord );

        currentRecord = currentRecord->nextOffset == 0 ? NULL :
            reinterpret_cast< detail::ResG3dPrimitiveTableRecord* >
            ( reinterpret_cast< uint8_t* >( currentRecord ) + currentRecord->nextOffset );
    }

    return true;
}

//------------------------------------------------------------------------------
//  ShaderArray バイナリをトレースする
//------------------------------------------------------------------------------
void Resource::TraceShaderBinaryArray( detail::BinaryData* pData ) NN_NOEXCEPT
{
    m_ShaderResourceSize = pData->GetBinarySize();
    void* shaderBinary   = pData->GetBinaryData();

    // プール内オフセットを計算
    nn::util::BytePtr binTop( m_pBinaryTop );

    // シェーダコンテナ初期化
    m_pResShaderFile = nn::gfx::ResShaderFile::ResCast( shaderBinary );
    NN_SDK_ASSERT_NOT_NULL( m_pResShaderFile );
    nn::gfx::ResShaderContainer* pContainer = m_pResShaderFile->GetShaderContainer();
    NN_SDK_ASSERT_NOT_NULL( pContainer );

    if ( m_pMemoryPool == nullptr )
    {
        pContainer->Initialize( m_pDevice );
    }
    else
    {
        pContainer->Initialize( m_pDevice, m_pMemoryPool, m_MemoryPoolOffset + binTop.Distance( pContainer ), m_MemoryPoolSize );
    }
    m_ShaderManager.Initialize( m_pDevice, &m_ResHeap, pContainer, m_ShaderDelaySetup );


    // 子につくコンピュートシェーダを初期化する
    detail::BinaryData* pCSData = pData->GetDirectChildData( VFX_MAKE_TAG( 'G', 'R', 'S', 'C' ) );

    m_ComputeShaderResourceSize = 0;
    if ( pCSData )
    {
        m_ComputeShaderResourceSize = pCSData->GetBinarySize();

        m_pComputeResShaderFile = nn::gfx::ResShaderFile::ResCast( pCSData->GetBinaryData() );
        pContainer = m_pComputeResShaderFile->GetShaderContainer();
        NN_SDK_ASSERT_NOT_NULL( pContainer );

//        NN_SDK_ASSERT( pContainer->GetShaderVariationCount() ==  1 );

        if ( m_pMemoryPool == nullptr )
        {
            pContainer->Initialize( m_pDevice );
        }
        else
        {
            pContainer->Initialize( m_pDevice, m_pMemoryPool, binTop.Distance( pContainer ) + m_MemoryPoolOffset, m_MemoryPoolSize );
        }
        m_ComputeShaderManager.Initialize( m_pDevice, &m_ResHeap, pContainer );
    }
}

//---------------------------------------------------------------------------
//  外部で初期化された nn::gfx::ResSahderFile をバインドします。
//---------------------------------------------------------------------------
bool Resource::BindExternalResShaderFile( nn::gfx::ResShaderFile* resShaderFile, nn::gfx::ResShaderFile* computeResShaderFile ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( resShaderFile );
    NN_SDK_ASSERT_NOT_NULL( computeResShaderFile );

    m_pExtResShaderFile         = resShaderFile;
    m_pExtComputeResShaderFile  = computeResShaderFile;

    // 外部から受け取った ResShaderFile で シェーダマネージャを初期化
    m_ShaderManager.Finalize( m_pDevice, &m_ResHeap );
    m_ComputeShaderManager.Finalize( m_pDevice, &m_ResHeap );

    m_ShaderManager.Initialize( m_pDevice, &m_ResHeap, m_pExtResShaderFile->GetShaderContainer(), m_ShaderDelaySetup );
    m_ComputeShaderManager.Initialize( m_pDevice, &m_ResHeap, m_pExtComputeResShaderFile->GetShaderContainer() );

    return true;
}


//------------------------------------------------------------------------------
//  EmitterSetArray バイナリをトレースする
//------------------------------------------------------------------------------
void Resource::TraceEmitterSetArray( detail::BinaryData* pData, detail::BufferSizeCalculator* bufferSizeCalculator ) NN_NOEXCEPT
{
    m_EmitterSetCount = pData->GetDirectChildCount( VFX_MAKE_TAG( 'E', 'S', 'E', 'T' ) );
    NN_SDK_ASSERT( m_EmitterSetCount != 0 );
    m_EmitterSetResArray = reinterpret_cast< EmitterSetResource* >
        ( m_ResHeap.Alloc( sizeof( EmitterSetResource ) * m_EmitterSetCount ) );
    NN_SDK_ASSERT_NOT_NULL( m_EmitterSetResArray );
    memset( m_EmitterSetResArray, NULL, sizeof( EmitterSetResource ) * m_EmitterSetCount );

    // バインド用
    m_BindEmitterSetResArray = reinterpret_cast< EmitterSetResource** >
        ( m_ResHeap.Alloc( sizeof( EmitterSetResource* ) * m_EmitterSetCount ) );
    NN_SDK_ASSERT_NOT_NULL( m_BindEmitterSetResArray );
    memset( m_BindEmitterSetResArray, NULL, sizeof( EmitterSetResource* ) * m_EmitterSetCount );

    detail::BinaryData* emSetData = pData->GetDirectChildData( VFX_MAKE_TAG( 'E', 'S', 'E', 'T' ) );
    NN_SDK_ASSERT_NOT_NULL( emSetData );
    uint32_t i = 0;

    while( emSetData )
    {
        m_EmitterSetResArray[ i ].pEmitterSetBinary = emSetData;
        InitializeEmitterSetResource( &m_EmitterSetResArray[ i ], bufferSizeCalculator );
        emSetData = emSetData->GetNextData();
        i++;
    }
}

//------------------------------------------------------------------------------
//  EmitterSetResource を初期化する
//------------------------------------------------------------------------------
void Resource::InitializeEmitterSetResource( EmitterSetResource* pEmitterSetResource, detail::BufferSizeCalculator* bufferSizeCalculator ) NN_NOEXCEPT
{
    pEmitterSetResource->isVisible = true;

    // エンディアンフリップしつつ構造体取得
    pEmitterSetResource->pResEmitterSet = pEmitterSetResource->pEmitterSetBinary->GetBinaryDataWithFlip< detail::ResEmitterSet >();

    // 総エミッタ数をインクリメント
    m_EmitterCount += pEmitterSetResource->pResEmitterSet->emitterAllNum;

    // 子を含まない総エミッタ数
    pEmitterSetResource->emitterCount = pEmitterSetResource->pEmitterSetBinary->GetDirectChildCount( VFX_MAKE_TAG( 'E', 'M', 'T', 'R' ) );

    // 子を含む総エミッタ数
    pEmitterSetResource->emitterAllCount = pEmitterSetResource->pResEmitterSet->emitterAllNum;

    // エミッタリソース管理配列
    pEmitterSetResource->pEmitterResourcePtr = m_ResHeap.Alloc( sizeof( EmitterResource ) * ( pEmitterSetResource->emitterAllCount ) + 32 );
    NN_SDK_ASSERT_NOT_NULL( pEmitterSetResource->pEmitterResourcePtr );
    pEmitterSetResource->pEmitterResource = new ( pEmitterSetResource->pEmitterResourcePtr )EmitterResource[ pEmitterSetResource->emitterAllCount ];

    // エミッタリソースの先頭
    detail::BinaryData* emData = pEmitterSetResource->pEmitterSetBinary->GetDirectChildData( VFX_MAKE_TAG( 'E', 'M', 'T', 'R' ) );
    NN_SDK_ASSERT_NOT_NULL( emData );

    EmitterResource* curEmitterResource     = NULL;
    EmitterResource* prevEmitterResource    = NULL;
    int emitterIndex                        = 0;
    int parentEmitterIndex                  = 0;

    pEmitterSetResource->isLifeInfinity = false;
    pEmitterSetResource->isLoopEffect = false;

    while( emData )
    {
        curEmitterResource = &pEmitterSetResource->pEmitterResource[ emitterIndex ];
        NN_SDK_ASSERT( emitterIndex < pEmitterSetResource->emitterAllCount );

        // リストをつなぐ
        curEmitterResource->m_NextEmitterResource = NULL;
        if( prevEmitterResource )
        {
            prevEmitterResource->m_NextEmitterResource = curEmitterResource;
        }

        // エミッタリソースの初期化
        InitializeEmitterResource( curEmitterResource, emData, parentEmitterIndex, bufferSizeCalculator );
        emitterIndex++;
        parentEmitterIndex++;
        m_EmitterResourceSize += emData->GetBinarySize();

        // エミッタセットが ループ or 無限寿命 エフェクト判定
        if( curEmitterResource->m_pResEmitter->ptcl.isLifeInfinity )
        {
            pEmitterSetResource->isLifeInfinity = true;
        }
        if( !curEmitterResource->m_pResEmitter->emission.isOneTime )
        {
            pEmitterSetResource->isLoopEffect = true;
        }

        // 子エミッタへの対応。
        memset( curEmitterResource->m_ChildEmitterResSet, 0, sizeof( EmitterResource* ) * SystemParameters_MaxEmitterInclusionCount );
        curEmitterResource->m_ChildEmitterResCount = 0;

        int childEmitterIndex = 0;

        if( emData->GetChildData() )
        {
            EmitterResource* childEmitterResource = NULL;
            //NN_SDK_ASSERT( i < emSetRes->emitterNum );
            // 各エミッタリソースごとの初期化
            detail::BinaryData* emChildData = emData->GetChildData();

            while( emChildData )
            {
                childEmitterResource = &pEmitterSetResource->pEmitterResource[ emitterIndex ];

                // エミッタリソースの初期化
                InitializeEmitterResource( childEmitterResource, emChildData, childEmitterIndex, bufferSizeCalculator );
                childEmitterResource->m_IsChildEmitter = true;
                emitterIndex++;

                // ループ・無限寿命判定
                if( childEmitterResource->m_pResEmitter->ptcl.isLifeInfinity )
                {
                    pEmitterSetResource->isLifeInfinity = true;
                }

                // チャイルドは親の寿命で放出を終了するためワンタイム扱い。
                curEmitterResource->m_ChildEmitterResSet[ childEmitterIndex ] = childEmitterResource;
                curEmitterResource->m_ChildEmitterResSet[ childEmitterIndex ]->m_NextEmitterResource = NULL;

                memset( curEmitterResource->m_ChildEmitterResSet[ childEmitterIndex ]->m_ChildEmitterResSet, 0, sizeof( EmitterResource* ) * SystemParameters_MaxEmitterInclusionCount );

                childEmitterIndex++;
                emChildData = emChildData->GetNextData();
            }

            curEmitterResource->m_ChildEmitterResCount = childEmitterIndex;
        }

        prevEmitterResource = curEmitterResource;

        emData = emData->GetNextData( VFX_MAKE_TAG( 'E', 'M', 'T', 'R' ) );
    }
}// NOLINT(readability/fn_size)


//------------------------------------------------------------------------------
//  EmitterResource を初期化する
//------------------------------------------------------------------------------
void Resource::InitializeEmitterResource( EmitterResource* pEmitterResource, detail::BinaryData* pResEmitter, int index, detail::BufferSizeCalculator* bufferSizeCalculator ) NN_NOEXCEPT
{
    pEmitterResource->m_EmitterIndex = index;

    // ResEmitter アドレス解決
    pEmitterResource->m_pResEmitter = pResEmitter->GetBinaryDataWithFlip< detail::ResEmitter >();

    // エミッタ静的UBOアドレス
    pEmitterResource->m_ResEmitterStaticConstantBuffer = &pEmitterResource->m_pResEmitter->staticUbo;

    // エミッタデータのアドレス解決
    pEmitterResource->ResolveBinaryData( pResEmitter );

    // エミッタ静的UBO用のバッファサイズを追加
    size_t emitterStaticBufferSize = sizeof(detail::ResCommon) + sizeof(detail::EmitterStaticUniformBlock);
    bufferSizeCalculator->AddBufferSize( emitterStaticBufferSize );

    // カスタムシェーダ用バッファサイズを追加
    if ( pEmitterResource->m_pCustomShaderParam )
    {
        bufferSizeCalculator->AddBufferSize( pEmitterResource->m_CustomShaderParamSize );
    }

    // エミッタプラグイン用バッファサイズを追加
    if ( pEmitterResource->m_pEmitterPluginData )
    {
        bufferSizeCalculator->AddBufferSize( sizeof( nn::util::Float4 ) * 8 );
    }

    // フィールドの定数バッファ用ワークを確保
    if( pEmitterResource->m_IsUseField )
    {
        bufferSizeCalculator->AddBufferSize( sizeof( detail::EmitterFieldConstantBuffer ) );
    }

}// NOLINT(readability/fn_size)


//------------------------------------------------------------------------------
//  EmitterResource のグラフィックスリソースの初期化処理をする
//------------------------------------------------------------------------------
void Resource::InitializeEmitterGraphicsResource( EmitterResource* pEmitterResource )
{
    // レンダーステートを初期化
    pEmitterResource->InitializeRenderState( m_pDevice );

    // テクスチャリソースをguidで検索して設定
    pEmitterResource->m_TextureDescSlot[ TextureSlotId_0 ].Invalidate();
    pEmitterResource->m_TextureDescSlot[ TextureSlotId_1 ].Invalidate();
    pEmitterResource->m_TextureDescSlot[ TextureSlotId_2 ].Invalidate();

    GetTextureDescriptorSlot( &pEmitterResource->m_TextureDescSlot[ TextureSlotId_0 ], pEmitterResource->m_pResEmitter->textureSampler0.guid );
    GetTextureDescriptorSlot( &pEmitterResource->m_TextureDescSlot[ TextureSlotId_1 ], pEmitterResource->m_pResEmitter->textureSampler1.guid );
    GetTextureDescriptorSlot( &pEmitterResource->m_TextureDescSlot[ TextureSlotId_2 ], pEmitterResource->m_pResEmitter->textureSampler2.guid );

    // エミッタ形状プリミティブリソースをindexで検索して設定
    if ( pEmitterResource->m_pResEmitter->volume.primitiveIndex != static_cast< uint64_t >( InvalidValueId_PrimitiveIndex ) )
    {
        // ボリュームは、vfxプリミティブから情報を取得する
        pEmitterResource->m_pVolumePrimitive = GetPrimitive( pEmitterResource->m_pResEmitter->volume.primitiveIndex );
    }
    else
    {
        pEmitterResource->m_pVolumePrimitive = NULL;
    }

    // トリミングは、vfxプリミティブ( ptclバイナリに含まれる )から情報を取得
    if ( pEmitterResource->m_pResEmitter->ptcl.isTriming &&
         pEmitterResource->m_pResEmitter->ptcl.trimmingPrimitiveIndex != static_cast< uint64_t >( InvalidValueId_PrimitiveIndex ) )
    {
        pEmitterResource->m_pPrimitive = GetPrimitive( pEmitterResource->m_pResEmitter->ptcl.trimmingPrimitiveIndex );
    }

    // エミッタリソースが保持するコンスタントバッファを割り当て
    pEmitterResource->InitializeConstantBuffer( &m_ResourceConstantBuffer );

    // エミッタリソースアップデート
    pEmitterResource->UpdateParams( this );

    // エミッタリソースの最終セットアップを行います。
    // この時点で、シェーダ、プリミティブのセットアップが完了していない場合は、
    // エミッタセット生成時に再度セットアップを試みます。
    if ( !m_ShaderDelaySetup )
    {
        pEmitterResource->Setup( this );
    }
}

//------------------------------------------------------------------------------
//  EmitterResource の終了処理をする
//------------------------------------------------------------------------------
void Resource::FinalizeEmitterResource( EmitterResource* pEmitterResource ) NN_NOEXCEPT
{
    pEmitterResource->FinalizeRenderState( m_pDevice );
    pEmitterResource->FinalizeVertexState( m_pDevice );
}

//------------------------------------------------------------------------------
//  ResTexture を 検索、取得します。
//------------------------------------------------------------------------------
const nn::gfx::ResTexture* Resource::SearchResTexture( uint64_t guid ) const NN_NOEXCEPT
{
    if ( guid == static_cast< uint64_t >( InvalidValueId_AssetGlobalId ) )
    {
        return nullptr;
    }

    if ( m_ResTextureFile && m_pTextureTable )
    {
        detail::ResNameTableRecord *currentRecord = m_pTextureTable;
        while ( currentRecord != NULL )
        {
            if ( currentRecord->guid == guid )
            {
                int index = m_ResTextureFile->GetTextureDic()->FindIndex( currentRecord->name );
                if ( index == nn::util::ResDic::Npos )
                {
                    break;
                }

                return m_ResTextureFile->GetResTexture( index );
            }

            currentRecord = currentRecord->nextOffset == 0 ? NULL :
                reinterpret_cast< detail::ResNameTableRecord* >
                ( reinterpret_cast< uint8_t* >( currentRecord ) + currentRecord->nextOffset );
        }
    }

    return nullptr;
}


//------------------------------------------------------------------------------
//  ディスクリプタスロットを取得する
//------------------------------------------------------------------------------
bool Resource::GetTextureDescriptorSlot( nn::gfx::DescriptorSlot* pDescriptorSlot, uint64_t guid ) const NN_NOEXCEPT
{
    if ( guid == static_cast<uint64_t>( InvalidValueId_AssetGlobalId ) ) return false;

    const char* textureName = nullptr;

    if ( m_ResTextureFile && m_pTextureTable )
    {
        detail::ResNameTableRecord *currentRecord = m_pTextureTable;
        while ( currentRecord != NULL )
        {
            if ( currentRecord->guid == guid )
            {
                textureName = currentRecord->name;

                int index = m_ResTextureFile->GetTextureDic()->FindIndex( currentRecord->name );
                if ( index == nn::util::ResDic::Npos )
                {
                    break;
                }

                nn::gfx::ResTexture* resTexture = m_ResTextureFile->GetResTexture( index );
                resTexture->GetUserDescriptorSlot( pDescriptorSlot );
                return true;
            }

            currentRecord = currentRecord->nextOffset == 0 ? NULL :
                reinterpret_cast<detail::ResNameTableRecord*>
                (reinterpret_cast<uint8_t*>(currentRecord) + currentRecord->nextOffset);
        }
    }

    // 自身のリソース内からテクスチャを引けなかった場合は、共通リソースから検索する
    if ( m_pResidentResource )
    {
        if ( m_pResidentResource->GetTextureDescriptorSlot( pDescriptorSlot, guid ) )
        {
            return true;
        }
        else
        {
            detail::OutputLog( "Appointed Texture is not found from Resident Resource.\n" );
            if ( textureName )
            {
                detail::OutputLog( "The Texture which was not found is %s.\n", textureName );
            }
        }
    }

    return false;
}

//------------------------------------------------------------------------------
//  プリミティブを検索する
//------------------------------------------------------------------------------
Primitive* Resource::GetPrimitive( uint64_t guid ) const NN_NOEXCEPT
{
    if ( guid == static_cast<uint64_t>( InvalidValueId_AssetGlobalId ) ) return NULL;

    for( int i = 0; i < m_PrimitiveCount; i++ )
    {
        if( m_PrimitiveArray[ i ].GetGlobalId() == guid )
        {
            return &m_PrimitiveArray[ i ];
        }
    }

    // 自身のリソース内からプリミティブを引けなかった場合は、共通リソースから検索する
    if ( m_pResidentResource )
    {
        Primitive* retPrimitive = m_pResidentResource->GetPrimitive( guid );
        if ( retPrimitive )
        {
            return retPrimitive;
        }
        else
        {
            detail::OutputLog( "Appointed Primitive is not found from Resident Resource.\n" );
        }
    }

    return NULL;
}

//------------------------------------------------------------------------------
//  プリミティブを検索する
//------------------------------------------------------------------------------
IPrimitive* Resource::GetG3dPrimitive( uint64_t guid ) const NN_NOEXCEPT
{
    int index = 0;

    if ( m_G3dPrimitiveArray && m_G3dPrimitiveTable )
    {
        detail::ResG3dPrimitiveTableRecord *currentRecord = m_G3dPrimitiveTable;
        while ( currentRecord != NULL )
        {
            if ( currentRecord->guid == guid )
            {
                return &m_G3dPrimitiveArray[ index ];
            }

            index++;

            currentRecord = currentRecord->nextOffset == 0 ? NULL :
                reinterpret_cast<detail::ResG3dPrimitiveTableRecord*>
                ( reinterpret_cast<uint8_t*>( currentRecord ) + currentRecord->nextOffset );
        }
    }

    // 自身のリソース内から引けなかった場合は、共通リソースから検索する
    if ( m_pResidentResource )
    {
        return m_pResidentResource->GetG3dPrimitive( guid );
    }

    return nullptr;
}

//---------------------------------------------------------------------------
//  エミッタセットIDの検索を行います。
//---------------------------------------------------------------------------
int Resource::SearchEmitterSetId( const char* name ) const NN_NOEXCEPT
{
    for( int i = 0; i < m_EmitterSetCount; i++ )
    {
        if( strcmp( name, m_EmitterSetResArray[ i ].pResEmitterSet->name ) == 0 )
        {
            return i;
        }
    }

    return InvalidValueId_EmitterSetId;
}

//---------------------------------------------------------------------------
//  エミッタセットIDの検索を行います。
//---------------------------------------------------------------------------
int Resource::SearchEmitterSetIdWithFilePath( const char* emitterSetFilePath ) const NN_NOEXCEPT
{
    for (int i = 0; i < m_EmitterSetCount; i++)
    {
        if ( m_EmitterSetResArray[i].pFilePath == nullptr )
        {
            continue;
        }

        if (strcmp( emitterSetFilePath, m_EmitterSetResArray[ i ].pFilePath ) == 0)
        {
            return i;
        }
    }

    return InvalidValueId_EmitterSetId;
}

//---------------------------------------------------------------------------
//  sourceResで指定されるリソースに繋ぎ換えます。
//---------------------------------------------------------------------------
bool Resource::BindResource( int targetSetId, EmitterSetResource* pEmitterSetResource ) NN_NOEXCEPT
{
    if( m_EmitterSetCount <= targetSetId )
    {
        return false;
    }

    m_BindEmitterSetResArray[ targetSetId ] = pEmitterSetResource;

    // 該当エミッタセットを再生成する
    m_pSystem->ReCreateEmitterSet( m_ResourceId, targetSetId );

    return true;
}

//---------------------------------------------------------------------------
//  targetSetIdで指定されるエミッタセットリソースの繋ぎ換えを解除します。
//---------------------------------------------------------------------------
bool Resource::UnbindResource( int targetSetId ) NN_NOEXCEPT
{
    if( m_EmitterSetCount <= targetSetId )
    {
        return false;
    }

    m_BindEmitterSetResArray[ targetSetId ] = NULL;

    // 該当エミッタセットを再生成する
    m_pSystem->ReCreateEmitterSet( m_ResourceId, targetSetId );

    return true;
}

//---------------------------------------------------------------------------
//  チャイルドエミッタが存在するかどうか。
//---------------------------------------------------------------------------
bool Resource::IsExistChildEmitter( int targetSetId ) const NN_NOEXCEPT
{
    EmitterSetResource* pEmitterSetResource = GetEmitterSetResource( targetSetId );
    if( !pEmitterSetResource )
    {
        return false;
    }

    bool ret = false;
    for( int i = 0; i < pEmitterSetResource->emitterCount; i++ )
    {
        if( pEmitterSetResource->pEmitterResource[ i ].m_ChildEmitterResSet[ 0 ] )
        {
            ret = true;
        }
    }
    return ret;
}

//---------------------------------------------------------------------------
//  指定EmitterSetが無限寿命、または、ループエフェクトかを取得する。
//---------------------------------------------------------------------------
bool Resource::IsNeedFade( int emitterSetId ) const NN_NOEXCEPT
{
    EmitterSetResource* pEmitterSetResource = GetEmitterSetResource( emitterSetId );
    if( !pEmitterSetResource )
    {
        return false;
    }

    return ( pEmitterSetResource->isLoopEffect || pEmitterSetResource->isLifeInfinity );
}

//---------------------------------------------------------------------------
//  リソースサイズの詳細をログ出力します。
//---------------------------------------------------------------------------
void Resource::OutputResourceInfo() const NN_NOEXCEPT
{
    detail::OutputLog( "-------[ResourceInfo]-------\n" );
    detail::OutputLog( "Resource       ID  :%8d\n", m_ResourceId );
    detail::OutputLog( "EmitterSet[0]  Name:%s\n", m_EmitterSetResArray[ 0 ].pResEmitterSet->name );
    detail::OutputLog( "---------[Resource]---------\n" );
    detail::OutputLog( "EmitterSet     Num :%8d\n", m_EmitterSetCount );
    detail::OutputLog( "Emitter        Num :%8d\n", m_EmitterCount );
    detail::OutputLog( "               Size:%8d\n", m_EmitterResourceSize );
    detail::OutputLog( "Shader         Num :%8d\n", m_ShaderManager.GetShaderCount() );
    detail::OutputLog( "            BinSize:%8d\n", m_ShaderResourceSize );
    detail::OutputLog( "ComputeShader  Num :%8d\n", m_ComputeShaderManager.GetShaderCount() );
    detail::OutputLog( "            BinSize:%8d\n", m_ComputeShaderResourceSize );
    detail::OutputLog( "Texture        Num :%8d\n", m_TextureCount );
    detail::OutputLog( "               Size:%8d\n", m_TextureResourceSize );
    detail::OutputLog( "Primitive      Num :%8d\n", m_PrimitiveCount );
    detail::OutputLog( "               Size:%8d\n", m_PrimitiveResourceSize );
    detail::OutputLog( "G3dPrimitive   Num :%8d\n", m_G3dPrimitiveCount );
    detail::OutputLog( "               Size:%8d\n", m_G3dPrimitiveResourceSize );
    detail::OutputLog( "-----------[Heap]-----------\n" );
    detail::OutputLog( "Allocate       Size:%8d\n", m_ResHeap.GetAllocatedSize() );
    detail::OutputLog( "Allocate      Count:%8d\n", m_ResHeap.GetAllocatedCount() );
    detail::OutputLog( "-Texture       Size:%8d\n", m_TextureAllocatedSize );
    detail::OutputLog( "-Primitive     Size:%8d\n", m_PrimitiveAllocatedSize );
    detail::OutputLog( "-G3dPrimitive  Size:%8d\n", m_G3dPrimitiveAllocatedSize );
    detail::OutputLog( "-Shader        Size:%8d\n", m_ShaderAllocatedSize );
    detail::OutputLog( "-ConstBuffer   Size:%8d\n", m_ConstantBufferAllocatedSize );
    detail::OutputLog( "-EmitterSet    Size:%8d\n", m_EmitterSetAllocatedSize );
    detail::OutputLog( "-Emitter       Size:%8d\n", m_EmitterAllocatedSize );
    detail::OutputLog( "----------------------------\n" );
}

} // namespace vfx
} // namespace nn

