﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_Misc.h>
#include <nn/vfx/vfx_MemUtil.h>
#include <nn/vfx/vfx_Random.h>

namespace nn {
namespace vfx {
namespace detail {

//------------------------------------------------------------------------------
//  Static メンバ
//------------------------------------------------------------------------------
nn::util::Vector3fType* Random::g_Vec3Table             = NULL;
nn::util::Vector3fType* Random::g_NormalizedVec3Table   = NULL;
bool                    Random::g_UseGlobalSeed         = false;
uint32_t                Random::g_GlobalSeed            = 256;
RandomGenerator         Random::g_Random;

//------------------------------------------------------------------------------
//  グローバルランダム取得関数
//------------------------------------------------------------------------------
RandomGenerator* Random::GetGlobalRandom() NN_NOEXCEPT
{
    return &Random::g_Random;
}

//------------------------------------------------------------------------------
//  コンストラクタ
//------------------------------------------------------------------------------
Random::Random() NN_NOEXCEPT
{
    m_Vec3RandomIndex = static_cast< uint16_t >( g_Random.GetUnsignedInteger() );
    m_NormalizedVec3RandomIndex = static_cast< uint16_t >( g_Random.GetUnsignedInteger() );
    m_RandomValue = g_Random.GetUnsignedInteger();
}

//------------------------------------------------------------------------------
//  乱数テーブル作成
//------------------------------------------------------------------------------
void Random::Initialize() NN_NOEXCEPT
{
    RandomGenerator rnd( 12345679 );

    g_Vec3Table = static_cast< nn::util::Vector3fType* >( detail::AllocFromStaticHeap( sizeof( nn::util::Vector3fType ) * RandomTableSettings_RandomTableSize ) );
    g_NormalizedVec3Table = static_cast< nn::util::Vector3fType* >( detail::AllocFromStaticHeap( sizeof( nn::util::Vector3fType ) * RandomTableSettings_RandomTableSize ) );
    MemUtil::FillZero( g_Vec3Table, sizeof( nn::util::Vector3fType ) * RandomTableSettings_RandomTableSize );
    MemUtil::FillZero( g_NormalizedVec3Table, sizeof( nn::util::Vector3fType ) * RandomTableSettings_RandomTableSize );

    for( int i = 0; i < RandomTableSettings_RandomTableSize; i++ )
    {
        nn::util::VectorSetX( &g_Vec3Table[ i ], rnd.GetFloat() * 2.0f - 1.0f );
        nn::util::VectorSetY( &g_Vec3Table[ i ], rnd.GetFloat() * 2.0f - 1.0f );
        nn::util::VectorSetZ( &g_Vec3Table[ i ], rnd.GetFloat() * 2.0f - 1.0f );

        nn::util::VectorSetX( &g_NormalizedVec3Table[ i ], rnd.GetFloat() * 2.0f - 1.0f );
        nn::util::VectorSetY( &g_NormalizedVec3Table[ i ], rnd.GetFloat() * 2.0f - 1.0f );
        nn::util::VectorSetZ( &g_NormalizedVec3Table[ i ], rnd.GetFloat() * 2.0f - 1.0f );
        nn::util::VectorNormalize( &g_NormalizedVec3Table[ i ], g_NormalizedVec3Table[ i ] );
    }
}

//------------------------------------------------------------------------------
//  乱数テーブル破棄
//------------------------------------------------------------------------------
void Random::Finalize() NN_NOEXCEPT
{
    detail::FreeFromStaticHeap( g_Vec3Table );
    detail::FreeFromStaticHeap( g_NormalizedVec3Table );

    g_Vec3Table = NULL;
    g_NormalizedVec3Table = NULL;
}

//---------------------------------------------------------------------------
//  シード設定。
//---------------------------------------------------------------------------
void Random::SetSeed( uint32_t val ) NN_NOEXCEPT
{
    if( !g_UseGlobalSeed )
    {
        m_Vec3RandomIndex = static_cast< uint16_t >( val >> 0 );
        m_NormalizedVec3RandomIndex = static_cast< uint16_t >( val >> 16 );
        m_RandomValue = val;
    }
    else
    {
        m_Vec3RandomIndex = static_cast< uint16_t >( g_GlobalSeed >> 0 );
        m_NormalizedVec3RandomIndex = static_cast< uint16_t >( g_GlobalSeed >> 16 );
        m_RandomValue = g_GlobalSeed;
    }
}


} // namespace detail
} // namespace vfx
} // namespace nn
