﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/vfx/vfx_ShaderManager.h>
#include <nn/vfx/vfx_Misc.h>

namespace nn {
namespace vfx {
namespace detail {

//---------------------------------------------------------------------------
//  エミッタ時間 キーフレームアニメーションを計算します。
//---------------------------------------------------------------------------
void CalculateEmitterKeyFrameAnimation( nn::util::Float3* pOutValue, bool* pOutIsAnimeOver, const ResAnimEmitterKeyParamSet* pAnimationParam, float time ) NN_NOEXCEPT
{
    // エミッタ時間アニメーション用キーフレーム処理。
    // 寿命ではなくフレーム数依存であることに注意！
    int term = 0;
    float dis = 0;
    const int lastKeyIndex = pAnimationParam->keyNum - 1;
    const int keyCount = pAnimationParam->keyNum;

    const float maxKeyFrameCount = pAnimationParam->keyValue[ lastKeyIndex ].w;

    // キー無し（デフォルト値を用いるので処理を返す）
    if( pAnimationParam->keyNum == 0 )
    {
        return;
    }

    // キーが一つのみ（固定値なので値をそのまま返す）
    if( pAnimationParam->keyNum == 1 )
    {
        Float3Copy( pOutValue, pAnimationParam->keyValue[ 0 ] );
        return;
    }

    // ここからアニメーション計算

    //[ 0, 最終フレーム数 ]の間の時間（ループ加味）
    float localTime = time;
    if( pAnimationParam->loop )
    {
        localTime = fmod( localTime, maxKeyFrameCount );
    }

    // まだ最初のキーの時間に到達していない（最初のキーのカラーでClampする）
    if( localTime < pAnimationParam->keyValue[ 0 ].w )
    {
        Float3Copy( pOutValue, pAnimationParam->keyValue[ 0 ] );
        return;
    }
    // 既に最終キーの時間を過ぎている（最終キーのカラーでClampする）
    if( pAnimationParam->keyValue[ lastKeyIndex ].w <= localTime )
    {
        Float3Copy( pOutValue, pAnimationParam->keyValue[ lastKeyIndex ] );
        *pOutIsAnimeOver = true;
        return;
    }

    // TODO：平たく書いた方がよいかも。
    nn::util::Vector3fType start;
    nn::util::Vector3fType end;
    for( term = 0; term < keyCount; term++ )
    {
        int index1 = term;
        int index2 = term + 1;
        float t1 = pAnimationParam->keyValue[ index1 ].w;
        float t2 = pAnimationParam->keyValue[ index2 ].w;

        if( ( t1 <= localTime ) && ( localTime < t2 ) )
        {
            nn::util::VectorLoad( &start, pAnimationParam->keyValue[ index1 ].v );
            nn::util::VectorLoad( &end, pAnimationParam->keyValue[ index2 ].v );
            dis = t2 - t1;

            //---------------------------------------------------------------------------
            // ( start + ( end - start ) / dis * ( localTime - t1 ) );
            //---------------------------------------------------------------------------
            nn::util::Vector3fType ret;
            nn::util::VectorSubtract( &ret, end, start );
            nn::util::VectorMultiply( &ret, ret, ( localTime - t1 ) / dis );
            nn::util::VectorAdd( &ret, start, ret );

            nn::util::VectorStore( pOutValue, ret );
            return;
        }
    }

    // ここには来ないはず
    return;
}

} // namespace detail
} // namespace vfx
} // namespace nn
