﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


//------------------------------------------------------------------------------
#include <nn/vfx/vfx_EmitterCalc.h>
#include <nn/vfx/vfx_System.h>
#include <nn/vfx/vfx_KeyFrameAnim.h>

namespace nn {
namespace vfx {
namespace detail {

//---------------------------------------------------------------------------
//  パーティクル情報を計算する
//---------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleInfo( Particle* pDstParticle, Emitter* pEmitter, float time, float life, int index ) NN_NOEXCEPT
{
    NN_UNUSED( pDstParticle );
    NN_UNUSED( pEmitter );
    NN_UNUSED( time );
    NN_UNUSED( life );
    NN_UNUSED( index );
}

//---------------------------------------------------------------------------
//  親パーティクル情報を計算しチャイルドパーティクルへ継承する
//---------------------------------------------------------------------------
void EmitterCalculator::InheritParentParticleInfo( Emitter* const pEmitter, int writeIndex ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pEmitter );
    ParticleProperty* pParticleProperty = pEmitter->m_ParticlePropertyFront;
    if( pEmitter->m_pEmitterData->emitter.calcType == EmitterCalculationMode_Cpu )
    {
        pParticleProperty = &pEmitter->m_CpuParticleProperty;
    }

    // 親エミッタに関する情報
    // MEMO: 現状親エミッタは必ず CPU エミッタなので、m_CpuParticleProperty を取ってOK
    const Emitter*                  pParent             = pEmitter->m_pParentEmitter;
    const int                       parentIndex         = pEmitter->m_ChildEmitterTempParentIndex;
    const detail::ParticleData*     pParentParticleData = &pParent->m_pParticleData[ parentIndex ];
    const detail::ParticleProperty* pParentParticleProp = &pParent->m_CpuParticleProperty;

    // 親パーティクルの状態を計算し継承する
    Particle parentParticle;
    nn::util::Vector3fType parentWorldVec;
    nn::util::Vector4fType parentScale;
    nn::util::Vector4fType parentRotate;
    nn::util::Vector4fType parentRandom;
    float                  parentTime;
    float                  parentLife;
    if( !pEmitter->m_pEmitterData->inherit.enableEmitterParticle )
    {
        // 軽量版チャイルド
        parentWorldVec = *pEmitter->m_pChildEmitterTempWorldVec;
        nn::util::VectorLoad( &parentScale,  pParentParticleProp->scale[ parentIndex ] );
        nn::util::VectorLoad( &parentRotate, pParentParticleProp->initRotate[ parentIndex ] );
        nn::util::VectorLoad( &parentRandom, pParentParticleProp->random[ parentIndex ] );

        // MEMO: 計算順序の関係上、通常チャイルドとそろえるため一つ戻す。
        parentTime = pParentParticleData->GetTime( pParent->GetTime() ) - pParent->m_FrameRate;
        parentLife = pParentParticleData->GetLife();
    }
    else
    {
        // 通常チャイルド
        parentWorldVec = pEmitter->m_ParentParticleWorldVec;
        parentScale = pEmitter->m_ParentParticleScale;
        parentRotate = pEmitter->m_ParentParticleRotate;
        parentRandom = pEmitter->m_ParentParticleRandom;
        parentTime = pEmitter->m_ParentParticleTime;
        parentLife = pEmitter->m_ParentParticleLife;
    }

    // 軽量版チャイルドの場合、ここで位置を継承
    if( !pEmitter->m_pEmitterData->inherit.enableEmitterParticle )
    {
        nn::util::Vector3fType pos;
        nn::util::VectorLoad( &pos, pParticleProperty->localPos[ writeIndex ].v );
        nn::util::Vector3fType temp = *pEmitter->m_pChildEmitterTempWorldPos;
        nn::util::VectorAdd( &pos, pos, temp );
        nn::util::VectorStore( pParticleProperty->localPos[ writeIndex ].v, pos );
    }

    // 速度継承は、親パーティクルから設定されたベクトルを利用する
    if( pEmitter->m_pEmitterData->inherit.velocity )
    {
        float inheritRate = pEmitter->m_pEmitterData->inherit.velocityRate;
        nn::util::Vector3fType temp;
        nn::util::Vector3fType localVec = NN_UTIL_VECTOR_3F_INITIALIZER(
            pParticleProperty->localVec[ writeIndex ].x,
            pParticleProperty->localVec[ writeIndex ].y,
            pParticleProperty->localVec[ writeIndex ].z );

        nn::util::VectorMultiply( &temp, parentWorldVec, inheritRate );
        nn::util::VectorAdd( &temp, temp, localVec );
        nn::util::VectorStore( pParticleProperty->localVec[ writeIndex ].v, temp );
    }

    // スケールを継承
    if( pEmitter->m_pEmitterData->inherit.scale )
    {
        CalculateParticleScaleVecFromTime( &parentParticle.localScale,
            pEmitter->m_pParentEmitter->m_pEmitterRes,
            parentScale,
            parentRandom,
            parentLife,
            parentTime );

        const float inheritRate = pEmitter->m_pEmitterData->inherit.scaleRate;
        nn::util::VectorMultiply( &parentParticle.localScale, parentParticle.localScale, inheritRate );
        pParticleProperty->scale[writeIndex].x = nn::util::VectorGetX( parentParticle.localScale );
        pParticleProperty->scale[writeIndex].y = nn::util::VectorGetY( parentParticle.localScale );
        pParticleProperty->scale[writeIndex].z = nn::util::VectorGetZ( parentParticle.localScale );
        // .w には、パーティクル生成タイムが入力されるため、コピーしない
    }

    // 回転値を継承
    if( pEmitter->m_pEmitterData->inherit.rotate )
    {
        CalculateRotationMatrix( &parentParticle.localRotate,
            pEmitter->m_pParentEmitter->m_pEmitterRes,
            parentRotate,
            parentRandom,
            parentTime );

        pParticleProperty->initRotate[ writeIndex ].x = nn::util::VectorGetX( parentParticle.localRotate );
        pParticleProperty->initRotate[ writeIndex ].y = nn::util::VectorGetY( parentParticle.localRotate );
        pParticleProperty->initRotate[ writeIndex ].z = nn::util::VectorGetZ( parentParticle.localRotate );
    }

    // カラー0 / アルファ0 継承
    if ( pParticleProperty->initColor0 )
    {
        // 継承する親カラーを計算する
        if( pEmitter->m_pEmitterData->inherit.color0 || pEmitter->m_pEmitterData->inherit.alpha0 )
        {
            nn::util::Vector3fType color0Anim;
            nn::util::VectorLoad( &color0Anim, pEmitter->m_pParentEmitter->m_EmitterAnimation.color0 );
            CalculateParticleColor0VecFromTime( &parentParticle.localColor0,
                pEmitter->m_pParentEmitter->m_pEmitterRes,
                parentRandom,
                pEmitter->m_pParentEmitter->m_Color0,
                color0Anim,
                pEmitter->m_pParentEmitter->m_EmitterAnimation.alpha0.x,
                parentLife,
                parentTime );

            if( pEmitter->m_pEmitterData->inherit.color0 )
            {
                pParticleProperty->initColor0[ writeIndex ].x = nn::util::VectorGetX( parentParticle.localColor0 );
                pParticleProperty->initColor0[ writeIndex ].y = nn::util::VectorGetY( parentParticle.localColor0 );
                pParticleProperty->initColor0[ writeIndex ].z = nn::util::VectorGetZ( parentParticle.localColor0 );
            }

            if( pEmitter->m_pEmitterData->inherit.alpha0 )
            {
                pParticleProperty->initColor0[ writeIndex ].w = nn::util::VectorGetW( parentParticle.localColor0 );
            }
        }
    }

    // カラー1 / アルファ1 継承
    if ( pParticleProperty->initColor1 )
    {
        // 継承する親カラーを計算する
        if( pEmitter->m_pEmitterData->inherit.color1 || pEmitter->m_pEmitterData->inherit.alpha1 )
        {
            nn::util::Vector3fType color1Anim;
            nn::util::VectorLoad( &color1Anim, pEmitter->m_pParentEmitter->m_EmitterAnimation.color0 );
            CalculateParticleColor1VecFromTime( &parentParticle.localColor1,
                pEmitter->m_pParentEmitter->m_pEmitterRes,
                parentRandom,
                pEmitter->m_pParentEmitter->m_Color1,
                color1Anim,
                pEmitter->m_pParentEmitter->m_EmitterAnimation.alpha0.x,
                parentLife,
                parentTime );

            if( pEmitter->m_pEmitterData->inherit.color1 )
            {
                pParticleProperty->initColor1[ writeIndex ].x = nn::util::VectorGetX( parentParticle.localColor1 );
                pParticleProperty->initColor1[ writeIndex ].y = nn::util::VectorGetY( parentParticle.localColor1 );
                pParticleProperty->initColor1[ writeIndex ].z = nn::util::VectorGetZ( parentParticle.localColor1 );
            }

            if( pEmitter->m_pEmitterData->inherit.alpha1 )
            {
                pParticleProperty->initColor1[ writeIndex ].w = nn::util::VectorGetW( parentParticle.localColor1 );
            }
        }
    }

    return;
}// NOLINT(impl/function_size)

//---------------------------------------------------------------------------
//  検索して死亡枠があればそこに追加、死亡枠が無い場合は後続に追加。
//---------------------------------------------------------------------------
int EmitterCalculator::EmitBySearchOrder( uint8_t* pOutEmitDone, Emitter* pEmitter, int emitCount, bool forceSearch, const EmitReservationInfo* pEmitReservationInfo ) NN_NOEXCEPT
{
    float random = pEmitter->m_Random.GetFloat();  // 一度のEmit処理の間常に一定値になる乱数を生成。
    bool  addIndex = true;
    int   writeLastIndex = -1;

    // パーティクル生成後コールバック
    ParticleEmitCallback particleEmitCallbackEP = NULL;
    ParticleEmitCallback particleEmitCallbackCA = NULL;
    ParticleEmitCallback particleEmitCallbackCS = NULL;

    if( pEmitter->m_pCallbackSet[ CallbackSetType_EmitterPlugin ] )
    {
        particleEmitCallbackEP = pEmitter->m_pCallbackSet[ CallbackSetType_EmitterPlugin ]->particleEmit;
    }
    if( pEmitter->m_pCallbackSet[ CallbackSetType_CustomAction ] )
    {
        particleEmitCallbackCA = pEmitter->m_pCallbackSet[ CallbackSetType_CustomAction ]->particleEmit;
    }
    if( pEmitter->m_pCallbackSet[ CallbackSetType_CustomShader ] )
    {
        particleEmitCallbackCS = pEmitter->m_pCallbackSet[ CallbackSetType_CustomShader ]->particleEmit;
    }

    if( pEmitter->m_pEmitterRes->m_pResEmitter->volume.primEmitType == PrimitiveEmissionType_Unison )
    {
        // 等分割系（円・線）の一斉放出の場合は、放出レートを分割数倍にする
        // 「分割数ランダム」の存在があるのでこのタイミングで決めるしかない。
        const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;
        if( pEmitter->m_pEmitterRes->m_pResEmitter->volume.volumeType == EmitterVolumeType_CircleEquallyDivided )
        {
            emitCount *= pEmitterVolume->numDivideCircle - static_cast< int >( random * pEmitterVolume->numDivideCircleRandom * 0.01f * pEmitterVolume->numDivideCircle );
        }
        else if( pEmitter->m_pEmitterRes->m_pResEmitter->volume.volumeType == EmitterVolumeType_LineEquallyDivided )
        {
            emitCount *= pEmitterVolume->numDivideLine - static_cast< int >( random * pEmitterVolume->numDivideLineRandom * 0.01f * pEmitterVolume->numDivideLine );
        }
    }

    // 放出処理
    if( pEmitter->m_ForceAbortEmitSearch && *pOutEmitDone )
    {
        // 一度以上放出がされている場合（先頭にパーティクルが生成された後）で、かつ
        // パーティクル配列先頭がまだ生きている（≒まだ粒が増え続けている状態）の場合、
        // 寿命ランダムを除けば、サーチが失敗する可能性が高い。
        // したがって探索は最初からあきらめる。
        const float time = pEmitter->m_pParticleData[ 0 ].GetTime( pEmitter->m_Time );
        const float life = pEmitter->m_pParticleData[ 0 ].GetLife();
        const bool isParticleDead = ( ( time - life ) > 0.0f );
        if( isParticleDead )
        {
            pEmitter->m_ForceAbortEmitSearch = false;
        }
    }

    for( int i = 0; i < emitCount; i++ )
    {
        bool abortSearch = pEmitter->m_ForceAbortEmitSearch || pEmitter->m_AbortEmitSearch;   // 空きを探すのが無駄な状態かどうか
        bool isSearch = false;
        if( forceSearch )
        {
            isSearch = true;
        }

        // 書き込み先バッファ
        int writeIndex = pEmitter->m_ParticleAttrFillIndex;
        addIndex = true;

        // 寿命ランダム＆ワンタイムでない場合は書き込み先を検索する
        // MEMO: ComputeShader が探索中に書き戻してくるリスクがあるので一旦除外。
        //if( pEmitter->m_pEmitterData->ptcl.lifeRandom != 0 && pEmitter->m_pEmitterData->emission.isOneTime == 0 )
        //{
        //    isSearch = true;
        //}

        // これまで生成したバッファ内から空きを検索し、
        // 空きがあればそこを利用する。( なるべく左詰め )
        if( isSearch && !abortSearch )
        {
            int _writeIndex = -1;

            if( pEmitter->m_ParticleCount == 0 )
            {
                writeIndex = 0;
            }
            else
            {
                if( pEmitter->m_pEmitterData->emitter.calcType == EmitterCalculationMode_Cpu )
                {
                    // CPUの場合は、終了処理が行われるので、createIDで死亡判断
                    for( int j = 0; j < pEmitter->m_ParticleCount; j++ )
                    {
                        // 最後にパーティクルを生成したインデックスの後ろから調べると
                        // 空きが早く見つかる（はず）
                        //int index = pEmitter->m_LastIndexParticleCreation + j + 1;
                        //if( index >= pEmitter->m_ParticleCount )
                        //{
                        //    // 巻き戻った時の折り返し
                        //    index -= pEmitter->m_ParticleCount;
                        //}
                        if( pEmitter->m_pParticleData[ j ].createId == 0 )
                        {
                            _writeIndex = j;
                            addIndex = false;
                            break;
                        }
                    }
                }
                else
                {
                    // GPUの場合は、時間と寿命で死亡判断
                    for( int j = 0; j < pEmitter->m_ParticleCount; j++ )
                    {
                        // 最後にパーティクルを生成したインデックスの後ろから調べると
                        // 空きが早く見つかる（はず）
                        //int index = pEmitter->m_LastIndexParticleCreation + j + 1;
                        //if( index >= pEmitter->m_ParticleCount )
                        //{
                        //    // 巻き戻った時の折り返し
                        //    index -= pEmitter->m_ParticleCount;
                        //}
                        const float time = pEmitter->m_pParticleData[ j ].GetTime( pEmitter->m_Time );
                        const float life = pEmitter->m_pParticleData[ j ].GetLife();

                        if( ( time - life ) > 0.0f )
                        {
                            _writeIndex = j;
                            addIndex = false;
                            break;
                        }
                    }
                }
            }

            if( _writeIndex != -1 )
            {
                writeIndex = _writeIndex;
            }
            else
            {
                // （パーティクルが死なずに生成され続ける放出初期に頻出パターン）
                // 空きが見つからなかった場合、最速でも次の Calculate が来るまで空きができることはないので、
                // 以降は探索をあきらめるようにする。
                pEmitter->m_AbortEmitSearch = true;
            }
        }

        // 不正なデータがアタッチされていないかチェック
        if( pEmitter->m_pParticleData[ writeIndex ].pEmitterPluginData ||
            pEmitter->m_pParticleData[ writeIndex ].pUserData )
        {
            Warning( reinterpret_cast< void* >( pEmitter ), RuntimeWarningId_ParticleInstanceIsDirty );
            return writeLastIndex;
        }

        // 放出がされたフラグを埋める
        *pOutEmitDone = true;

        // CPUパーティクルの場合は、CPUプロパティバッファの方を初期化し、
        // GPUパーティクルの場合は、m_ParticlePropertyFrontの方を初期化し以降いじらない
        detail::ParticleProperty*   pParticleProperty = pEmitter->m_ParticlePropertyFront;
        if( pEmitter->m_pEmitterData->emitter.calcType == EmitterCalculationMode_Cpu )
        {
            pParticleProperty = &pEmitter->m_CpuParticleProperty;
        }

        // パーティクルの初期状態を計算する
        ParticleData*       pParticleData       = &pEmitter->m_pParticleData[ writeIndex ];
        ParentParticleData* pParentParticleData = ( pEmitter->m_pEmitterRes->m_ChildEmitterResCount ) ? &pEmitter->m_pParentParticleData[ writeIndex ] : NULL;
        if( InitializeParticle( pEmitter,
            writeIndex,
            pParticleData,
            pParentParticleData,
            pParticleProperty, writeIndex,
            i, emitCount, random, pEmitReservationInfo ) )
        {
            writeLastIndex = writeIndex;
            pEmitter->m_LastIndexParticleCreation = writeLastIndex;

            detail::Float3Copy( &pParticleProperty->localDiff[ writeIndex ], pParticleProperty->localVec[ writeIndex ] );

            // チャイルドエミッタ継承処理
            if( pEmitter->m_pParentEmitter )
            {
                InheritParentParticleInfo( pEmitter, writeIndex );
            }

            // パーティクル生成コールバック
            {
                // マニュアル放出で EmitParticle 時にユーザーデータが設定されている場合は差し込む
                void* pInitialUserData = ( pEmitReservationInfo ) ? pEmitReservationInfo->pUserData : NULL;

                InvokeParticleEmitCallback( pEmitter, writeIndex, particleEmitCallbackEP, particleEmitCallbackCA, particleEmitCallbackCS, pInitialUserData );
            }

            // バッファ書き込みサイズの調整
            if( addIndex )
            {
                // バッファ書き込み量を加算
                pEmitter->m_ParticleAttrFillIndex = ( writeIndex + 1 );

                // バッファ折り返しチェック
                if( pEmitter->m_ParticleAttrFillIndex == pEmitter->m_MaxParticleAttrFill )
                {
                    pEmitter->m_ParticleAttrFillIndex = 0;
                }

                // 書き込んだ最大数
                if( pEmitter->m_ParticleCount < pEmitter->m_MaxParticleAttrFill )
                {
                    pEmitter->m_ParticleCount++;
                }
            }
        }
    }

    return writeLastIndex;
}// NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  放出処理を行います。
//---------------------------------------------------------------------------
int EmitterCalculator::Emit( uint8_t* pOutEmitDone, Emitter* pEmitter, int emitCount, bool forceSearch, const EmitReservationInfo* pEmitReservationInfo ) NN_NOEXCEPT
{
    int writeLastIndex = EmitBySearchOrder( pOutEmitDone, pEmitter, emitCount, forceSearch, pEmitReservationInfo );
    return writeLastIndex;
}


// Cafe挙動と合わせる為、_MatrixFromQuaternion をカスタマイズ
void _MatrixFromQuaternion(nn::util::Matrix4x3fType* pOutValue, const nn::util::Vector4fType& quaternion)
{
    float s = 2.0f / ( (nn::util::VectorGetX( quaternion ) * nn::util::VectorGetX( quaternion ) ) +
        ( VectorGetY( quaternion ) * VectorGetY( quaternion ) ) +
        ( VectorGetZ( quaternion ) * VectorGetZ( quaternion ) ) +
        ( VectorGetW( quaternion ) * VectorGetW( quaternion ) ));

    float xs = VectorGetX( quaternion ) * s;
    float ys = VectorGetY( quaternion ) * s;
    float zs = VectorGetZ( quaternion ) * s;

    float wx = VectorGetW( quaternion ) * xs;
    float wy = VectorGetW( quaternion ) * ys;
    float wz = VectorGetW( quaternion ) * zs;

    float xx = VectorGetX( quaternion ) * xs;
    float xy = VectorGetX( quaternion ) * ys;
    float xz = VectorGetX( quaternion ) * zs;

    float yy = VectorGetY( quaternion ) * ys;
    float yz = VectorGetY( quaternion ) * zs;
    float zz = VectorGetZ( quaternion ) * zs;

    nn::util::Matrix4x3fType matrix = NN_UTIL_MATRIX_4X3F_INITIALIZER(
        ( 1.f - (yy + zz) ), ( xy + wz ),         ( xz - wy ),
        (xy - wz),           ( 1.f - (xx + zz) ), ( yz + wx ),
        (xz + wy ),          ( yz - wx ),         ( 1.f - (xx + yy) ),
        0.0f,                0.0f,                0.0f );
    *pOutValue = matrix;
}

//------------------------------------------------------------------------------
//  パーティクルを初期化します。
//------------------------------------------------------------------------------
bool EmitterCalculator::InitializeParticle(
    Emitter*                pEmitter,
    int                     attrIndex,
    ParticleData*           pParticleData,
    ParentParticleData*     pParentParticleData,
    ParticleProperty*       pParticleProperty,
    int                     particleIndex,
    int                     emitIndex,
    int                     maxEmitCount,
    float                   randomValue,
    const EmitReservationInfo* pEmitReservationInfo ) NN_NOEXCEPT
{
    ResEmitter* pResEmitter = pEmitter->m_pEmitterData;
    EmitterSet* pEmitterSet = pEmitter->m_pEmitterSet;
    float         life = 0.0f;

    // localPos/localVec の初期化: （以降、フロントバッファから取得してOK）
    nn::util::Vector3fType localPos;
    nn::util::Vector3fType localVec;

    bool result = g_EmitFunctions[ pResEmitter->volume.volumeType ](
        &localPos,
        &localVec,
        pEmitter, emitIndex, maxEmitCount, randomValue, &pEmitter->m_EmitterAnimation );

    if( !result )
    {
        return false;
    }

    // パーティクル生成ID
    pEmitter->m_ParticleCreateId++;
    pParticleData->createId = pEmitter->m_ParticleCreateId;

    // Y軸拡散速度を計算する
    if( pEmitter->m_pEmitterData->ptclVel.xzDiffusion != 0.0f )
    {
        nn::util::Vector3fType yAxisVel = NN_UTIL_VECTOR_3F_INITIALIZER(
            nn::util::VectorGetX( localPos ),
            0.0f,
            nn::util::VectorGetZ( localPos ) );

        if( nn::util::VectorLengthSquared( yAxisVel ) <= std::numeric_limits< float >::epsilon() )
        {
            nn::util::VectorSetX( &yAxisVel, pEmitter->m_Random.GetFloat() * 2.0f - 1.0f );
            nn::util::VectorSetZ( &yAxisVel, pEmitter->m_Random.GetFloat() * 2.0f - 1.0f );
        }
        nn::util::VectorNormalize( &yAxisVel, yAxisVel );
        nn::util::VectorMultiply( &yAxisVel, yAxisVel, pEmitter->m_pEmitterData->ptclVel.xzDiffusion );
        nn::util::VectorAdd( &localVec, localVec, yAxisVel );
    }

    float velRand = 1.0f - pEmitter->m_Random.GetFloat() * ( pEmitter->m_pEmitterData->ptclVel.velRandom / 100.f ) * pEmitterSet->m_RandomVel;
    float dirVel = pEmitter->m_EmitterAnimation.designatedDirScale * pEmitterSet->m_DirectionalVel;
    float angle = pEmitter->m_pEmitterData->ptclVel.diffusionDirAngle;

    // 位置ランダム
    if( pEmitter->m_pEmitterData->emission.posRand != 0.0f )
    {
        nn::util::Vector3fType posRandomVec3f;
        const nn::util::Vector3fType& randomVec3f = pEmitter->m_Random.GetNormalizedVec3();
        nn::util::VectorMultiply( &posRandomVec3f, randomVec3f, pEmitter->m_pEmitterData->emission.posRand );
        nn::util::VectorAdd( &localPos, localPos, posRandomVec3f );
    }

    nn::util::Vector3fType ax, ay, az, aw;

    if( pEmitReservationInfo )
    {
        // マニュアル放出の場合で、各パーティクルの位置情報がある場合は加算。
        if ( pEmitReservationInfo->isEmitMatrix )
        {
            // 生成時のエミッタマトリクス
            if ( pParticleProperty->emitterMatrixSrt0 )
            {
                // 「位置のみ追従」「追従なし」
                nn::util::MatrixGetAxisX( &ax, pEmitReservationInfo->emitParam );
                nn::util::MatrixGetAxisY( &ay, pEmitReservationInfo->emitParam );
                nn::util::MatrixGetAxisZ( &az, pEmitReservationInfo->emitParam );
                nn::util::VectorSet( &aw, 0, 0, 0 );

                nn::util::Vector3fType emitPos;
                nn::util::MatrixGetAxisW( &emitPos, pEmitReservationInfo->emitParam );

                if( pEmitter->GetResEmitter()->emitter.followType == detail::ParticleFollowType_EmitterPosition )
                {
                    // 「位置のみ追従」: 位置だけ加算
                    // MEMO: 位置だけはエミッタ追従するので、あまり想定された使い方ではない
                    nn::util::VectorAdd( &localPos, localPos, emitPos );
                }
                else
                {
                    // 「追従なし」: 生成時のエミッタマトリクスに位置を加算
                    nn::util::VectorAdd( &aw, aw, emitPos );
                }
            }
            else
            {
                // 「完全追従」: 与えられた行列で変換
                // MEMO: 粒の大きさ・回転ではなく、 localPos を SRT 変換するのであまり想定された使い方ではない
                nn::util::VectorTransform( &localPos, localPos, pEmitReservationInfo->emitParam );
            }
        }
        else
        {
            if( pParticleProperty->emitterMatrixSrt0 )
            {
                // 生成時のエミッタエミッタマトリクスを記録
                nn::util::VectorSet( &ax, 1, 0, 0 );
                nn::util::VectorSet( &ay, 0, 1, 0 );
                nn::util::VectorSet( &az, 0, 0, 1 );
                nn::util::VectorSet( &aw, 0, 0, 0 );

                nn::util::Vector3fType emitPos;
                nn::util::MatrixGetAxisW( &emitPos, pEmitReservationInfo->emitParam );

                if( pEmitter->GetResEmitter()->emitter.followType == detail::ParticleFollowType_EmitterPosition )
                {
                    // 「位置のみ追従」: 位置だけ加算
                    nn::util::VectorAdd( &localPos, localPos, emitPos );
                }
                else
                {
                    // 「追従なし」: 生成時のエミッタマトリクスに位置を加算
                    nn::util::VectorAdd( &aw, aw, emitPos );
                }
            }
            else
            {
                // 「完全追従」: 位置だけ加算
                nn::util::Vector3fType emitPos;
                nn::util::MatrixGetAxisW( &emitPos, pEmitReservationInfo->emitParam );
                nn::util::VectorAdd( &localPos, localPos, emitPos );
            }
        }
    }
    else
    {
        nn::util::MatrixGetAxisX( &ax, pEmitter->m_MatrixSrt );
        nn::util::MatrixGetAxisY( &ay, pEmitter->m_MatrixSrt );
        nn::util::MatrixGetAxisZ( &az, pEmitter->m_MatrixSrt );
        nn::util::MatrixGetAxisW( &aw, pEmitter->m_MatrixSrt );
    }

    // 生成時のエミッタマトリクス
    if ( pParticleProperty->emitterMatrixSrt0 )
    {
        pParticleProperty->emitterMatrixSrt0[ particleIndex ].v[ 0 ] = nn::util::VectorGetX( ax );
        pParticleProperty->emitterMatrixSrt0[ particleIndex ].v[ 1 ] = nn::util::VectorGetX( ay );
        pParticleProperty->emitterMatrixSrt0[ particleIndex ].v[ 2 ] = nn::util::VectorGetX( az );
        pParticleProperty->emitterMatrixSrt0[ particleIndex ].v[ 3 ] = nn::util::VectorGetX( aw );
        pParticleProperty->emitterMatrixSrt1[ particleIndex ].v[ 0 ] = nn::util::VectorGetY( ax );
        pParticleProperty->emitterMatrixSrt1[ particleIndex ].v[ 1 ] = nn::util::VectorGetY( ay );
        pParticleProperty->emitterMatrixSrt1[ particleIndex ].v[ 2 ] = nn::util::VectorGetY( az );
        pParticleProperty->emitterMatrixSrt1[ particleIndex ].v[ 3 ] = nn::util::VectorGetY( aw );
        pParticleProperty->emitterMatrixSrt2[ particleIndex ].v[ 0 ] = nn::util::VectorGetZ( ax );
        pParticleProperty->emitterMatrixSrt2[ particleIndex ].v[ 1 ] = nn::util::VectorGetZ( ay );
        pParticleProperty->emitterMatrixSrt2[ particleIndex ].v[ 2 ] = nn::util::VectorGetZ( az );
        pParticleProperty->emitterMatrixSrt2[ particleIndex ].v[ 3 ] = nn::util::VectorGetZ( aw );
    }

    nn::util::VectorStore( pParticleProperty->localPos[particleIndex].v, localPos );

    nn::util::Vector3f designatedDir;
    nn::util::VectorLoad( &designatedDir, pEmitter->m_pEmitterData->ptclVel.designatedDir );

    if( pEmitter->m_pEmitterData->emission.isWorldOrientedVelocity )
    {
        // 指定方向初速をワールド座標系で適用
        // ビルボードと同じで、要するに後でワールド変換されるのを加味して逆に回しておく
        pEmitter->TransformToLocalVec( &designatedDir, designatedDir, particleIndex );
    }

    // ローカル速度
    if( angle == 0.0f )
    {
        nn::util::VectorSetX( &localVec, ( nn::util::VectorGetX( localVec ) + nn::util::VectorGetX( designatedDir ) * dirVel ) * velRand );
        nn::util::VectorSetY( &localVec, ( nn::util::VectorGetY( localVec ) + nn::util::VectorGetY( designatedDir ) * dirVel ) * velRand );
        nn::util::VectorSetZ( &localVec, ( nn::util::VectorGetZ( localVec ) + nn::util::VectorGetZ( designatedDir ) * dirVel ) * velRand );
    }
    else
    {
        // TODO : なるべく : 最適化

        // 指定方向拡散角度の処理
        // (0,1,0)を基準にランダムを適用した円錐状ベクトルを生成する
        angle = angle / 90.0f;
        angle = 1.0f - angle;

        float rot = pEmitter->m_Random.GetFloat() * 2.0f * nn::util::FloatPi;
        float sinV;
        float cosV;
        nn::util::SinCosEst( &sinV, &cosV, rot );
        float y = pEmitter->m_Random.GetFloat() * ( 1.0f - angle ) + angle;
        float r = SafeSqrt( 1.0f - y * y );

        nn::util::Vector3fType velocity = NN_UTIL_VECTOR_3F_INITIALIZER( r * cosV, y, r * sinV );

        // (0, 1, 0)を指定方向に回転させる回転行列を生成し適用する
        nn::util::Vector4fType q;

        nn::util::Vector3fType from = NN_UTIL_VECTOR_3F_INITIALIZER( 0.0f, 1.0f, 0.0f );
        nn::util::QuaternionMakeVectorRotation( &q, from, designatedDir );

        nn::util::Matrix4x3fType qmat;
        //nn::util::MatrixFromQuaternion( &qmat, q );
        _MatrixFromQuaternion( &qmat, q );
        nn::util::VectorTransform( &velocity, velocity, qmat );

        nn::util::Vector3fType vecTemp;
        nn::util::VectorMultiply( &vecTemp, velocity, dirVel);
        nn::util::VectorAdd( &vecTemp, localVec, vecTemp );
        nn::util::VectorMultiply( &localVec, vecTemp, velRand);
    }

    // ローカル拡散速度
    const nn::util::Vector3fType& randomVec3f = pEmitter->m_Random.GetVector3();

    nn::util::VectorSetX( &localVec, nn::util::VectorGetX( localVec ) + nn::util::VectorGetX( randomVec3f ) * pEmitter->m_pEmitterData->ptclVel.diffusion.x );
    nn::util::VectorSetY( &localVec, nn::util::VectorGetY( localVec ) + nn::util::VectorGetY( randomVec3f ) * pEmitter->m_pEmitterData->ptclVel.diffusion.y );
    nn::util::VectorSetZ( &localVec, nn::util::VectorGetZ( localVec ) + nn::util::VectorGetZ( randomVec3f ) * pEmitter->m_pEmitterData->ptclVel.diffusion.z );


    // エミッタ速度継承
    {
        nn::util::Vector3fType vecTemp;
        float inheritRate = pEmitter->m_pEmitterData->ptclVel.emVelInherit;

        nn::util::VectorMultiply( &vecTemp, pEmitter->m_EmitterLocalVec, inheritRate);
        nn::util::VectorAdd( &localVec, localVec, vecTemp );
    }

    // ワールド加算ベクトル
    nn::util::Vector3fType move;
    nn::util::Matrix4x3fType invMatrixRt;
    nn::util::MatrixInverse( &invMatrixRt, pEmitter->m_MatrixRt );
    nn::util::VectorTransformNormal( &move, pEmitterSet->m_AdditionalVel, invMatrixRt );
    nn::util::VectorAdd( &localVec, localVec, move );

    nn::util::VectorStore( pParticleProperty->localVec[particleIndex].v, localVec );

    // スケール
    nn::util::Float4& scale = pParticleProperty->scale[particleIndex];
    if( pResEmitter->ptclScale.baseRandom.x != pResEmitter->ptclScale.baseRandom.y )
    {
        scale.x = pEmitter->m_EmitterAnimation.particleScale.x *
            ( 1.f - pResEmitter->ptclScale.baseRandom.x / 100.0f * pEmitter->m_Random.GetFloat() ) *
            nn::util::VectorGetX( pEmitterSet->m_ParticleEmissionScale );
        scale.y = pEmitter->m_EmitterAnimation.particleScale.y *
            ( 1.f - pResEmitter->ptclScale.baseRandom.y / 100.0f * pEmitter->m_Random.GetFloat() ) *
            nn::util::VectorGetY( pEmitterSet->m_ParticleEmissionScale );
        scale.z = pEmitter->m_EmitterAnimation.particleScale.z *
            ( 1.f - pResEmitter->ptclScale.baseRandom.z / 100.0f * pEmitter->m_Random.GetFloat() ) *
            nn::util::VectorGetZ( pEmitterSet->m_ParticleEmissionScale );
    }
    else
    {
        float random = ( 1.f - pResEmitter->ptclScale.baseRandom.x / 100.0f * pEmitter->m_Random.GetFloat() );
        scale.x = pEmitter->m_EmitterAnimation.particleScale.x * random * nn::util::VectorGetX( pEmitterSet->m_ParticleEmissionScale );
        scale.y = pEmitter->m_EmitterAnimation.particleScale.y * random * nn::util::VectorGetY( pEmitterSet->m_ParticleEmissionScale );
        scale.z = pEmitter->m_EmitterAnimation.particleScale.z * random * nn::util::VectorGetZ( pEmitterSet->m_ParticleEmissionScale );
    }

    pParticleData->collisionCount = 0;

    // 運動量ランダム
    pParticleProperty->scale[particleIndex].w = 1.0f + pResEmitter->ptcl.dynamicsRand - pResEmitter->ptcl.dynamicsRand * pEmitter->m_Random.GetFloat() * 2.0f;

    // 生成時間
    pParticleData->createTime = pEmitter->m_Time;
    pParticleProperty->localVec[particleIndex].w = pParticleData->createTime;

    // パーティクル寿命
    if( !pResEmitter->ptcl.isLifeInfinity )
    {
        life = static_cast< float >( ( pEmitter->m_EmitterAnimation.particleLife - pEmitter->m_EmitterAnimation.particleLife * ( pEmitter->m_Random.GetInteger( pResEmitter->ptcl.lifeRandom ) * 0.01f ) ) );
        life *= pEmitter->m_ParticleLifeScale;
        life *= pEmitter->m_pEmitterSet->GetParticleLifeScale();
    }
    else
    {
        life = static_cast< float >( SystemParameters_InfiniteLifeValue );
    }

    pParticleData->life = life;
    pParticleProperty->localPos[particleIndex].w = life;

    if( pParentParticleData )
    {
        // 軽量版チャイルドで使用しているメンバのクリア
        for( int i = 0; i < SystemParameters_MaxEmitterInclusionCount; ++i )
        {
            pParentParticleData->lightChildEmitCount[ i ] = 0;
            pParentParticleData->lightChildEmitSaving[ i ] = 0;
            pParentParticleData->lightChildEmitInterval[ i ] = 0;
            pParentParticleData->lightChildEmitDone[ i ] = 0;
        }
        pParentParticleData->lightChildEmitterTime = 0;
        pParentParticleData->lightChildEmitterLife = pParticleData->GetLife();
    }

    // ランダム値
    pParticleProperty->random[particleIndex].x = pEmitter->m_Random.GetFloat();
    pParticleProperty->random[particleIndex].y = pEmitter->m_Random.GetFloat();
    pParticleProperty->random[particleIndex].z = pEmitter->m_Random.GetFloat();
    pParticleProperty->random[particleIndex].w = pEmitter->m_Random.GetFloat();

    // 初期回転値
    pParticleProperty->initRotate[particleIndex].x = nn::util::VectorGetX( pEmitter->m_pEmitterRes->m_InitialRotate ) + nn::util::VectorGetX( pEmitter->GetEmitterSet()->m_InitialRoate );
    pParticleProperty->initRotate[particleIndex].y = nn::util::VectorGetY( pEmitter->m_pEmitterRes->m_InitialRotate ) + nn::util::VectorGetY( pEmitter->GetEmitterSet()->m_InitialRoate );
    pParticleProperty->initRotate[particleIndex].z = nn::util::VectorGetZ( pEmitter->m_pEmitterRes->m_InitialRotate ) + nn::util::VectorGetZ( pEmitter->GetEmitterSet()->m_InitialRoate );
    pParticleProperty->initRotate[particleIndex].w = 0;

    if ( pParticleProperty->initColor0 )
    {
        const nn::util::Float4 One = NN_UTIL_FLOAT_4_INITIALIZER( 1, 1, 1, 1 );

        // カラー0値
        pParticleProperty->initColor0[particleIndex] = One;

        // カラー1値
        pParticleProperty->initColor1[particleIndex] = One;
    }

    //----------------------------------
    // 子エミッタが存在する場合の処理
    //----------------------------------
    for( int i = 0; i < pEmitter->m_pEmitterRes->m_ChildEmitterResCount; i++ )
    {
        if( pEmitter->m_pChildEmitterResSet[ i ] )
        {
            if( pEmitter->m_pChildEmitterResSet[ i ]->m_pResEmitter->inherit.enableEmitterParticle )
            {
                Emitter* pChildEmitter = pEmitterSet->CreateEmitter( pEmitter->m_pChildEmitterResSet[ i ], 0, pEmitter, i );
                if( !pChildEmitter )
                {
                    continue;
                }

                // 親パーティクルに子エミッタを登録
                pParentParticleData->pChildEmitter[ i ] = pChildEmitter;

                // 親エミッタの情報を登録
                pChildEmitter->m_ParentEmitterCreateId = pEmitter->m_EmitterCreateId;       // 親エミッタの生成ID
                pChildEmitter->m_ParentParticleCreateId = pEmitter->m_ParticleCreateId;     // 親パーティクルの生成ID
                pChildEmitter->m_ParentParticleLife = life;                                 // 親パーティクルの情報
                pChildEmitter->m_ParentParticleBirthTime = pEmitter->m_Time;                // 親パーティクルの生成時間
                pChildEmitter->m_ParentParticleAttrIndex = attrIndex;                       // 親となるパーティクルの情報を保存する

                nn::util::VectorLoad( &pChildEmitter->m_ParentParticleLocalPos, pParticleProperty->localPos[ particleIndex ].v );
                nn::util::VectorLoad( &pChildEmitter->m_ParentParticleLocalVec, pParticleProperty->localVec[ particleIndex ].v );
                nn::util::VectorLoad( &pChildEmitter->m_ParentParticleScale, pParticleProperty->scale[ particleIndex ] );
                nn::util::VectorLoad( &pChildEmitter->m_ParentParticleRotate, pParticleProperty->initRotate[ particleIndex ] );
                nn::util::VectorLoad( &pChildEmitter->m_ParentParticleRandom, pParticleProperty->random[ particleIndex ] );
                nn::util::MatrixIdentity( &pChildEmitter->m_MatrixSrt );
                nn::util::MatrixIdentity( &pChildEmitter->m_MatrixRt );
            }
            else
            {
                // 「軽量版チャイルド」の場合はエミッタを作成しない（ nn::vfx::EmitterSet::Initialize() 時に共有エミッタを作成している）
                // 共通エミッタはすでに m_pChildHead[ i ] にあるのでポインタを渡しておく
                pParentParticleData->pChildEmitter[ i ] = pEmitter->m_pChildHead[ i ];
            }
        }
        else
        {
            pParentParticleData->pChildEmitter[ i ] = NULL;
        }
    }

    return true;
} // NOLINT(readability/fn_size)

} // namespace detail
} // namespace vfx
} // namespace nn
