﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_CurlNoiseData.h>
#include <nn/vfx/vfx_MemUtil.h>
#include <nn/vfx/vfx_System.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif

namespace nn {
namespace vfx {
namespace detail {

CurlNoiseTable      g_CurlNoiseTable;
CurlNoiseTexture    g_CurlNoiseTexture;

//! カールノイズテーブル（符号付8ビット整数）
static const int8_t CurlNoiseVectorFieldData[ 32 * 32 * 32 * 3 ] =
{
#include "vfx_CurlNoiseDataTable.ipp"
};

//---------------------------------------------------
//  CurlNoiseフィールド コールバック
//---------------------------------------------------
void CalculateParticleBehavior_FieldCurlNoise( nn::util::Vector3fType* pOutPos, nn::util::Vector3fType* pOutVec, Emitter* pEmitter, const ParticleProperty* pParticleProperty, int particlleIndex ) NN_NOEXCEPT
{
    const ResFieldCurlNoiseData* pCurlNoiseData = pEmitter->GetEmitterResource()->m_pFieldCurlNoiseData;
    const float rate = pEmitter->GetFrameRate();
    const float emitterTime = pEmitter->GetTime();

    nn::util::Vector3fType speedMult;
    nn::util::VectorLoad( &speedMult, pCurlNoiseData->fieldCurlNoiseInfluence.v );
    const float sizeMult = pCurlNoiseData->fieldCurlNoiseScale;              // 渦のスケール

    nn::util::Vector3fType speedOffset;
    nn::util::VectorLoad( &speedOffset, pCurlNoiseData->fieldCurlNoiseSpeed.v );
    float offsetBase = pCurlNoiseData->fieldCurlNoiseBase;               // ノイズのオフセット
    if( pCurlNoiseData->isFieldCurlNoiseBaseRandom & 0x01 ) offsetBase *= pParticleProperty->random[particlleIndex].x;  // ノイズのオフセットランダム
    nn::util::Vector3fType offsetBase3f = NN_UTIL_VECTOR_3F_INITIALIZER( offsetBase, offsetBase, offsetBase );

    // パーティクル位置から受ける力を計算
    nn::util::Vector3fType particlePos( *pOutPos );
    if( pCurlNoiseData->isWorldCoordinate > 0 )
    {
        pEmitter->TransformToWorldPos( &particlePos, particlePos, particlleIndex );
    }

    nn::util::Vector3fType textureCoord;
    nn::util::Vector3fType temp[ 2 ];
    nn::util::VectorMultiply( &temp[ 0 ], particlePos, sizeMult );
    nn::util::VectorMultiply( &temp[ 1 ], speedOffset, emitterTime );
    nn::util::VectorAdd( &textureCoord, temp[ 0 ], temp[ 1 ] );
    nn::util::VectorAdd( &textureCoord, textureCoord, offsetBase3f );

    // サンプリング（ 32 x 32 x 32 の空間から ）
    nn::util::Vector3fType curlNoise;
    const int interporateType = ( pCurlNoiseData->isFieldCurlNoiseInterpolation & 0x01 );                 // 補間タイプ(0:補間なし 1以上:補間あり)
    if( interporateType == 0 )
    {
        g_CurlNoiseTable.GetCurlNoiseInt8Value( &curlNoise,
            static_cast< int >( nn::util::VectorGetX( textureCoord ) ),
            static_cast< int >( nn::util::VectorGetY( textureCoord ) ),
            static_cast< int >( nn::util::VectorGetZ( textureCoord ) ) );
    }
    else
    {
        g_CurlNoiseTable.GetCurlNoiseS8Interpolate( &curlNoise,
            nn::util::VectorGetX( textureCoord ),
            nn::util::VectorGetY( textureCoord ),
            nn::util::VectorGetZ( textureCoord ) );
    }

    if( pCurlNoiseData->isWorldCoordinate > 0 )
    {
        pEmitter->TransformToLocalVec( &curlNoise, curlNoise, particlleIndex );
    }

    // 力を速度に加算
    nn::util::VectorMultiply( &temp[ 0 ], curlNoise, speedMult );
    nn::util::VectorMultiply( &temp[ 0 ], temp[ 0 ], rate );
    nn::util::VectorAdd( pOutVec, *pOutVec, temp[ 0 ] );
}

//---------------------------------------------------
//  CurlNoise パラメータ初期化処理
//---------------------------------------------------
void InitializeCurlNoise( nn::gfx::Device* pDevice, Heap* heap ) NN_NOEXCEPT
{
    g_CurlNoiseTable.Initialize( 32, reinterpret_cast< const void* >( &CurlNoiseVectorFieldData[ 0 ] ) );
    g_CurlNoiseTexture.Initialize( pDevice, heap );
}

//---------------------------------------------------
//  CurlNoise パラメータ終了処理
//---------------------------------------------------
void FinalizeCurlNoise( nn::gfx::Device* pDevice ) NN_NOEXCEPT
{
    g_CurlNoiseTexture.Finalize( pDevice );
}

//---------------------------------------------------------------------------
//  CurlNoise テクスチャでヒープから確保したメモリサイズを取得する
//---------------------------------------------------------------------------
size_t GetCurlNoiseTextureAllocatedSize() NN_NOEXCEPT
{
    return g_CurlNoiseTexture.m_AllocatedSize;
}

//---------------------------------------------------------------------------
//  カールノイズテクスチャを取得します。
//---------------------------------------------------------------------------
const CurlNoiseTexture& GetCurlNoiseTexture() NN_NOEXCEPT
{
    return g_CurlNoiseTexture;
}

//---------------------------------------------------------------------------
//  テクスチャービューをディスクリプタプールに登録します。
//---------------------------------------------------------------------------
void RegisterCurlNoiseTextureViewToDescriptorPool( nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    g_CurlNoiseTexture.RegisterTextureViewToDescriptorPool( pRegisterTextureSlotCallback, pUserData );
}

//---------------------------------------------------------------------------
//  テクスチャービューをディスクリプタプールから解放します。
//---------------------------------------------------------------------------
void UnRegisterCurlNoiseTextureViewToDescriptorPool( nn::vfx::UnregisterTextureViewSlot pUunregisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    g_CurlNoiseTexture.UnRegisterTextureViewToDescriptorPool( pUunregisterTextureSlotCallback, pUserData );
}

//---------------------------------------------------------------------------
//  初期化処理
//---------------------------------------------------------------------------
void CurlNoiseTexture::Initialize( nn::gfx::Device* pDevice, Heap* heap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );
    NN_UNUSED( heap );
    NN_SDK_ASSERT( m_pAllocatedBufferPtr == NULL );

    static const int CurlNoiseTextureWidth  = 32;
    static const int CurlNoiseTextureHeight = 32;
    static const int CurlNoiseTextureDepth  = 32;

    nn::gfx::Texture::InfoType texInfo;
    texInfo.SetDefault();
    texInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_Texture );
    texInfo.SetWidth( CurlNoiseTextureWidth );
    texInfo.SetHeight( CurlNoiseTextureHeight );
    texInfo.SetDepth( CurlNoiseTextureDepth );
    texInfo.SetImageStorageDimension( nn::gfx::ImageStorageDimension_3d );
    texInfo.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );

    nn::gfx::MemoryPool::InfoType mpInfo;
    mpInfo.SetDefault();
    mpInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached );

    size_t texAlign  = nn::gfx::Texture::CalculateMipDataAlignment( pDevice, texInfo );
    size_t poolAlign = nn::gfx::MemoryPool::GetPoolMemoryAlignment( pDevice, mpInfo );

    m_TextureSize   = CurlNoiseTextureWidth * CurlNoiseTextureHeight * CurlNoiseTextureDepth * 4;
    m_AllocatedSize   = m_TextureSize + texAlign;
    m_AllocatedSize   = nn::util::align_up( m_AllocatedSize, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( pDevice, mpInfo ) );
    m_pAllocatedBufferPtr = AllocFromStaticHeap( m_AllocatedSize, poolAlign );
    NN_SDK_ASSERT_NOT_NULL( m_pAllocatedBufferPtr );

    mpInfo.SetPoolMemory( m_pAllocatedBufferPtr, m_AllocatedSize );
    m_MemoryPool.Initialize( pDevice, mpInfo );

    m_pTextureImage = nn::util::BytePtr( m_pAllocatedBufferPtr ).Advance( texAlign ).Get();

#if NN_GFX_IS_TARGET_NVN
    // CurlNoiseVectorFieldData そのままではコピーできないので、一時領域に画像データを作る。
    uint32_t* tempBuff = reinterpret_cast<uint32_t *>( heap->Alloc( CurlNoiseTextureDepth * CurlNoiseTextureHeight * CurlNoiseTextureWidth * sizeof( uint32_t  ) ) );
    NN_SDK_ASSERT_NOT_NULL( tempBuff );
#endif
    int index = 0;
    const int pitchZ = CurlNoiseTextureDepth * CurlNoiseTextureHeight;
    const int pitchY = CurlNoiseTextureHeight;
    // RGBAテーブルへ置き換え
    for( int z = 0; z < CurlNoiseTextureDepth; z++ )
    {
        for( int y = 0; y < CurlNoiseTextureHeight; y++ )
        {
            for( int x = 0; x < CurlNoiseTextureWidth; x++ )
            {
                const int i = ( z * pitchZ ) + ( y * pitchY ) + x;
                const uint8_t r = static_cast< uint8_t >( CurlNoiseVectorFieldData[ index + 0 ] + 128 );
                const uint8_t g = static_cast< uint8_t >( CurlNoiseVectorFieldData[ index + 1 ] + 128 );
                const uint8_t b = static_cast< uint8_t >( CurlNoiseVectorFieldData[ index + 2 ] + 128 );
                const uint8_t a = 255;
                index += 3;

#if NN_GFX_IS_TARGET_GL
                // ARGB の順番で配置
                const uint32_t color = ( a << 24 ) | ( r << 16 ) | ( g << 8 ) | ( b << 0 );
                ( reinterpret_cast< uint32_t* >( m_pTextureImage ) )[ i ] = color;
#elif NN_GFX_IS_TARGET_NVN
                // ARGB の順番で配置
                const uint32_t color = ( a << 24 ) | ( r << 16 ) | ( g << 8 ) | ( b << 0 );
                tempBuff[ i ] = color;
#elif NN_GFX_IS_TARGET_VK
                // ARGB の順番で配置
                const uint32_t color = ( a << 24 ) | ( r << 16 ) | ( g << 8 ) | ( b << 0 );
                ( reinterpret_cast< uint32_t* >( m_pTextureImage ) )[ i ] = color;
#else
    #error "未対応の低レベル API"
#endif
            }
        }
    }

    texInfo.SetMipCount( 1 );
//  texInfo.SetMipDataPtr( &m_pTextureImage, 1 );
    m_Texture.Initialize( pDevice, texInfo, &m_MemoryPool, texAlign, m_TextureSize );

    {
        nn::gfx::TextureView::InfoType info;
        info.SetDefault();
        info.SetTexturePtr( &m_Texture );
        info.SetImageDimension( nn::gfx::ImageDimension_3d );
        info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );

        m_TextureView.Initialize( pDevice, info );
    }

#if NN_GFX_IS_TARGET_NVN
    // 一時領域から Texture へ画像データを CPU 上で転送
    NVNcopyRegion region;
    region.xoffset = region.yoffset = region.zoffset = 0;
    region.width = region.height = region.depth = 32;
    nvnTextureWriteTexels( m_Texture.ToData()->pNvnTexture, nullptr, &region, tempBuff );
    nvnTextureFlushTexels( m_Texture.ToData()->pNvnTexture, nullptr, &region );
    heap->Free( tempBuff );
    tempBuff = NULL;
#endif
}

//---------------------------------------------------------------------------
//  終了処理
//---------------------------------------------------------------------------
void CurlNoiseTexture::Finalize( nn::gfx::Device* pDevice ) NN_NOEXCEPT
{
    if( !m_pAllocatedBufferPtr ) return;

    m_Texture.Finalize( pDevice );
    m_TextureView.Finalize( pDevice );

    m_MemoryPool.Finalize( pDevice );
    FreeFromStaticHeap( m_pAllocatedBufferPtr );
    m_pAllocatedBufferPtr = NULL;
}

//---------------------------------------------------------------------------
//  無効化
//---------------------------------------------------------------------------
void CurlNoiseTexture::Invalidate() NN_NOEXCEPT
{
    m_pTextureImage = NULL;
    m_AllocatedSize = 0;
}

//---------------------------------------------------------------------------
//  テクスチャービューをディスクリプタプールに登録します。
//---------------------------------------------------------------------------
void CurlNoiseTexture::RegisterTextureViewToDescriptorPool( nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    pRegisterTextureSlotCallback( &m_DescriptorSlot, m_TextureView, pUserData );
}

//---------------------------------------------------------------------------
//  テクスチャービューをディスクリプタプールから解放します。
//---------------------------------------------------------------------------
void CurlNoiseTexture::UnRegisterTextureViewToDescriptorPool( nn::vfx::UnregisterTextureViewSlot pUnregisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    pUnregisterTextureSlotCallback( &m_DescriptorSlot, m_TextureView, pUserData );
    m_DescriptorSlot.Invalidate();
}

}
}
}
