﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_AreaLoop.h>
#include <nn/vfx/vfx_EmitterCalc.h>
#include <nn/vfx/vfx_System.h>

namespace nn {
namespace vfx {
namespace detail {


//-------------------------------------------------------------------------
// 範囲内ループの 定数バッファ 構造体
//-------------------------------------------------------------------------
struct AreaLoopConstantBuffer
{
    nn::util::Float4        param0;     //!< リピート毎のオフセット(xyz) / カメラ前ループフラグ
    nn::util::Float4        param1;     //!< 端をαで薄める割合(xyz) / empty
    nn::util::Float4x4      mat;        //!< 座標系変換行列
    nn::util::Float4x4      invMat;     //!< 座標系変換逆行列
    nn::util::Float4        param2;     //!< クリッピングタイプ / クリッピングY座標 / empty / empty
    nn::util::Float4        param3;     //!< 仮想箱の大きさ(xyz) / empty
};

bool AreaLoopSystem::Initialize( System* pSystem ) NN_NOEXCEPT
{
    NN_UNUSED( pSystem );

    CallbackSet set;
    set.emitterPreCalculate = NULL;
    set.emitterInitialize = AreaLoopSystem::EmitterInitializeCallback;
    set.particleEmit = NULL;
    set.particleRemove = NULL;
    set.particleCalculate = NULL;
    set.emitterPostCalculate = NULL;
    set.emitterDraw = AreaLoopSystem::EmitterDrawCallback;
    set.emitterFinalize = NULL;
    set.renderStateSet = AreaLoopSystem::RenderStateSetCallback;
    pSystem->SetEmitterPluginCallbackSet( EmitterPluginCallbackIndex_4, set );

    return true;
}

bool AreaLoopSystem::EmitterInitializeCallback( EmitterInitializeArg& arg ) NN_NOEXCEPT
{
    NN_UNUSED( arg );
    return true;//arg.pEmitter->InitializeCustomConstantBuffer( ConstantBufferIndex, sizeof( AreaLoopConstantBuffer ) );
}

bool AreaLoopSystem::RenderStateSetCallback( RenderStateSetArg& arg ) NN_NOEXCEPT
{
    NN_UNUSED( arg );
    // 何もしない！（自動定数バッファセットをキャンセル）
    return true;
}

bool AreaLoopSystem::EmitterDrawCallback( EmitterDrawArg& arg ) NN_NOEXCEPT
{
    AreaLoopSystem::Draw( arg.pCommandBuffer, arg.pEmitter, arg.shaderType, arg.pUserParam, arg.pDrawParameterArg );
    return true;
}

bool AreaLoopSystem::Draw( nn::gfx::CommandBuffer* pCommandBuffer, Emitter* pEmitter, ShaderType shaderType, void* pUserParam, DrawParameterArg* pDrawParameterArg ) NN_NOEXCEPT
{
    ResEPAreaLoop* areaLoopData = reinterpret_cast< ResEPAreaLoop* >( pEmitter->GetEmitterResource()->m_pEmitterPluginData );

    //-------------------------------------------------------------------------
    // シェーダ切り替え
    //-------------------------------------------------------------------------
    Shader* pShader = pEmitter->GetShader( shaderType );
    if( !pShader )
    {
        Warning( pEmitter, RuntimeWarningId_NoShaderExists );
        return false;
    }

    pCommandBuffer->SetShader( pShader->GetShader(), nn::gfx::ShaderStageBit_All );

    //-------------------------------------------------------------------------
    // 定数バッファ デフォルトパラメータを設定する
    //-------------------------------------------------------------------------
    AreaLoopConstantBuffer defalutConstantBuffer;

    defalutConstantBuffer.param1.x = areaLoopData->alphaRatio.x;
    defalutConstantBuffer.param1.y = areaLoopData->alphaRatio.y;
    defalutConstantBuffer.param1.z = areaLoopData->alphaRatio.z;
    defalutConstantBuffer.param1.w = 0;
    defalutConstantBuffer.param2.x = static_cast< float >( areaLoopData->clippingType );
    defalutConstantBuffer.param2.y = areaLoopData->clippingWorldHeight;
    defalutConstantBuffer.param2.z = 0;
    defalutConstantBuffer.param2.w = 0;
    defalutConstantBuffer.param3.x = areaLoopData->areaSize.x;
    defalutConstantBuffer.param3.y = areaLoopData->areaSize.y;
    defalutConstantBuffer.param3.z = areaLoopData->areaSize.z;
    defalutConstantBuffer.param3.w = 0;

    //-------------------------------------------------------------------------
    // 変換行列の事前計算
    //-------------------------------------------------------------------------
    {
        //-------------------------------------------------------------------------
        // MEMO: このあたりの行列計算はまだ（可読性／保守性を落とさない範囲で）最適化できるはず。
        //-------------------------------------------------------------------------
        {
            nn::util::Matrix4x4fType temp;
            nn::util::MatrixIdentity( &temp );
            nn::util::MatrixStore( &defalutConstantBuffer.mat, temp );
        }

        // 共通化できる部分もあるが、可読性を重視して行列を作る部分は完全に分岐させる
        if( areaLoopData->isCameraLoop == 0 )
        {
            // 標準タイプ
            nn::util::Matrix4x4fType boxMatrix;
            nn::util::MatrixIdentity( &boxMatrix );

            // 仮想箱系へのRT行列（箱の大きさは割合位置を求めるのに頂点シェーダ側で使用する）
            // Rotate
            nn::util::Matrix4x4fType boxRotMatrix;
            nn::util::Vector3fType rot = NN_UTIL_VECTOR_3F_INITIALIZER( areaLoopData->areaRotate.x, areaLoopData->areaRotate.y, areaLoopData->areaRotate.z );
            EmitterCalculator::MakeRotationMatrixXYZ( &boxRotMatrix, rot );
            nn::util::MatrixMultiply( &boxMatrix, boxRotMatrix, boxMatrix );

            // Translate
            // 標準の場合、エミッタセットSRTを考慮
            nn::util::Matrix4x4fType m = NN_UTIL_MATRIX_4X4F_INITIALIZER(
                0, 0, 0, 0,
                0, 0, 0, 0,
                0, 0, 0, 0,
                areaLoopData->areaPos.x, areaLoopData->areaPos.y, areaLoopData->areaPos.z, 0 );
            nn::util::MatrixAdd( &boxMatrix, boxMatrix, m );

            // 標準の場合、EmitterSetSRTを適用する
            nn::util::Matrix4x4fType result;
            nn::util::Matrix4x4fType emitterSrt;

            detail::Matrix4x3fTo4x4f(&emitterSrt, pEmitter->GetMatrixSrt() );
            nn::util::MatrixMultiply( &result, boxMatrix, emitterSrt );
            nn::util::MatrixStore( &defalutConstantBuffer.mat, result );
            //defalutConstantBuffer.mat.SetMult( boxMatrix, nw::math::Matrix44( pEmitter->GetMatrixSrt() ).Transpose() );
        }
        else
        {
            // カメラ前ループの場合
            nn::util::Matrix4x4fType boxMatrix;
            nn::util::MatrixIdentity( &boxMatrix );
            ViewParam* pViewParam = pDrawParameterArg->m_pViewParam;

            // カメラ位置への変換行列を作成。
            nn::util::Matrix4x4fType cameraMatrix;
            nn::util::MatrixIdentity( &cameraMatrix );

            nn::util::Matrix4x4fType m = NN_UTIL_MATRIX_4X4F_INITIALIZER(
                0, 0, 0, 0,
                0, 0, 0, 0,
                0, 0, 0, 0,
                pViewParam->eyePos.x, pViewParam->eyePos.y, pViewParam->eyePos.z, 0 );
            nn::util::MatrixAdd( &cameraMatrix, cameraMatrix, m );

            // 仮想箱系へのRT行列（箱の大きさは割合位置を求めるのに頂点シェーダ側で使用する）
            // Rotate:
            nn::util::Matrix4x4fType boxRotMatrix;
            nn::util::Vector3fType rot = NN_UTIL_VECTOR_3F_INITIALIZER( areaLoopData->areaRotate.x, areaLoopData->areaRotate.y, areaLoopData->areaRotate.z );
            EmitterCalculator::MakeRotationMatrixXYZ( &boxRotMatrix, rot );
            nn::util::MatrixMultiply( &boxMatrix, boxRotMatrix, boxMatrix );

            // Translate:
            // ビルボード行列を使ってカメラ系での移動量に回転させておく
            const nn::util::Float4x4& bldMatrix = pViewParam->billboardMatrix;
            nn::util::Vector3fType pos = NN_UTIL_VECTOR_3F_INITIALIZER( areaLoopData->areaPos.x, areaLoopData->areaPos.y, areaLoopData->areaPos.z );

            const float tx = nn::util::VectorGetX( pos ) * bldMatrix.m[ 0 ][ 0 ] + nn::util::VectorGetY( pos ) * bldMatrix.m[ 0 ][ 1 ] + nn::util::VectorGetZ( pos ) * bldMatrix.m[ 0 ][ 2 ];
            const float ty = nn::util::VectorGetX( pos ) * bldMatrix.m[ 1 ][ 0 ] + nn::util::VectorGetY( pos ) * bldMatrix.m[ 1 ][ 1 ] + nn::util::VectorGetZ( pos ) * bldMatrix.m[ 1 ][ 2 ];
            const float tz = nn::util::VectorGetX( pos ) * bldMatrix.m[ 2 ][ 0 ] + nn::util::VectorGetY( pos ) * bldMatrix.m[ 2 ][ 1 ] + nn::util::VectorGetZ( pos ) * bldMatrix.m[ 2 ][ 2 ];
            nn::util::Matrix4x4fType mTrans = NN_UTIL_MATRIX_4X4F_INITIALIZER(
                0,  0,  0,  0,
                0,  0,  0,  0,
                0,  0,  0,  0,
                tx, ty, tz, 0 );
            nn::util::MatrixAdd( &boxMatrix, boxMatrix, mTrans );

            // EmitterSetSRTは考慮しない（それほど意味が無い上に混乱の元になるので）
            // カメラ系行列 → 仮想箱系の順番
            {
                nn::util::Matrix4x4fType temp;
                nn::util::MatrixMultiply( &temp, boxMatrix, cameraMatrix );
                nn::util::MatrixStore( &defalutConstantBuffer.mat, temp );
            }
        }

        //-------------------------------------------------------------------------
        // 逆行列をあらかじめ計算して保存しておく
        //-------------------------------------------------------------------------
        {
            nn::util::Matrix4x4fType temp;
            nn::util::MatrixLoad( &temp, defalutConstantBuffer.mat );
            nn::util::MatrixInverse( &temp, temp );
            nn::util::MatrixStore( &defalutConstantBuffer.invMat, temp );
        }
    }

    //-------------------------------------------------------------------------
    // 描画ループ
    //-------------------------------------------------------------------------
    nn::util::Float3 repeatOffset = NN_UTIL_FLOAT_3_INITIALIZER( 0, 0, 0 );
    const int maxCount = static_cast< int >( areaLoopData->repeatNum + 1 );
    for( int i = 0; i < maxCount; ++i )
    {
        // メモリ確保
        nn::gfx::GpuAddress address;
        AreaLoopConstantBuffer* pConstantBuffer = reinterpret_cast<AreaLoopConstantBuffer*>(
            pDrawParameterArg->m_pDrawTempBuffer->Map( &address, sizeof( AreaLoopConstantBuffer ) ) );
        if ( !pConstantBuffer ) { break; }

        // 定数バッファのパラメータのセット
        detail::Float3Copy( &pConstantBuffer->param0, repeatOffset );
        pConstantBuffer->param0.w = areaLoopData->isCameraLoop;
        pConstantBuffer->param1 = defalutConstantBuffer.param1;
        pConstantBuffer->param2 = defalutConstantBuffer.param2;
        pConstantBuffer->param3 = defalutConstantBuffer.param3;

        // 定数バッファ の行列のセット
        pConstantBuffer->mat = defalutConstantBuffer.mat;
        pConstantBuffer->invMat = defalutConstantBuffer.invMat;

        // 定数バッファ 設定
        const int emitterPluginConstantBufferSlotV = pShader->GetVertexConstantBufferSlot( Shader::ConstantBufferType_EmitterPlugin );
        const int emitterPluginConstantBufferSlotP = pShader->GetPixelConstantBufferSlot( Shader::ConstantBufferType_EmitterPlugin );

        if( emitterPluginConstantBufferSlotV != InvalidValueId_ConstantBufferSlotId )
        {
            pCommandBuffer->SetConstantBuffer( emitterPluginConstantBufferSlotV, nn::gfx::ShaderStage_Vertex, address, sizeof( AreaLoopConstantBuffer ) );
        }
        if( emitterPluginConstantBufferSlotP != InvalidValueId_ConstantBufferSlotId )
        {
            pCommandBuffer->SetConstantBuffer( emitterPluginConstantBufferSlotP, nn::gfx::ShaderStage_Pixel, address, sizeof( AreaLoopConstantBuffer ) );
        }

        // 描画処理に回す
        pEmitter->GetEmitterCalculator()->DrawEmitterUsingBoundShader( pCommandBuffer, pEmitter, pShader, pUserParam, pDrawParameterArg );

        // リピート用オフセットを加算
        repeatOffset.x += areaLoopData->repeatOffsetPos.x;
        repeatOffset.y += areaLoopData->repeatOffsetPos.y;
        repeatOffset.z += areaLoopData->repeatOffsetPos.z;
    }

    return true;
} //NOLINT(impl/function_size)

} // namespace detail
} // namespace vfx
} // namespace nn

