﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NOMINMAX
#define NOMINMAX
#endif

#include <sstream>

#include <nn/util/util_Arithmetic.h>

#include <nn/nn_Macro.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_StringUtil.h>
#include <nn/ui2d/detail/ui2d_Log.h>

#include <nn/ui2d/viewer/ui2d_Viewer2.h>
#include <nn/font/font_ScalableFontUtil.h>

#include <nn/ui2d/viewer/ui2d_ToolConnector.h>

#if defined(NN_UI2D_VIEWER_ENABLED)

//--------------------------------------------------------------------------------------------------
// ミドルウェア情報
#include <nn/nn_Middleware.h>
#include <nn/nn_Version.h>

#define NN_DETAIL_XX_MACRO_TO_STRING(x) NN_DETAIL_XX_MACRO_TO_STRING_(x)
#define NN_DETAIL_XX_MACRO_TO_STRING_(x) #x

#define NW_UI2D_VIEWER_MIDDLEWARE_SYMBOL(buildOption) "NintendoWare_Ui2d_Viewer2_For_Develop-" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MAJOR) "_" \
NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MINOR) "_" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MICRO) "-" #buildOption

namespace {
#if defined(NN_SDK_BUILD_DEBUG)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_UI2D_VIEWER_MIDDLEWARE_SYMBOL(Debug));
#elif defined(NN_SDK_BUILD_DEVELOP)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_UI2D_VIEWER_MIDDLEWARE_SYMBOL(Develop));
#elif defined(NN_SDK_BUILD_RELEASE)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_UI2D_VIEWER_MIDDLEWARE_SYMBOL(Release));
#endif
}

//--------------------------------------------------------------------------------------------------

namespace
{

    //--------------------------------------------------------------------------------------------------

    float StrToFloat_(const char* pStr)
    {
        std::stringstream ss;
        ss << pStr;

        float outValue;
        ss >> outValue;

        return outValue;
    }
}

//--------------------------------------------------------------------------------------------------


namespace nn
{
namespace ui2d
{
namespace viewer
{

//--------------------------------------------------------------------------------------------------

static const size_t RecvBufferSize = 512;

//--------------------------------------------------------------------------------------------------
ScreenManager::ScreenManager()
#if defined(NN_BUILD_CONFIG_OS_WIN)
    : m_ResourceAccessor(*DirResourceAccessor::Create())
    , m_pUserDataForDescriptorSlotAllocator(NULL)
#else
    : m_pUserDataForDescriptorSlotAllocator(NULL)
#endif
{
}

//--------------------------------------------------------------------------------------------------
ScreenManager::~ScreenManager()
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    delete &m_ResourceAccessor;
#endif
}

//---------------------------------------------------------------------------
nn::ui2d::viewer::Screen*
ScreenManager::CreateScreen()
{
    auto* pScreen = nn::ui2d::Layout::AllocateAndConstruct<nn::ui2d::viewer::Screen>();
    NN_SDK_ASSERT_NOT_NULL(pScreen);

    pScreen->Initialize(this);

    return pScreen;
}

//---------------------------------------------------------------------------
nn::ui2d::viewer::Screen*
ScreenManager::CreateScreenAnimMode(nn::gfx::Device* pDevice, const void* lytRes, const char* pName, const nn::ui2d::Layout::BuildOption& buildOption, BuildResultInformation* pBuildResultInformation)
{
    NN_UNUSED(pName);
    NN_SDK_ASSERT_NOT_NULL(lytRes);

    auto* pScreen = CreateScreen();

    pScreen->BuildLayout(pDevice, lytRes, &m_ResourceAccessor, &m_ResourceAccessor, &buildOption, pBuildResultInformation);

    DoRegisterResourceTextureView();

    return pScreen;
}

//---------------------------------------------------------------------------
nn::ui2d::viewer::Screen*
ScreenManager::CreateScreen(nn::gfx::Device* pDevice, const void* lytRes, const char* pName, BuildResultInformation* pBuildResultInformation)
{
    NN_UNUSED(pName);
    NN_SDK_ASSERT_NOT_NULL(lytRes);

    auto* pScreen = CreateScreen();

    pScreen->BuildLayout(pDevice, lytRes, &m_ResourceAccessor, pBuildResultInformation);

    DoRegisterResourceTextureView();

    return pScreen;
}



//--------------------------------------------------------------------------------------------------
void ScreenManager::RegisterAndCompleteAllGlyphsForScalableFontTextureCache(nn::ui2d::viewer::Screen* pScreen)
{
    if (pScreen->GetLayout() == NULL)
    {
        return;
    }

    nn::font::RegisterGlyphFromTextBoxRecursive(pScreen->GetLayout()->GetRootPane());

    nn::font::TextureCache* pTextureCache = m_ScalableFontManager.GetTextureCache();
    if (pTextureCache != NULL)
    {
        pTextureCache->UpdateTextureCache();
        pTextureCache->CompleteTextureCache();
    }
}

//--------------------------------------------------------------------------------------------------

Screen::Screen()
    : m_Layout(NULL)
    , m_pControlCreator(NULL)
    , m_IsPerspectiveProjection(false)
    , m_PerspectiveFovyDeg(40.f)
    , m_PerspectiveNear(1.f)
    , m_PerspectiveFar(10000.f)
    , m_IsViewMtxDirty(false)
    , m_AnimateOnce(false)
{
}

//--------------------------------------------------------------------------------------------------

void Screen::Initialize(ScreenManager* pScreenManager)
{
    m_pScreenManager = pScreenManager;
    m_DrawInfo.SetGraphicsResource(m_pScreenManager->pGraphicsResource);
}

//--------------------------------------------------------------------------------------------------

ViewerScreen::ViewerScreen()
: m_pScreen(NULL)
, m_ViewSize(ViewSize_Original)
, m_MainViewportSize(Size::Create(0.0f, 0.0f))
, m_IsSrgbWriteEnabled(true)
, m_IsHudVisible(true)
, m_Fps(Fps_60)
, m_LoadFunction(NULL)
, m_UnloadFunction(NULL)
, m_GetUserShaderInformationFromUserDataCallback(NULL)
, m_PostCalculateCallback(NULL)
, m_PreReloadLayoutDataCallback(NULL)
, m_PostReloadLayoutDataCallback(NULL)
{
    NN_USING_MIDDLEWARE(g_MiddlewareInfo);

    m_PreviewOption.reset();

    const float defaultColor = 169.0f / 255.0f;
    m_BackgroundColor[0] = defaultColor;
    m_BackgroundColor[1] = defaultColor;
    m_BackgroundColor[2] = defaultColor;
    m_BackgroundColor[3] = 1.0f;
}

//--------------------------------------------------------------------------------------------------

ViewerScreen::~ViewerScreen()
{
}

//--------------------------------------------------------------------------------------------------

void ViewerScreen::Initialize(Screen& screen, const InitializeArg& initializeArg)
{
    m_MainViewportSize = initializeArg.mainViewportSize;

    NN_SDK_ASSERT_NOT_NULL(initializeArg.loadFunction);
    NN_SDK_ASSERT_NOT_NULL(initializeArg.unloadFunction);
    m_LoadFunction = initializeArg.loadFunction;
    m_UnloadFunction = initializeArg.unloadFunction;

    m_pScreen = &screen;

    ToolConnector::Initialize(
        Layout::GetAllocateFunction(),
        Layout::GetFreeFunction(),
        Layout::GetUserDataForAllocator(),
        RecvBufferSize,
        RecvBufferSize,
        false,
        initializeArg.toolConnectorThreadPriority);

    // Test
    screen.SetControlCreator(nn::ui2d::Layout::AllocateAndConstruct<nn::ui2d::DefaultControlCreatorEx>(&screen.GetButtonGroup(), &screen.GetControlList()));
}

//--------------------------------------------------------------------------------------------------

void ViewerScreen::Finalize(nn::gfx::Device* pDevice)
{
    ToolConnector::Finalize();
    FinalizeResources(pDevice);

    // Test
    auto* pControlCreator = m_pScreen->GetControlCreator();
    nn::ui2d::Layout::DeleteObj(pControlCreator);
    m_pScreen->SetControlCreator(nullptr);
}

//--------------------------------------------------------------------------------------------------


void ViewerScreen::SetBackgroundColor(const char* pHexColorString)
{
    uint32_t    color;
    std::sscanf(pHexColorString, "0x%x", &color);

    m_BackgroundColor[0] = ((color & 0xFF000000) >> 24) / 255.0f;
    m_BackgroundColor[1] = ((color & 0x00FF0000) >> 16) / 255.0f;
    m_BackgroundColor[2] = ((color & 0x0000FF00) >> 8) / 255.0f;
    m_BackgroundColor[3] = ((color & 0x000000FF) >> 0) / 255.0f;
}

//--------------------------------------------------------------------------------------------------

bool ViewerScreen::ConvertHostToFsPath(char *outFsPath, const char *mountPoint, const char *inHostPath)
{
    NN_SDK_ASSERT_NOT_NULL(outFsPath);
    NN_SDK_ASSERT_NOT_NULL(mountPoint);
    NN_SDK_ASSERT_NOT_NULL(inHostPath);

    if ( std::strlen(inHostPath) <= 0 )
    {
        return false;
    }

    char part[PathLengthMax];

    if (0 == strncmp("/cygdrive/", inHostPath, 10) && isalpha(inHostPath[10]) && inHostPath[11] == '/')
    {
        // CYGWIN形式
        nn::util::SNPrintf(outFsPath, PathLengthMax, "%s/%c", mountPoint, inHostPath + 10);

        nn::util::Strlcpy(part, inHostPath + 11, PathLengthMax);
    }
    else if (isalpha(inHostPath[0]) && inHostPath[1] == ':')
    {
        // DOS形式
        nn::util::SNPrintf(outFsPath, PathLengthMax, "%s/%c", inHostPath[0]);
        nn::util::Strlcpy(part, inHostPath + 2, PathLengthMax);
    }
    else
    {
        // パスが不正。
        return false;
    }

    size_t partLen = std::strlen(part);
    for(size_t i = 0; i < partLen; i ++)
    {
        size_t length = strlen(outFsPath);
        NN_SDK_ASSERT(length < PathLengthMax - 1);

        outFsPath[length]        = part[i] == '\\' ? '/' : part[i];
        outFsPath[length + 1]    = NULL;
    }

    return true;
}

//--------------------------------------------------------------------------------------------------

#if defined(NN_BUILD_CONFIG_OS_WIN)

namespace
{
    bool IsOptionExist_(int argc, const char** argv, const char* name)
    {
        for (int i = 0; i < argc; i++)
        {
            if (strcmp(argv[i], name) == 0)
            {
                return true;
            }
        }

        return false;
    }

    const char* GetOptionValue_(int argc, const char** argv, const char* name)
    {
        for (int i = 0; i < argc; i++)
        {
            if (strcmp(argv[i], name) == 0)
            {
                if (i + 1 < argc)
                {
                    return argv[i + 1];
                }
                else {
                    return nullptr;
                }
            }
        }

        return nullptr;
    }
}

//--------------------------------------------------------------------------------------------------

bool ViewerScreen::PreviewByCommandLineOption(nn::gfx::Device* pDevice, int argc, const char** argv)
{
    NN_UNUSED(argc);
    NN_UNUSED(argv);

    m_PreviewOption.reset();

    const char* pValue = nullptr;

    pValue = GetOptionValue_(argc, argv, "-m");
    if(pValue != nullptr && pValue[0] == 'c')
    {
        // モードを指定する
        m_PreviewOption.previewMode = PreviewMode_Control;
    }

    pValue = GetOptionValue_(argc, argv, "-n");
    if(pValue != nullptr)
    {
        // 複数レイアウトがあった場合のルートのレイアウト名を指定する
        nn::util::Strlcpy(m_PreviewOption.layoutName, pValue, PathLengthMax);
        std::strncat(m_PreviewOption.layoutName, ".bflyt", PathLengthMax - strlen(m_PreviewOption.layoutName) - 1);
    }

    if(IsOptionExist_(argc, argv, "-g"))
    {
        // sRGBライトを行い、ガンマ値を補正します。
        m_PreviewOption.isSrgbWriteEnabled = true;
    }

    pValue = GetOptionValue_(argc, argv, "-bg_color");
    if (pValue != nullptr)
    {
        SetBackgroundColor(pValue);
    }

    if (argc > 1)
    {
        // 最後の引数をファイル名と仮定します。
        nn::util::Strlcpy(m_PreviewOption.path, argv[argc - 1], PathLengthMax);
        StartPreview(pDevice, m_PreviewOption);
        return true;
    }
    else
    {
        return false;
    }
}

#endif

//--------------------------------------------------------------------------------------------------

bool ViewerScreen::PreviewByCommandStr(nn::gfx::Device* pDevice, const char* commandStr)
{
    m_PreviewOption.reset();

    ParseCommandStr(&m_PreviewOption, commandStr);

    StartPreview(pDevice, m_PreviewOption);

    return true;
}

//--------------------------------------------------------------------------------------------------

void ViewerScreen::UpdateSystem(nn::gfx::Device* pDevice)
{
    uint8_t* message = ToolConnector::GetPacket();
    if (message)
    {
        PreviewByCommandStr(pDevice, reinterpret_cast<const char*>(message));
    }
}

//--------------------------------------------------------------------------------------------------

void ViewerScreen::UpdateInputs(nn::gfx::Device* pDevice, const nn::ui2d::viewer::Viewer::InputDeviceState& inputDeviceState)
{
    NN_UNUSED(pDevice);
    NN_UNUSED(inputDeviceState);


    // HUD の表示・非表示きりかえ
    if(inputDeviceState.isTrigX)
    {
        SetHudVisible(!m_IsHudVisible);
    }

    if (m_pScreen == NULL)
    {
        return;
    }

    nn::ui2d::Layout* pLayout = m_pScreen ? m_pScreen->GetLayout() : NULL;
    if (m_PreviewOption.previewMode == PreviewMode_Control && pLayout != NULL)
    {
        bool is_pointer_on = inputDeviceState.isMousePointerOn;
        if (is_pointer_on)
        {
            // ポインタの位置をレイアウトの空間に変換し、ButtonGroupのUpdateに与える
            nn::ui2d::Size viewSize = GetViewSize();
            nn::ui2d::Size layoutSize = pLayout->GetLayoutSize();
            float widthRate = layoutSize.width / viewSize.width;
            float heightRate = layoutSize.height / viewSize.height;
            float x, y; // 画面上のポインタ位置。中心原点、右下が大きくなる座標系
            float viewport_width, viewport_heigth;
            bool is_touch;
            bool is_release;

            x = 2.0f * inputDeviceState.pointerX / m_MainViewportSize.width - 1.0f;
            y = 2.0f * inputDeviceState.pointerY / m_MainViewportSize.height - 1.0f;
            viewport_width = m_MainViewportSize.width;
            viewport_heigth = m_MainViewportSize.height;

            is_touch = inputDeviceState.isTrigMouseLButton;
            is_release = inputDeviceState.isReleaseMouseLButton;

            nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER( x * (viewport_width / 2.f) * widthRate, -y * (viewport_heigth / 2.f) * heightRate );

            m_pScreen->GetButtonGroup().Update(&pos, is_touch, is_release);
            // test
            m_pScreen->TestUpdateControlGroupInput(&pos, is_touch, is_release);
        }
        else
        {
            // ポインタが画面に入っていないときはnullを与えてもよい
            m_pScreen->GetButtonGroup().Update(NULL, false);
        }

        //----------------------------------------------------------
        {
            auto& ctrlList = m_pScreen->GetControlList();
            for (auto iterCtrl = ctrlList.begin(); iterCtrl != ctrlList.end(); iterCtrl++)
            {
                auto& stateCtrl = *static_cast<nn::ui2d::StateMachineControl*>(const_cast<nn::ui2d::ControlBase*>(&(*iterCtrl)));
                if (!stateCtrl.GetStateMachine().IsInitialized())
                {
                    continue;
                }

                // TEST:キー操作で、変数を可変できるようにします。
                nn::ui2d::StateMachineVariableManager& variableManager = stateCtrl.GetStateMachine().m_StateMachineVariableManager;
                for (auto iter = variableManager.m_StateMachineVariableList.begin(); iter != variableManager.m_StateMachineVariableList.end(); iter++)
                {
                    if (iter->m_Content.m_Type != nn::ui2d::StateMachineVariableType_Float)
                    {
                        continue;
                    }

                    float val = variableManager.GetValueAsFloat(iter->m_Name);
                    if (inputDeviceState.isHoldLeft)
                    {
                        val -= 0.01f;
                    }
                    else if (inputDeviceState.isHoldRight)
                    {
                        val += 0.01f;
                    }

                    variableManager.SetFloatValue(iter->m_Name, val);
                }
            }
        }
    }
    else
    {
        nn::ui2d::viewer::AnimationManager2& animMgr = m_pScreen->GetAnimationManager();

        // 再生停止・リセット
        if (inputDeviceState.isTrigA)
        {
            if (animMgr.IsAnimationPlaying())
            {
                animMgr.StopAnimation();
            }
            else
            {
                animMgr.StartAnimation();
            }
        }
        else if (inputDeviceState.isTrigB)
        {
            animMgr.ResetAnimation();
        }

        nn::ui2d::Animator* animator = animMgr.GetAnimator();
        if (animator)
        {
            // 左右キーで、アニメーションフレームをコントロール
            if (inputDeviceState.isHoldLeft)
            {
                animator->StopAtCurrentFrame();
                if (animator->IsMinFrame())
                {
                    animator->StopAtEndFrame();
                }
                else
                {
                    animator->SetFrame(animator->GetFrame() - 1);
                }
            }
            else if (inputDeviceState.isHoldRight)
            {
                animator->StopAtCurrentFrame();
                if (animator->IsMaxFrame())
                {
                    animator->StopAtStartFrame();
                }
                else
                {
                    animator->SetFrame(animator->GetFrame() + 1);
                }
            }

            // 上下キーで、再生するアニメーションを切り替え
            if (inputDeviceState.isTrigUp)
            {
                if (animMgr.GetCurrentAnimationNo() == 0)
                {
                    animMgr.SetCurrentAnimationNo(pDevice, animMgr.GetAnimationCount() - 1);
                }
                else
                {
                    animMgr.SetCurrentAnimationNo(pDevice, animMgr.GetCurrentAnimationNo() - 1);
                }
                animMgr.StartAnimation();
            }
            else if (inputDeviceState.isTrigDown)
            {
                if (animMgr.GetAnimationCount() <= animMgr.GetCurrentAnimationNo() + 1)
                {
                    animMgr.SetCurrentAnimationNo(pDevice, 0);
                }
                else
                {
                    animMgr.SetCurrentAnimationNo(pDevice, animMgr.GetCurrentAnimationNo() + 1);
                }
                animMgr.StartAnimation();
            }
        }
    }

}// NOLINT(impl/function_size)

//--------------------------------------------------------------------------------------------------

void ViewerScreen::UpdateMenu()
{
    // 一時的にメニュー表示を廃止しています。
}

//--------------------------------------------------------------------------------------------------

void ViewerScreen::FinalizeResources(nn::gfx::Device* pDevice)
{
    auto& screenManager = *GetScreenManager();

    if (m_pScreen->GetLayout() != NULL)
    {
        // レイアウトが読み込まれていたらスケーラブルフォントを解放する。
        // m_ResourceAccessor.Finalize でアーカイブファイルが解放されるよりも先に
        // fcpx が使っている ResFont を解放しなければならない。
        screenManager.m_ScalableFontManager.FinalizeComplexFontSet(
            pDevice,
            screenManager.m_UnregisterTextureViewSlot,
            screenManager.m_pUserDataForDescriptorSlotAllocator);
    }

    // レイアウトリソースの解放
    m_pScreen->FinalizeResources(pDevice);

    // 既にアタッチされていたらDetach
    void* pDetachedData = screenManager.DetachArchive(pDevice);
    if (pDetachedData != NULL)
    {
        m_UnloadFunction(pDetachedData);
    }
}

//--------------------------------------------------------------------------------------------------
void ViewerScreen::AttatchArchive(const char* pArcPath)
{
    auto& screenManager = *GetScreenManager();

#if defined(NN_BUILD_CONFIG_OS_WIN)
    screenManager.AttatchArchive(pArcPath);
#else
    // バイナリリソースのルートディレクトリを指定してリソースアクセサを生成。
    void* pData = m_LoadFunction(pArcPath, nn::ui2d::ArchiveResourceAlignment);
    NN_SDK_ASSERT_NOT_NULL(pData);
    screenManager.AttatchArchive(pData, ".");
#endif
}

//--------------------------------------------------------------------------------------------------

void ViewerScreen::StartPreview(nn::gfx::Device* pDevice, const PreviewOption& option)
{
    NN_SDK_ASSERT_NOT_NULL(m_pScreen);

    auto& screenManager = *GetScreenManager();

    // 再生状態を継続するために以前の設定を残す
    Screen::AnimationPlayingState oldAnimationState;
    bool isAnimPlaying = m_pScreen->GetAnimationPlayingState(&oldAnimationState);

    // リソース再構築
    {
        FinalizeResources(pDevice);

        if (m_PreReloadLayoutDataCallback != NULL)
        {
            (*m_PreReloadLayoutDataCallback)();
        }

        // アーカイブをアタッチ
        AttatchArchive(option.path);

        // スケーラブルフォント関連の初期化
        screenManager.InitializeComplexFontSet(pDevice);
    }

    // スクリーン再構築
    {
        BuildResultInformation buildResultInformation;
        buildResultInformation.SetDefault();
        {
            // レイアウト名が指定されていないときは探す
            const char* layoutName = std::strlen(option.layoutName) <= 0 ? screenManager.m_ResourceAccessor.FindFile(nn::ui2d::ResourceTypeLayout, NULL) : option.layoutName;
            const void* lytRes = screenManager.m_ResourceAccessor.FindResourceByName(nn::ui2d::ResourceTypeLayout, layoutName);
            NN_SDK_ASSERT_NOT_NULL(lytRes);

            if (option.previewMode == PreviewMode_Animation)
            {
                nn::ui2d::Layout::BuildOption buildOption;
                buildOption.SetDefault();
                buildOption.pGetUserShaderInformationFromUserDataCallback = m_GetUserShaderInformationFromUserDataCallback;

                m_pScreen->BuildLayout(pDevice, lytRes, &screenManager.m_ResourceAccessor, &screenManager.m_ResourceAccessor, &buildOption, &buildResultInformation);
                screenManager.DoRegisterResourceTextureView();

                // アニメーション再生
                if (m_pScreen != NULL)
                {
                    const Screen::AnimationPlayingState* pOldAnimationState = m_PreviewOption.continueAnimation && isAnimPlaying ? &oldAnimationState : NULL;
                    if (pOldAnimationState != NULL)
                    {
                        m_pScreen->SetAnimationPlayingState(pDevice, *pOldAnimationState);
                    }
                    else
                    {
                        m_pScreen->GetAnimationManager().StartAnimation();
                    }
                }
            }
            else
            {
                m_pScreen->BuildLayout(pDevice, lytRes, &screenManager.m_ResourceAccessor, &buildResultInformation);
                screenManager.DoRegisterResourceTextureView();
            }
        }

        SetViewSize(m_ViewSize);

        if (m_PostReloadLayoutDataCallback != NULL)
        {
            (*m_PostReloadLayoutDataCallback)(*m_pScreen->GetDrawInfo(), buildResultInformation);
        }
    }

    m_IsSrgbWriteEnabled = option.isSrgbWriteEnabled != 0 ? true : false;
}

//--------------------------------------------------------------------------------------------------

ViewerScreen::OptionMode ViewerScreen::GetOptionMode(const char* buf)
{
    if (std::strcmp(buf, "-m") == 0)
    {
        return OptionMode_PreviewMode;
    }
    else if (std::strcmp(buf, "-n") == 0)
    {
        return OptionMode_LayoutName;
    }
    else if (std::strcmp(buf, "-projection") == 0)
    {
        return OptionMode_PerspectiveProjection;
    }
    else if (std::strcmp(buf, "-fov") == 0)
    {
        return OptionMode_PerspectiveFovy;
    }
    else if (std::strcmp(buf, "-near") == 0)
    {
        return OptionMode_Near;
    }
    else if (std::strcmp(buf, "-far") == 0)
    {
        return OptionMode_Far;
    }
    else if (std::strcmp(buf, "-viewmode") == 0)
    {
        return OptionMode_ViewSize;
    }
    else if (std::strcmp(buf, "-bg_color") == 0)
    {
        return OptionMode_BackgroundColor;
    }

    return OptionMode_None;
}

//--------------------------------------------------------------------------------------------------

void ViewerScreen::AnalyseOptionArgument(PreviewOption* option, OptionMode mode, const char* buf)
{
    NN_SDK_ASSERT_NOT_NULL(m_pScreen);

    switch(mode)
    {
    case OptionMode_PreviewMode:
        // モードを指定する
        if (buf[0] == 'c')
        {
            option->previewMode = PreviewMode_Control;
        }
        else
        {
            option->previewMode = PreviewMode_Animation;
        }
        break;
    case OptionMode_LayoutName:
        // レイアウト名を指定する
        nn::util::Strlcpy(option->layoutName, buf, PathLengthMax);
        std::strncat(option->layoutName, ".bflyt", PathLengthMax - strlen(option->layoutName) - 1);
        break;
    case OptionMode_PerspectiveProjection:
        m_pScreen->SetIsPerspectiveProjection(std::strcmp(buf, "pers") == 0);
        break;
    case OptionMode_PerspectiveFovy:
        m_pScreen->SetPerspectiveFovyDeg(StrToFloat_(buf));
        break;
    case OptionMode_Near:
        m_pScreen->SetPerspectiveNear(StrToFloat_(buf));
        break;
    case OptionMode_Far:
        m_pScreen->SetPerspectiveFar(StrToFloat_(buf));
        break;
    case OptionMode_ViewSize:
        if (std::strcmp(buf, "original") == 0)
        {
            m_ViewSize = ViewSize_Original;
        }
        else if (std::strcmp(buf, "fit_screen") == 0)
        {
            m_ViewSize = ViewSize_FitScreen;
        }
        break;
    case OptionMode_BackgroundColor:
        SetBackgroundColor(buf);
        break;
    case OptionMode_None:
    default:
        NN_SDK_ASSERT(false);
    }
}

//--------------------------------------------------------------------------------------------------

void ViewerScreen::ParseCommandStr(PreviewOption* option, const char* commandStr)
{
    OptionMode optionMode = OptionMode_None;
    const char* pCursor = commandStr;
    const char* pCursorNext = commandStr;

    char buf[PathLengthMax];

    while( pCursorNext != NULL)
    {
        pCursorNext = strchr(pCursor, ' ');

        if(pCursorNext == NULL)
        {
            // PCのファイルシステムにおける絶対パスが指定されます。
            nn::util::Strlcpy(option->path, pCursor, PathLengthMax);
            continue;
        }

        size_t copyLen = pCursorNext - pCursor;
        nn::util::Strlcpy(buf, pCursor, static_cast<int>(copyLen + 1));

        pCursor = pCursorNext + 1;

        if (optionMode == OptionMode_None)
        {
            if ((optionMode = GetOptionMode(buf)) != OptionMode_None)
            {
                continue;
            }
        }

        // 引数指定のないオプションを直接解析する。
        if (std::strcmp(buf, "-g") == 0)
        {
            option->isSrgbWriteEnabled = true;
            continue;
        }
        else if (std::strcmp(buf, "-is30fps") == 0)
        {
            m_Fps = Fps_30;
            continue;
        }
        else if (std::strcmp(buf, "-is60fps") == 0)
        {
            m_Fps = Fps_60;
            continue;
        }
        else if (std::strcmp(buf, "-continue-animation") == 0)
        {
            option->continueAnimation = true;
            continue;
        }

        if (optionMode != OptionMode_None)
        {
            AnalyseOptionArgument(option, optionMode, buf);
        }
        else
        {
            return ;
        }
        optionMode = OptionMode_None;
    }
}

//--------------------------------------------------------------------------------------------------

nn::ui2d::Size ViewerScreen::GetViewSize() const
{
    if (m_pScreen != NULL && m_ViewSize == ViewSize_Original)
    {
        return m_pScreen->GetViewSize();
    }
    else if (m_ViewSize == ViewSize_FitScreen)
    {
        return nn::ui2d::Size::Create(m_MainViewportSize.width, m_MainViewportSize.height);
    }
    else
    {
        return nn::ui2d::Size::Create(1920.f, 1080.f);
    }
}

//--------------------------------------------------------------------------------------------------

void ViewerScreen::SetViewSize(ViewSize viewSize)
{
    m_ViewSize = viewSize;

    auto targetViewSize = GetViewSize();
    m_pScreen->SetViewSize(targetViewSize, m_MainViewportSize);
}

//-------------------------------------------------------------------------
void ViewerScreen::GetBackgroundColor(float* pBackgroundColor) const
{
    const float srgbGammaConst = 2.2f;

    if (IsSrgbWriteEnabled())
    {
        pBackgroundColor[0] = std::pow(m_BackgroundColor[0], srgbGammaConst);
        pBackgroundColor[1] = std::pow(m_BackgroundColor[1], srgbGammaConst);
        pBackgroundColor[2] = std::pow(m_BackgroundColor[2], srgbGammaConst);
    }
    else
    {
        pBackgroundColor[0] = m_BackgroundColor[0];
        pBackgroundColor[1] = m_BackgroundColor[1];
        pBackgroundColor[2] = m_BackgroundColor[2];
    }
    pBackgroundColor[3] = m_BackgroundColor[3];
}

//-------------------------------------------------------------------------
void ViewerScreen::WriteHud(nn::font::TextWriter& writer) const
{
    if(!IsHudVisible())
    {
        return;
    }

    writer.Printf("---- New Viewer Test ---- \n");

    if (m_PreviewOption.previewMode == PreviewMode_Control)
    {
        writer.Printf(" Control Mode -------------------------------------------------------------------------\n");
        writer.Printf(" B:Reset  X:HideHUD\n");
        writer.Printf(" --------------------------------------------------------------------------------------\n");
    }else{
        writer.Printf(" Animation Mode -----------------------------------------------------------------------\n");
        writer.Printf(" A:Play/Stop  B:Reset  Left/Right:FrameControl  Up/Down:SelectAnimation  X:HideHUD\n");
        writer.Printf(" --------------------------------------------------------------------------------------\n");
    }

    nn::ui2d::Size  showSize = GetViewSize();

    // FitScreen モードの時は画面にレイアウトデータを画面サイズに拡大して表示しているため
    // GetViewSize で取得されるサイズは画面の大きさになります。
    // サイズに常に画面サイズを表示するよりもレイアウトデータのサイズを表示したほうがユーザーにとって
    // 自然で有益なため、表示するサイズを置き換えます。
    if (m_pScreen != NULL && m_pScreen->GetLayout() != NULL && m_ViewSize == ViewSize_FitScreen)
    {
        showSize = m_pScreen->GetLayout()->GetLayoutSize();
    }
    writer.Printf("  Size: %.0fx%.0f\n", showSize.width, showSize.height);
    writer.Printf("   FPS: %s\n", this->GetFps() == Fps_60 ? "60fps" : "30fps");
    writer.Printf(" Gamma: %s\n", this->IsSrgbWriteEnabled() ? "Linear" : "sRGB");

    if (m_PreviewOption.previewMode == PreviewMode_Control)
    {
        auto& ctrlList = m_pScreen->GetControlList();
        for (auto iterCtrl = ctrlList.begin(); iterCtrl != ctrlList.end(); iterCtrl++)
        {
            auto& stateCtrl = *static_cast<nn::ui2d::StateMachineControl*>(const_cast<nn::ui2d::ControlBase*>(&(*iterCtrl)));
            if (!stateCtrl.GetStateMachine().IsInitialized())
            {
                continue;
            }

            writer.Printf(" ----------------------------------------\n");

            {
                StateLayerSet& stateLayers = stateCtrl.GetStateMachine().m_StateLayerSet;
                for (auto iterLayer = stateLayers.begin(); iterLayer != stateLayers.end(); iterLayer++)
                {
                    writer.Printf(" Layer [%s]: current=%s animFrame=%4.1f/%4.1f Transition=%d, TransitionEnd=%d \n",
                        iterLayer->m_pName,
                        iterLayer->m_pCurrentState != nullptr ? iterLayer->m_pCurrentState->m_pName : "---",
                        iterLayer->m_AnimatorSlot.GetAnimationFrame(),
                        iterLayer->m_AnimatorSlot.GetAnimationFrameMax(),
                        !iterLayer->IsStateTransitionCompleted(),
                        iterLayer->IsStateTransitionJustCompleted());

                    auto* pTransition = iterLayer->GetActiveTransition();
                    for (auto iterState = iterLayer->m_StateSet.begin(); iterState != iterLayer->m_StateSet.end(); iterState++)
                    {
                        const bool isPrev = pTransition != nullptr && strcmp(iterState->m_pName, pTransition->m_pPrevStateName) == 0;
                        const bool isNext = pTransition != nullptr && strcmp(iterState->m_pName, pTransition->m_pNextStateName) == 0;
                        const bool blinkerFlag = (static_cast<int>(iterLayer->m_AnimatorSlot.GetAnimationFrame()) / 10) % 2 == 0;

                        writer.Printf(" [%s] %s %s %s\n",
                            iterState->m_pName,
                            strcmp(iterState->m_pName, iterLayer->m_pCurrentState->m_pName) == 0 ? "*" : " ",
                            isPrev && blinkerFlag ? "P" : " ",
                            isNext && !blinkerFlag ? "N" : " ");
                    }
                }
            }

            writer.Printf(" -------\n");

            // イベントキュー
            {
                StateMachineEventQueue& queue = stateCtrl.GetStateMachine().m_StateMachineEventQueue;
                float px = writer.GetCursorX();
                writer.Printf(" EventQueue:");
                for (auto iter = queue.m_StateMachineEvents.begin(); iter != queue.m_StateMachineEvents.end(); iter++)
                {
                    writer.Printf("[%d]", iter->kind);
                }

                writer.Printf("\n");
                writer.SetCursorX(px);
            }
        }
    }

    //writer.Printf("  Proj: %s\n", this->m_IsPerspectiveProjection ? "Perspective" : "Ortho");
    //writer.Printf("  FOVY: %.0f (degree)\n", m_PerspectiveFovyDeg);
    //writer.Printf("  Near: %.0f \n", m_PerspectiveNear);
    //writer.Printf("   Far: %.0f \n", m_PerspectiveFar);
}

} // namespace viewer
} // namespace ui2d
} // namespace nn

#endif // NN_UI2D_VIEWER_ENABLED
