﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/ui2d/viewer/ui2d_ToolConnector.h>
#include <nn/ige/ige_HtcsHelper.h>
#include <nn/ui2d/detail/ui2d_Log.h>

#include <nn/os.h>
#if defined(NN_UI2D_VIEWER_ENABLED)

namespace nn {
namespace ui2d {
namespace viewer {

/*
    定数
    */
static const char           STR_HEADER[] = "NW4FLytViewer";
static const int            InvalidSocket = -1;

volatile bool ToolConnector::g_IsConnected = false;

static volatile bool        g_IsOK = false;
static volatile bool        g_IsDetermined = true;

static AlignedAllocateFunctionWithUserData  g_pFnAlloc;
static FreeFunctionWithUserData             g_pFnFree;
static void*                g_pUserDataForAlloc;
static size_t               g_ReadBufferSize;       //!< 1度に受信するパケットサイズの上限です。
static uint8_t*             g_pReadBuffer;          //!< 通信スレッド用の受信バッファです。
static uint8_t*             g_pPresentBuffer;       //!< コネクタ利用者向けの公開バッファです。

enum
{
    taskAlignSize = 0x1000,
    taskMemsize   = 0x2000
};

nn::os::ThreadType          g_Thread;
uint8_t*                    g_StackBase;
TaskFunc                    g_Func;         //!< タスクメイン関数です。
#endif

static int                  g_Handle = InvalidSocket;

static size_t               g_ReadSize = 0;         //!< 実際に受信したサイズを格納します。
static volatile bool        g_IsReceived = false;  //!< パケットを受信済みかどうかを格納します。true の間は次の受信はブロックされます。

void ProcessThread(void* pArg) NN_NOEXCEPT
{
    NN_UNUSED(pArg);
    g_Func(0, NULL);
}
//------------------------------------------------------------------------------
//  初期化
//------------------------------------------------------------------------------
bool ToolConnector::Initialize(
    nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
    nn::FreeFunctionWithUserData            pFreeFunction,
    void*                                   pUserDataForAllocator,
    size_t                                  bufferSize,
    size_t                                  firstReadSize,
    bool                                    noUseInternalThread,
    int                                     priority
    ) NN_NOEXCEPT
{
    g_pFnAlloc          = pAllocateFunction;
    g_pFnFree           = pFreeFunction;
    g_pUserDataForAlloc = pUserDataForAllocator;
    g_pReadBuffer       = NULL;
    g_pPresentBuffer    = NULL;
    g_ReadBufferSize    = bufferSize;
    g_ReadSize          = firstReadSize;
    g_IsDetermined      = true;
    g_IsConnected       = false;
    g_IsReceived       = false;

    // 読み込みバッファ初期化
    g_pReadBuffer = reinterpret_cast< uint8_t* >( g_pFnAlloc( g_ReadBufferSize, 8, g_pUserDataForAlloc ) );
    memset( g_pReadBuffer, 0, g_ReadBufferSize );

    // 取得用バッファ初期化
    // gPresentBuffer の Size を 0 終端用に 1 増やす。
    g_pPresentBuffer = reinterpret_cast< uint8_t* >( g_pFnAlloc( g_ReadBufferSize + 1, 8, g_pUserDataForAlloc ) );
    memset( g_pPresentBuffer, 0, g_ReadBufferSize + 1 );

    // 通信スレッドの起動
    if( noUseInternalThread == false )
    {
        g_StackBase = reinterpret_cast< uint8_t* >( g_pFnAlloc( taskMemsize, taskAlignSize, g_pUserDataForAlloc ) );
        g_Func      = ThreadToolReceiver;

        nn::Result result = nn::os::CreateThread( &g_Thread,
                                                  ProcessThread,
                                                  reinterpret_cast< void* >( NULL ),
                                                  reinterpret_cast< void* >( g_StackBase ),
                                                  taskMemsize,
                                                  static_cast< int >( priority ) );
        if( result.IsSuccess() == false )
        {
            g_IsOK = false;
        }
        else
        {
            nn::os::SetThreadName(&g_Thread, "NnUi2dViewerToolConnector");
            nn::os::ChangeThreadPriority( &g_Thread, priority );

            nn::os::StartThread( &g_Thread );
            g_IsOK = true;
        }
    }
    else
    {
        g_IsOK = true;
    }

    return g_IsOK;
}

//------------------------------------------------------------------------------
//  破棄
//------------------------------------------------------------------------------
void ToolConnector::Finalize() NN_NOEXCEPT
{
    if( g_IsOK )
    {
        g_IsConnected = false;
        g_IsDetermined = false;
        g_IsOK = false;

        if (g_Handle != InvalidSocket)
        {
            nn::ige::HtcsHelperClose(g_Handle);
            g_Handle = InvalidSocket;
        }

        nn::os::DestroyThread(&g_Thread);
    }

    if( g_StackBase != NULL )
    {
        g_pFnFree( g_StackBase, g_pUserDataForAlloc );
        g_StackBase = NULL;
    }

    if( g_pReadBuffer )
    {
        g_pFnFree( g_pReadBuffer, g_pUserDataForAlloc );
        g_pReadBuffer = NULL;
    }

    if( g_pPresentBuffer )
    {
        g_pFnFree( g_pPresentBuffer, g_pUserDataForAlloc );
        g_pPresentBuffer = NULL;
    }
}

uint8_t* ToolConnector::GetPacket(size_t* pReceiveSize) NN_NOEXCEPT
{
    if( g_IsReceived )
    {
        // 公開バッファにまだコピーしていなかったらコピーします。
        memcpy( g_pPresentBuffer, g_pReadBuffer, g_ReadSize);

        // g_ReadSize は 0 終端分を含まないので埋める。
        g_pPresentBuffer[g_ReadSize] = 0;

        if (pReceiveSize != NULL)
        {
            // 受信実サイズが必要であればお知らせします。
            *pReceiveSize = g_ReadSize;
        }

        g_IsReceived = false;
    }
    else
    {
        return NULL;
    }

    return g_pPresentBuffer;
}

bool ToolConnector::SendPacket( void *pSendBuffer, size_t sendSize ) NN_NOEXCEPT
{
    if( g_Handle != InvalidSocket )
    {
        return SendPacketInternal( g_Handle, pSendBuffer, sendSize );
    }

    return false;
}

bool ToolConnector::SendPacketInternal( int param, void *pSendBuffer, size_t sendSize ) NN_NOEXCEPT
{
    const int sock = param;

    int len = static_cast< int >( nn::ige::HtcsHelperSend(sock, pSendBuffer, sendSize, 0) );
    if( len == InvalidSocket )
    {
        NN_DETAIL_UI2D_ERROR( "ERROR: Send Packet error %d\n", nn::ige::HtcsHelperGetLastError() );
        return false;
    }

    return true;
}

uint32_t ToUint32LittleEndian(uint8_t* bytes)
{
    return static_cast<uint32_t>(bytes[0])
        | static_cast<uint32_t>(bytes[1]) << 8
        | static_cast<uint32_t>(bytes[2]) << 16
        | static_cast<uint32_t>(bytes[3]) << 24;
}

bool ToolConnector::ReadPacket( int param ) NN_NOEXCEPT
{
    // ヘッダ分読み込み
    size_t readSize = ReadPacketInternal( param, g_pReadBuffer, 4 );

    // 受信データがない場合はfalseとして終了
    if( readSize == 0 )
    {
        return false;
    }

    // 内容のサイズ
    size_t dataSize = ToUint32LittleEndian(g_pReadBuffer);

    if (dataSize > g_ReadBufferSize)
    {
        // バッファサイズを越えるときは false として終了
        return false;
    }

    // 内容の読み込み
    readSize = ReadPacketInternal(param, g_pReadBuffer, dataSize);

    // 受信データがない場合はfalseとして終了
    if (readSize == 0)
    {
        return false;
    }

    g_ReadSize = readSize;

    return true;
}

uint32_t ToolConnector::ReadPacketInternal(int handle, uint8_t *pReceiveBuffer, size_t receiveSize) NN_NOEXCEPT
{
    const int sock = handle;
    size_t receivedSize = 0;
    while (receiveSize > receivedSize)
    {
        int status = static_cast<int>(nn::ige::HtcsHelperRecv(sock, pReceiveBuffer + receivedSize, receiveSize - receivedSize, 0));
        if (status == InvalidSocket)
        {
            NN_DETAIL_UI2D_ERROR("Error in recv socket %d\n", nn::ige::HtcsHelperGetLastError());
            return 0;
        }
        else if (status == 0)
        {
            NN_DETAIL_UI2D_ERROR("Connenction closed %d\n", nn::ige::HtcsHelperGetLastError());
            return 0;
        }

        receivedSize += status;
    }

    return static_cast< uint32_t >( receivedSize );
}

bool OpenSocket( int *sock0, int *sock ) NN_NOEXCEPT
{
    // ソケットの作成
    *sock0 = nn::ige::HtcsHelperSocket();
    if( *sock0 == InvalidSocket )
    {
        return false;
    }

    // ソケットの設定
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::ige::HtcsHelperGetPeerNameAny();

    for( int i = 0; i < sizeof(STR_HEADER) / sizeof(char); ++i )
    {
        addr.portName.name[i] = STR_HEADER[i];
    }

    // Bindしつつ、Accept待ち時にスレッドが死ねるようにノンブロッキングにする
    nn::ige::HtcsHelperBind( *sock0, &addr );
    nn::ige::HtcsHelperFcntl( *sock0, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK );

    // TCPクライアントからの接続要求を待てる状態にする
    int ret = nn::ige::HtcsHelperListen( *sock0, 5 );
    if( ret == InvalidSocket )
    {
        return false;
    }

    // TCPクライアントからの接続要求を受け付ける
    NN_DETAIL_UI2D_INFO( "Waiting for connect...\n" );
    while( g_IsDetermined )
    {
        *sock = nn::ige::HtcsHelperAccept( *sock0, NULL );
        if (*sock != InvalidSocket)
        {
            nn::ige::HtcsHelperFcntl( *sock, nn::htcs::HTCS_F_SETFL, 0 );
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }

    if( *sock == InvalidSocket )
    {
        NN_DETAIL_UI2D_ERROR( "Socket open error %d\n", nn::ige::HtcsHelperGetLastError() );
        *sock0 = InvalidSocket;
        return false;
    }

    g_Handle = *sock;
    NN_DETAIL_UI2D_INFO( "Server start \n" );
    return true;
}

int ToolConnector::ThreadToolReceiver( int intArg, void *ptrArg ) NN_NOEXCEPT
{
    NN_UNUSED( intArg );
    NN_UNUSED( ptrArg );

    int sock0 = InvalidSocket;
    int sock = InvalidSocket;
    g_IsConnected = false;

    while( g_IsDetermined )
    {
        // 接続試行処理
        if( g_IsConnected == false )
        {
            if( sock0 != InvalidSocket )
            {
                // このクローズでDaemonからの登録を解除する
                nn::ige::HtcsHelperClose( sock0 );
                sock0 = InvalidSocket;
            }

            g_IsConnected = OpenSocket( &sock0, &sock );
            if( g_IsConnected )
            {
                g_Handle = sock;
            }
            else
            {
                // スリープしながら試行を続ける
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                continue;
            }
        }

        // 通信処理
        if( g_IsConnected == true )
        {
            if( g_IsReceived )
            {
                // 消費されるまでまつ
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 10 ) );
                continue;
            }

            // 受信と受信の通知
            uint8_t ack[1] = {};
            if( !ReadPacket( sock ) || !SendPacket(ack, 1))
            {
                if( nn::ige::HtcsHelperClose(sock) == InvalidSocket )
                {
                    NN_DETAIL_UI2D_ERROR( "ERROR: Can't socket close %d\n", nn::ige::HtcsHelperGetLastError() );
                }

                g_IsConnected = false;
                g_Handle = InvalidSocket;

                // 受信に失敗したので g_IsReceived は false のまま変えない
            }
            else
            {
                g_IsReceived = true;
            }
        }
    }

    // TCPセッションの終了
    if (sock != InvalidSocket)
    {
        nn::ige::HtcsHelperClose(sock);
        nn::ige::HtcsHelperShutdown( sock, nn::htcs::HTCS_SHUT_RDWR );
    }

    NN_DETAIL_UI2D_INFO( "Server stop\n" );

    return 0;
}// NOLINT(readability/fn_size)

} // namespace viewer
} // namespace ui2d
} // namespace nn

