﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <cstdlib>

#include <nn/util/util_PlacementArray.h>
#include <nn/util/util_StringUtil.h>
#include <nn/ui2d/viewer/ui2d_AnimationManager2.h>
#include <nn/ui2d/viewer/ui2d_AllPaneAnimator.h>

#if defined(NN_UI2D_VIEWER_ENABLED)

namespace nn
{
namespace ui2d
{
namespace viewer
{

#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(push)
#pragma warning(disable:4351)
#endif
AnimationManager2::AnimationManager2()
: m_Layout(NULL)
, m_ResourceAccessor(NULL)
, m_AnimationCount(0)
, m_CurrentAnimationNo(0)
, m_CurrentAnimationName()
, m_TargetMode(TargetMode_Pane)
, m_CurrentTargetNo(0)
, m_CurrentTargetName("")
, m_Animator(NULL)
, m_LayoutNameLength(0)
, mAnimInfoArray()
, mAnimInfoArrayCount(0)
{
}
#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(pop)
#endif

void AnimationManager2::Setup(
    nn::gfx::Device* pDevice,
    nn::ui2d::Layout* layout,
    nn::ui2d::IFindableResourceAccessor* resourceAccessor
)
{
    m_Layout = layout;
    m_ResourceAccessor = resourceAccessor;

    const char* layoutName = layout->GetName();
    const char* animFileName = NULL;

    m_LayoutNameLength = static_cast<int32_t >(strlen(layoutName));

    memset(mAnimInfoArray, 0, sizeof(AnimInfo) * mAnimInfoArrayCount);
    mAnimInfoArrayCount = 0;

    const char* ext = ".bflan";
    const char separator = '_';

    do {
        animFileName = resourceAccessor->FindFile(nn::ui2d::ResourceTypeAnimation, animFileName);
        if (animFileName != NULL)
        {
            // 先頭がレイアウト名と一致しているのが候補
            if (nn::util::Strncmp(animFileName, layoutName, m_LayoutNameLength) == 0)
            {
                nn::ui2d::AnimResource animRes(resourceAccessor->FindData(NULL, nn::ui2d::ResourceTypeAnimation, animFileName));

                // ファイル名とタグ名から関連付けられているかを判定する
                bool isRelated = false;

                // NULL と空文字列のどちらもありえる
                if (animRes.GetTagName() == NULL || animRes.GetTagName()[0] == 0)
                {
                    // 全体のアニメーション
                    isRelated = std::strcmp(animFileName + m_LayoutNameLength, ext) == 0;
                }
                else
                {
                    // タグごとのアニメーション
                    const char* pTagNameHead = animFileName + m_LayoutNameLength + 1;
                    const int tagNameLength = static_cast<int32_t>(strlen(animRes.GetTagName()));
                    isRelated = animFileName[m_LayoutNameLength] == separator
                        && nn::util::Strncmp(pTagNameHead, animRes.GetTagName(), tagNameLength) == 0
                        && std::strcmp(pTagNameHead + tagNameLength, ext) == 0;
                }

                if (isRelated)
                {
                    // 現在プレビューしているレイアウトに属しているアニメーション
                    AnimInfo& info = mAnimInfoArray[mAnimInfoArrayCount];
                    mAnimInfoArrayCount++;

                    nn::util::Strlcpy(info.name, animFileName, AnimFileNameMax);

                    info.tagOrder = animRes.GetTagOrder();
                    if (info.tagOrder == 0xffff)
                    {
                        // AllアニメーションのtagOrderは0xffffになる。これは先頭にする。
                        info.tagOrder = -1;
                    }

                    if (mAnimInfoArrayCount == AnimCountMax)
                    {
                        break;
                    }
                }
            }
        }
    } while (animFileName != NULL);

    m_AnimationCount = mAnimInfoArrayCount;

    // 区間タグの通し番号順にソートする。
    qsort(mAnimInfoArray, mAnimInfoArrayCount, sizeof(AnimInfo), AnimInfo::CompareAnimationInfo);

    if (GetAnimationCount() > 0)
    {
        SetCurrentAnimationNo(pDevice, 0);
    }
}

void AnimationManager2::Finalize()
{
    if (m_Animator)
    {
        m_Animator->Unbind();
        m_Layout->DeleteAnimTransform(m_Animator);
        m_Animator = NULL;
    }
    m_Layout = NULL;
    m_ResourceAccessor = NULL;
    m_AnimationCount = 0;
    m_CurrentAnimationNo = 0;
    m_CurrentTargetNo = 0;
    m_CurrentTargetName = "";
    m_LayoutNameLength = 0;
    mAnimInfoArrayCount = 0;
}

int  AnimationManager2::GetAnimationCount() const
{
    if (m_AnimationCount > 0)
    {
        if (m_TargetMode == TargetMode_Group)
        {
            return m_AnimationCount - 1;
        }
        else
        {
            return m_AnimationCount;
        }
    }
    else
    {
        return 0;
    }
}

int  AnimationManager2::GetCurrentAnimationNo() const
{
    if (m_AnimationCount > 0)
    {
        if (m_TargetMode == TargetMode_Group)
        {
            return m_CurrentAnimationNo - 1;
        }
        else
        {
            return m_CurrentAnimationNo;
        }
    }
    else
    {
        return 0;
    }
}

void AnimationManager2::SetTargetMode(nn::gfx::Device* pDevice, TargetMode mode)
{
    m_TargetMode = mode;
    if (GetAnimationCount() > 0)
    {
        SetCurrentAnimationNo(pDevice, 0);
    }
}

void AnimationManager2::SetCurrentAnimationNo(nn::gfx::Device* pDevice, int  no)
{
    if (no < GetAnimationCount())
    {
        if (m_TargetMode == TargetMode_Pane)
        {
            m_CurrentAnimationNo = no;
        }
        else
        {
            m_CurrentAnimationNo = no + 1;
        }
        if (m_CurrentAnimationNo == 0)
        {
            nn::util::Strlcpy(m_CurrentAnimationName, "All", nn::ui2d::AnimTagNameStrMax + 1);
        }
        else
        {
            this->GetAnimTagNameFromAnimFileName(m_CurrentAnimationName, mAnimInfoArray[m_CurrentAnimationNo].name);
        }
        m_CurrentAnimResource.Set(m_ResourceAccessor->FindData(NULL, nn::ui2d::ResourceTypeAnimation, mAnimInfoArray[m_CurrentAnimationNo].name));
        SetCurrentTargetNo(pDevice, 0);
    }
    else
    {
        NN_SDK_ASSERT(false, "no[%d] exceeds animation num[%d]", no, GetAnimationCount());
    }
}

int  AnimationManager2::GetCurrentTargetCount()
{
    if (m_Animator)
    {
        if (m_TargetMode == TargetMode_Pane)
        {
            return m_CurrentAnimResource.GetResourceBlock()->animContCount + 1;
        }
        else
        {
            return m_CurrentAnimResource.GetGroupCount() + 1;
        }
    }
    else
    {
        return 0;
    }
}

void AnimationManager2::SetCurrentTargetNo(nn::gfx::Device* pDevice, int  no)
{
    m_CurrentTargetNo = no;
    if (m_Animator)
    {
        m_Animator->Unbind();
        m_Layout->DeleteAnimTransform(m_Animator);
        m_Animator = NULL;
    }
    if (m_TargetMode == TargetMode_Pane)
    {
        const nn::ui2d::ResAnimationBlock* pRes = m_CurrentAnimResource.GetResourceBlock();
        if (no <= pRes->animContCount)
        {
            if (no == 0)
            {
                m_CurrentTargetName = "All";
                viewer::AllPaneAnimator* animator = m_Layout->CreateAnimTransform<viewer::AllPaneAnimator>(pDevice, m_CurrentAnimResource);
                animator->Setup(m_Layout, true);
                m_Animator = animator;
            }
            else
            {
                const uint32_t *const animContOffsets = nn::util::ConstBytePtr(pRes, pRes->animContOffsetsOffset).Get<uint32_t>();
                const nn::ui2d::ResAnimationContent& animCont = *nn::util::ConstBytePtr(pRes, animContOffsets[no - 1]).Get<nn::ui2d::ResAnimationContent>();
                m_CurrentTargetName = animCont.name;
                if (m_CurrentAnimationNo == 0)
                {
                    nn::ui2d::PaneAnimator* animator = m_Layout->CreateAnimTransform<nn::ui2d::PaneAnimator>(pDevice, m_CurrentAnimResource);
                    animator->Setup(m_Layout->GetRootPane()->FindPaneByName(m_CurrentTargetName), true);
                    m_Animator = animator;
                }
                else
                {
                    m_Animator = m_Layout->CreatePaneAnimator(pDevice, m_CurrentAnimationName, m_CurrentTargetName, true);
                }
            }
        }
        else
        {
            NN_SDK_ASSERT(false, "no[%d] exceeds anim cont num[%d]", no, static_cast<int>(m_CurrentAnimResource.GetResourceBlock()->animContCount));
        }
    }
    else
    {
        if (no <= m_CurrentAnimResource.GetGroupCount())
        {
            if (no == 0)
            {
                m_CurrentTargetName = "All";
                if (m_CurrentAnimResource.GetGroupCount() == 0)
                {
                    // グループが関連付けられていない場合はAllPaneAnimatorで再生する
                    viewer::AllPaneAnimator* animator = m_Layout->CreateAnimTransform<viewer::AllPaneAnimator>(pDevice, m_CurrentAnimResource);
                    animator->Setup(m_Layout, true);
                    m_Animator = animator;
                }
                else
                {
                    // グループが関連付けられていればGroupArrayAnimatorで再生する
                    char name[nn::ui2d::AnimTagNameStrMax + 1];
                    GetAnimTagNameFromAnimFileName(name, mAnimInfoArray[m_CurrentAnimationNo].name);
                    m_Animator = m_Layout->CreateGroupArrayAnimator(pDevice, name, true);
                }
            }
            else
            {
                m_CurrentTargetName = m_CurrentAnimResource.GetGroupArray()[no - 1].GetName();
                m_Animator = m_Layout->CreateGroupAnimator(pDevice, m_CurrentAnimationName, m_CurrentTargetName, true);
            }
        }
        else
        {
            NN_SDK_ASSERT(false, "no[%d] exceeds group num[%d]", no, m_CurrentAnimResource.GetGroupCount());
        }
    }
}

bool AnimationManager2::IsCurrentAnimationLoop() const
{
    if (m_Animator)
    {
        const nn::ui2d::ResAnimationBlock* pRes = m_CurrentAnimResource.GetResourceBlock();
        return (pRes != NULL && pRes->loop == 1);
    }
    else
    {
        return false;
    }
}

void AnimationManager2::StartAnimation()
{
    if (m_Animator)
    {
        if (m_Animator->IsMaxFrame())
        {
            m_Animator->Play(IsCurrentAnimationLoop() ? nn::ui2d::Animator::PlayType_Loop : nn::ui2d::Animator::PlayType_OneShot, 1.f);
        }
        else
        {
            m_Animator->PlayFromCurrent(IsCurrentAnimationLoop() ? nn::ui2d::Animator::PlayType_Loop : nn::ui2d::Animator::PlayType_OneShot, 1.f);
        }
    }
}

void AnimationManager2::StopAnimation()
{
    if (m_Animator)
    {
        m_Animator->StopAtCurrentFrame();
    }
}

void AnimationManager2::ResetAnimation()
{
    if (m_Animator)
    {
        m_Animator->StopAtStartFrame();
    }
}

bool AnimationManager2::IsAnimationPlaying() const
{
    if (m_Animator)
    {
        return m_Animator->IsPlaying();
    }
    else
    {
        return false;
    }
}

float AnimationManager2::GetCurrentAnimationFrame() const
{
    if (m_Animator)
    {
        return m_Animator->GetFrame();
    }
    else
    {
        return 0;
    }
}

void AnimationManager2::SetCurrentAnimationFrame(float frame)
{
    if (m_Animator)
    {
        m_Animator->SetFrame(frame);
    }
}

float AnimationManager2::GetAnimationFrameMax() const
{
    if (m_Animator)
    {
        return m_Animator->GetFrameMax();
    }
    else
    {
        return 0;
    }
}

int AnimationManager2::GetAnimationNoByTagName(const char * pName) const
{
    int i;
    for (i = 0; i < mAnimInfoArrayCount; i++)
    {
        if (i == 0)
        {
            // 0 番目はタグ名がリソース名から取得できないのでスキップ
            continue;
        }

        const char * pSubStrStart = mAnimInfoArray[i].name + (m_LayoutNameLength + 1);
        const int AnimFileExtentionLength = 6;
        int subLength = std::min(static_cast<int>(std::strlen(pSubStrStart) - AnimFileExtentionLength), nn::ui2d::AnimTagNameStrMax);
        if (nn::util::Strncmp(pSubStrStart, pName, subLength) == 0 && pName[subLength] == 0)
        {
            break;
        }
    }

    return i;
}

void AnimationManager2::GetAnimTagNameFromAnimFileName(char* dst, const char* src)
{
    NN_SDK_ASSERT_NOT_NULL(dst);

    // NULL 終端文字をコピーしないので、いったんバッファをゼロクリアしておきます。
    memset(dst, 0, nn::ui2d::AnimTagNameStrMax + 1);

    const char* pSubStrStart = src + ( m_LayoutNameLength + 1 );
    const int AnimFileExtentionLength = 6;

    std::memcpy( dst, pSubStrStart, std::min( static_cast<int>( std::strlen( pSubStrStart ) - AnimFileExtentionLength ), nn::ui2d::AnimTagNameStrMax ) );
}

} // namespace viewer
} // namespace ui2d
} // namespace nn

#endif // NN_UI2D_VIEWER_ENABLED
