﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_Util.h>
#include <nn/ui2d/ui2d_Bounding.h>
#include <nn/ui2d/ui2d_Layout.h>
#include <nn/ui2d/ui2d_Group.h>
#include <nn/ui2d/ui2d_Animation.h>
#include <nn/ui2d/ui2d_Resources.h>
#include <nn/ui2d/ui2d_Parts.h>
#include <nn/ui2d/ui2d_Picture.h>
#include <nn/ui2d/ui2d_TextBox.h>
#include <nn/ui2d/ui2d_Window.h>
#include <nn/ui2d/ui2d_Capture.h>
#include <nn/ui2d/ui2d_Alignment.h>
#include <nn/ui2d/ui2d_Scissor.h>
#include <nn/ui2d/ui2d_ResourceAccessor.h>
#include <nn/util/util_Matrix.h>
#include <nn/font/font_ScalableFont.h>
#include <nn/font/font_PairFont.h>
#include <nn/font/font_ResFont.h>

#include <random>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#ifdef _WIN32
#ifndef WIN32
#define WIN32
#endif
#endif
#pragma warning(push)
#pragma warning(disable:4100)
#pragma warning(disable:4063)
#pragma warning(pop)
#endif

namespace
{

bool
Contains(
    const nn::font::Rectangle&  rect,
    const nn::util::Float2&       point
)
{
    return rect.left <= point.v[0] && point.v[0] <= rect.right && rect.bottom <= point.v[1] && point.v[1] <= rect.top;
}

struct TexSpec
{
    nn::gfx::ImageFormat gfxFormat;
    nn::gfx::ChannelMapping channelMappingRed;
    nn::gfx::ChannelMapping channelMappingGreen;
    nn::gfx::ChannelMapping channelMappingBlue;
    nn::gfx::ChannelMapping channelMappingAlpha;
};

bool IsRawFont(const void* pFontData)
{
    // 先頭 4byte がビッグエンディアンで「00 01 00 00」「4F 54 54 4F」「74 74 63 66」の
    // いずれかであれば、TrueType もしくは OpenType フォントのバイナリです。
    const uint32_t header = *static_cast<const uint32_t*>(pFontData);
    return header == 0x00000100 || header == 0x4f54544f || header == 0x66637474;
}

void SetResScalableFontDescriptionToArg(nn::font::TextureCache::InitializeArg* pArg, int fontFace, uint32_t scalableFontDescription, const nn::font::ResScalableFontDescription* pSource, uint32_t version)
{
    bool isOldResource9_0 = version <= 0x09000000;

    pArg->scaleWidths[fontFace][scalableFontDescription] = pSource->scaleWidth;
    pArg->scaleHeights[fontFace][scalableFontDescription] = isOldResource9_0 ? 1.0f : pSource->scaleHeight;
    pArg->ignorePalt[fontFace][scalableFontDescription] = pSource->bits.ignorePalt;
    pArg->deleteBearingX[fontFace][scalableFontDescription] = pSource->bits.deleteBearingX;
    pArg->bearingOffsetX[fontFace][scalableFontDescription] = pSource->bearingOffsetX;
    pArg->forceMonospacedEnabled[fontFace][scalableFontDescription] = pSource->bits.forceMonospacedEnabled;
    pArg->forceMonospacedWidth[fontFace][scalableFontDescription] = pSource->forceMonospacedWidth;
    pArg->baselineOffset[fontFace][scalableFontDescription] = static_cast<int>(pSource->baselineOffset);
}

void SetResScalableFontDescriptionToArg(nn::font::TextureCache::InitializeArg* pArg, int fontFace, uint32_t scalableFontDescription, const nn::font::ResScalableFontDescriptionOld* pSource, uint32_t version)
{
    NN_UNUSED(pArg);
    NN_UNUSED(fontFace);
    NN_UNUSED(scalableFontDescription);
    NN_UNUSED(pSource);
    NN_UNUSED(version);
}

void SetResMultiScalableFontToArg(nn::font::ScalableFont::InitializeArg* pArg, const nn::font::ResMultiScalableFont* pSource)
{
    pArg->lineFeedOffset = pSource->lineFeedOffset;
    if (pSource->alternateChar != 0)
    {
        pArg->alternateChar = static_cast<uint32_t>(pSource->alternateChar);
    }
}

void SetResMultiScalableFontToArg(nn::font::ScalableFont::InitializeArg* pArg, const nn::font::ResMultiScalableFontOld* pSource)
{
    NN_UNUSED(pArg);
    NN_UNUSED(pSource);
}

} // namespace {no-name}

namespace nn {
namespace ui2d {

//----------------------------------------
bool
LoadTexture(ResourceTextureInfo* pResTextureInfo, nn::gfx::Device* pDevice, const void* pImgRes)
{
    NN_SDK_ASSERT_NOT_NULL(pResTextureInfo);
    NN_SDK_ASSERT_NOT_NULL(pImgRes);

    nn::gfx::ResTexture* pResTexture = const_cast<nn::gfx::ResTexture*>(reinterpret_cast<const nn::gfx::ResTexture*>(pImgRes));
    pResTextureInfo->InitializeFromResource(pDevice, pResTexture);

    return true;
}

void
LoadArchiveShader(ShaderInfo* pShaderInfo, nn::gfx::Device* pDevice, void* pShaderRes, const void* pVariationTable,
    nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize, int textureSlotCount)
{
    pShaderInfo->InitializeWithVariationTable(pDevice, pShaderRes, pVariationTable, pMemoryPool, memoryPoolOffset, memoryPoolSize, textureSlotCount);
}

void
FreeArchiveShader(nn::gfx::Device* pDevice, ShaderInfo* pShaderInfo)
{
    pShaderInfo->Finalize(pDevice, false);
}

void ConvertBlendsToArchiveShaderName(char* pName, int firstBlend, int secondBlend)
{
    NN_SDK_ASSERT(0 <= firstBlend && firstBlend <= 999, "name[%s]", pName);
    NN_SDK_ASSERT(0 <= secondBlend && secondBlend <= 999, "name[%s]", pName);

    // "xxx_xxx" の形式に変換する
    int n;

    n = firstBlend / 100;
    pName[0] = static_cast<char>('0' + n);
    firstBlend -= n * 100;

    n = firstBlend / 10;
    pName[1] = static_cast<char>('0' + n);
    firstBlend -= n * 10;

    pName[2] = static_cast<char>('0' + firstBlend);

    pName[3] = '_';

    n = secondBlend / 100;
    pName[4] = static_cast<char>('0' + n);
    secondBlend -= n * 100;

    n = secondBlend / 10;
    pName[5] = static_cast<char>('0' + n);
    secondBlend -= n * 10;

    pName[6] = static_cast<char>('0' + secondBlend);

    pName[7] = '\0';
}

void ConvertStageBitsToArchiveShaderDetailedCombinerName(char* pName, int nameSize, int firstBlend, int secondBlend, int* pStageBits)
{
    NN_UNUSED(nameSize);
    ConvertBlendsToArchiveShaderName(pName, firstBlend, secondBlend);
    const int strLengthMax = (1 + 8) * 4 + 1; // ("_" + "00000000") * 4 + '\0'
    char pStr[strLengthMax];
    size_t nameLength = std::strlen(pName); // pName に格納した文字数
    NN_UNUSED(nameLength);
    for (int i = 0; i < DetailedCombinerStageBitsCountWithVariationTable; i+=4)
    {
        nameLength += std::sprintf(pStr, "_%X_%X_%X_%X", pStageBits[i + 0], pStageBits[i + 1], pStageBits[i + 2], pStageBits[i + 3]);
        NN_SDK_ASSERT(nameLength < static_cast<size_t>(nameSize));
        std::strcat(pName, pStr);
    }
}

bool ConvertArchiveShaderNameToBlends(int* pFirstBlend, int* pSecondBlend, const char* pName)
{
    if (strlen(pName) < 7)
    {
        return false;
    }

    // "xxx_xxx" の形式の文字列を数値に分解する
    int n;

    n = static_cast<int>(pName[0] - '0');
    if (n < 0 || 9 < n)
    {
        return false;
    }
    *pFirstBlend = n * 100;

    n = static_cast<int>(pName[1] - '0');
    if (n < 0 || 9 < n)
    {
        return false;
    }
    *pFirstBlend += n * 10;

    n = static_cast<int>(pName[2] - '0');
    if (n < 0 || 9 < n)
    {
        return false;
    }
    *pFirstBlend += n;

    if (pName[3] != '_')
    {
        return false;
    }

    n = static_cast<int>(pName[4] - '0');
    if (n < 0 || 9 < n)
    {
        return false;
    }
    *pSecondBlend = n * 100;

    n = static_cast<int>(pName[5] - '0');
    if (n < 0 || 9 < n)
    {
        return false;
    }
    *pSecondBlend += n * 10;

    n = static_cast<int>(pName[6] - '0');
    if (n < 0 || 9 < n)
    {
        return false;
    }
    *pSecondBlend += n;

    return true;
}

void ConvertArchiveShaderDetailedCombinerNameToStageBits(int* pStageBits, const char* pName)
{
    const int blendStringLength = 8; // 先頭の "xxx_xxx_" の長さ
    pName += blendStringLength;

    // 詳細コンバイナ用の 16進 を数値化
    for (int i = 0; i < DetailedCombinerStageBitsCountWithVariationTable; i++)
    {
        const int hexStringsCount = 8 + 1; // "00000000" が最大文字数
        char hexString[hexStringsCount];
        memset(hexString, 0, hexStringsCount);
        int count = 0;

        // 文字終端、または '_' までを抜く
        char* pHexString = hexString;
        for (;;)
        {
            *pHexString++ = *pName++;
            count++;
            NN_SDK_ASSERT(count <= hexStringsCount);

            if (*pName == '\0')
            {
                break;
            }
            else if (*pName == '_')
            {
                pName++;
                break;
            }
        }

        uint32_t value = 0;
        uint32_t mul = 1;
        for (int j = count - 1; j >= 0; j--)
        {
            uint32_t code = 0;

            if (hexString[j] >= 'A' && hexString[j] <= 'F')
            {
                // 16 進数の「'A'～'F'」を 10 進数の「10～15」に直す。
                code += 10 + hexString[j] - 'A';
            }
            else
            {
                NN_SDK_ASSERT('0' <= hexString[j] && hexString[j] <= '9', "Wrong format of detailed combiner's archive shader name. [%s]", pName);
                code += hexString[j] - '0';
            }

            value += code * mul;
            mul <<= 4;
        }

        pStageBits[i] = (int)value;
    }
}

int SearchShaderVariationIndexFromTable(const void* pVariationTable, int firstBlend, int secondBlend)
{
    const uint16_t* pTable = static_cast<const uint16_t*>(pVariationTable);
    int num = static_cast<int>(*pTable); // 先頭にバリエーション数が入っている
    pTable++;
    // firstBlend, secondBlend, variationIndex の組み合わせを走査。
    // num の数は少なく、また初期化時に 1 回だけ行われる処理のため、
    // 線形探索でもパフォーマンス的な影響は少ない。
    for (int i = 0; i < num; i++)
    {
        int firstBlendInTable = static_cast<int>(pTable[0]);
        int secondBlendInTable = static_cast<int>(pTable[1]);
        int variationIndexInTable = static_cast<int>(pTable[2]);
        pTable += 3;
        if (firstBlendInTable == firstBlend && secondBlendInTable == secondBlend)
        {
            return variationIndexInTable;
        }
    }
    return -1;
}

int SearchShaderVariationDetailedCombinerIndexFromTable(const void* pVariationTable, int* pStageBits)
{
    const uint32_t* pTable = static_cast<const uint32_t*>(pVariationTable);
    const int variationCount = static_cast<int>(*pTable); // 先頭にバリエーション数が入っている
    pTable++;
    for (int i = 0; i < variationCount; i++)
    {
        const int elementCount = static_cast<int>(pTable[0]);
        pTable++;
        if (elementCount == 0)
        {
            continue;
        }
        const int variationIndexInTable = static_cast<int>(pTable[0]);
        pTable++;

        if (elementCount == DetailedCombinerStageBitsCountWithVariationTable)
        {
            bool match = true; // StageBits がすべて一致するかどうか
            for (int j = 0; j < DetailedCombinerStageBitsCountWithVariationTable; j++)
            {
                int code = static_cast<int>(pTable[j]);
                if (pStageBits[j] != code)
                {
                    match = false;
                    break;
                }
            }
            if (match)
            {
                return variationIndexInTable;
            }
            pTable += elementCount;
        }
    }
    return -1;
}

void
BindAnimation(
    AnimTransform*  pAnimTrans,
    Group*          pGroup,
    bool            bEnable
)
{
    pAnimTrans->BindGroup(pGroup);
    pAnimTrans->SetEnabled(bEnable);
}

void
UnbindAnimation(
    AnimTransform*  pAnimTrans,
    Group*          pGroup
)
{
    pAnimTrans->UnbindGroup(pGroup);
}

bool
IsContain(
    const Pane*               pPane,
    const nn::util::Float2&   pos
)
{
    nn::util::MatrixT4x3fType invGlbMtx;
    nn::util::MatrixInverse(&invGlbMtx, pPane->GetGlobalMtx());       // ペインのグローバル行列の逆行列を求める

    nn::util::Vector3fType pos3;
    nn::util::VectorSet(&pos3, pos.v[0], pos.v[1], 0.f);
    nn::util::Vector3fType lclPos;
    nn::util::VectorTransform(&lclPos, pos3, invGlbMtx);

    const nn::util::Float2 vector = NN_UTIL_FLOAT_2_INITIALIZER(nn::util::VectorGetX(lclPos), nn::util::VectorGetY(lclPos));
    return Contains(pPane->GetPaneRect(), vector);
}

Pane* FindHitPane(Pane* pPane, const nn::util::Float2& pos)
{
    // 非表示のペインはヒットチェックの対象としない。
    if (! pPane->IsVisible())
    {
        return 0;
    }

    // 子供のヒットチェック
    for (PaneList::reverse_iterator it = pPane->GetChildList().rbegin(); it != pPane->GetChildList().rend(); ++it)
    {
        if (Pane *const ret = FindHitPane(&(*it), pos))
        {
            return ret;
        }
    }

    // Bounding ペインか
    if (nn::ui2d::Bounding* const pBounding = DynamicCast<nn::ui2d::Bounding*>(pPane))
    {
        if (IsContain(pBounding, pos))
        {
            return pBounding;  // 含まれている。
        }
    }

    return 0;
}

const Pane* FindHitPane(const Pane* pPane, const nn::util::Float2& pos)
{
    return FindHitPane(const_cast<Pane*>(pPane), pos);
}

Pane* FindHitPane(Layout* pLayout, const nn::util::Float2& pos)
{
    return FindHitPane(pLayout->GetRootPane(), pos);
}

const Pane* FindHitPane(const Layout* pLayout, const nn::util::Float2& pos)
{
    return FindHitPane(const_cast<Layout*>(pLayout), pos);
}

/*
    次のペインを返す。
*/
Pane* GetNextPane(Pane* pPane)
{
    if (! pPane->GetChildList().empty())  // 子供がいれば、最初の子供を返す。
    {
        PaneList::iterator paneIter = pPane->GetChildList().begin();
        return &(*paneIter);
    }

    // 弟を探す。
    // 弟が無ければ、親の弟へとたどる。
    for(;;)
    {
        if (pPane->GetParent() == 0)    // 親が NULL の場合はルートペイン
        {
            return 0;
        }

        PaneList& childList = pPane->GetParent()->GetChildList();
        PaneList::iterator nextIter = childList.iterator_to(*pPane);
        nextIter++;
        PaneList::iterator endIter = pPane->GetParent()->GetChildList().end();
        if (nextIter != endIter)
        {
            return &(*nextIter);
        }

        pPane = pPane->GetParent();
    }
}

Pane*
ClonePaneTree(const Pane* pPane, nn::gfx::Device* pDevice)
{
    return ClonePaneTree(pPane, pDevice, NULL, NULL);
}

Pane*
ClonePaneTree(const Pane* pPane, nn::gfx::Device* pDevice, ResourceAccessor* pResAccessor, const char* pNewRootName)
{
    return ClonePaneTree(pPane, pDevice, pResAccessor, pNewRootName, NULL);
}

Pane*
ClonePaneTree(const Pane* pPane, nn::gfx::Device* pDevice, ResourceAccessor* pResAccessor, const char* pNewRootName, const Layout* pLayout)
{
    Pane* pNewPane = NULL;
    const Picture* pPicture = DynamicCast<const Picture*>(pPane);
    if (pPicture)
    {
        pNewPane = Layout::AllocateAndConstruct<Picture, const Picture&, nn::gfx::Device*, ResourceAccessor*, const char*, const Layout*>(*pPicture, pDevice, pResAccessor, pNewRootName, pLayout, Layout::AlignmentForPaneInstance);
    }
    else
    {
        const TextBox* pTextBox = DynamicCast<const TextBox*>(pPane);
        if (pTextBox)
        {
            pNewPane = Layout::AllocateAndConstruct<TextBox, const TextBox&, nn::gfx::Device*, ResourceAccessor*, const char*, const Layout*>(*pTextBox, pDevice, pResAccessor, pNewRootName, pLayout, Layout::AlignmentForPaneInstance);
        }
        else
        {
            const Window* pWindow = DynamicCast<const Window*>(pPane);
            if (pWindow)
            {
                pNewPane = Layout::AllocateAndConstruct<Window, const Window&, nn::gfx::Device*, ResourceAccessor*, const char*, const Layout*>(*pWindow, pDevice, pResAccessor, pNewRootName, pLayout, Layout::AlignmentForPaneInstance);
            }
            else
            {
                const Bounding* pBounding = DynamicCast<const Bounding*>(pPane);
                if (pBounding)
                {
                    pNewPane = Layout::AllocateAndConstruct<Bounding, const Bounding&>(*pBounding, Layout::AlignmentForPaneInstance);
                }
                else
                {
                    const Capture* pCapture = DynamicCast<const Capture*>(pPane);
                    if (pCapture)
                    {
                        pNewPane = Layout::AllocateAndConstruct<Capture, const Capture&>(*pCapture, Layout::AlignmentForPaneInstance);
                    }
                    else
                    {
                        const Scissor* pScissor = DynamicCast<const Scissor*>(pPane);
                        if (pScissor)
                        {
                            pNewPane = Layout::AllocateAndConstruct<Scissor, const Scissor&>(*pScissor, Layout::AlignmentForPaneInstance);
                        }
                        else
                        {
                            const Alignment* pAlignment = DynamicCast<const Alignment*>(pPane);
                            if (pAlignment)
                            {
                                pNewPane = Layout::AllocateAndConstruct<Alignment, const Alignment&>(*pAlignment, Layout::AlignmentForPaneInstance);
                            }
                            else
                            {
                                const Parts* pParts = DynamicCast<const Parts*>(pPane);
                                if (pParts)
                                {
                                    pNewPane = Layout::AllocateAndConstruct<Parts, const Parts&>(*pParts, Layout::AlignmentForPaneInstance);
                                }
                                else
                                {
                                    pNewPane = Layout::AllocateAndConstruct<Pane, const Pane&>(*pPane, Layout::AlignmentForPaneInstance);
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    for (PaneList::const_iterator iter = pPane->GetChildList().begin(); iter != pPane->GetChildList().end(); ++iter)
    {
        Pane* pChildTree = ClonePaneTree(&(*iter), pDevice, pResAccessor, pNewRootName, pLayout);
        pNewPane->AppendChild(pChildTree);
    }

    return pNewPane;
}

Pane*
ClonePaneTreeWithPartsLayout(const Pane* pPane, Layout* pPartsLayout, nn::gfx::Device* pDevice, Layout* pPartsParentLayout)
{
    return ClonePaneTreeWithPartsLayout(pPane, pPartsLayout, pDevice, pPartsParentLayout, NULL, NULL, NULL);
}

Pane*
ClonePaneTreeWithPartsLayout(const Pane* pPane, Layout* pPartsLayout, nn::gfx::Device* pDevice, Layout* pPartsParentLayout, ResourceAccessor* pResAccessor, const char* pNewRootName)
{
    return ClonePaneTreeWithPartsLayout(pPane, pPartsLayout, pDevice, pPartsParentLayout, pResAccessor, pNewRootName, NULL);
}

Pane*
ClonePaneTreeWithPartsLayout(const Pane* pPane, Layout* pPartsLayout, nn::gfx::Device* pDevice, Layout* pPartsParentLayout, ResourceAccessor* pResAccessor, const char* pNewRootName, const Layout* pLayout)
{
    Pane* pNewPane = NULL;
    bool needCloneChild = true;
    const Picture* pPicture = DynamicCast<const Picture*>(pPane);
    if (pPicture)
    {
        pNewPane = Layout::AllocateAndConstruct<Picture, const Picture&, nn::gfx::Device*, ResourceAccessor*, const char*, const Layout*>(*pPicture, pDevice, pResAccessor, pNewRootName, pLayout, Layout::AlignmentForPaneInstance);
    }
    else
    {
        const TextBox* pTextBox = DynamicCast<const TextBox*>(pPane);
        if (pTextBox)
        {
            pNewPane = Layout::AllocateAndConstruct<TextBox, const TextBox&, nn::gfx::Device*, ResourceAccessor*, const char*, const Layout*>(*pTextBox, pDevice, pResAccessor, pNewRootName, pLayout, Layout::AlignmentForPaneInstance);
        }
        else
        {
            const Window* pWindow = DynamicCast<const Window*>(pPane);
            if (pWindow)
            {
                pNewPane = Layout::AllocateAndConstruct<Window, const Window&, nn::gfx::Device*, ResourceAccessor*, const char*, const Layout*>(*pWindow, pDevice, pResAccessor, pNewRootName, pLayout, Layout::AlignmentForPaneInstance);
            }
            else
            {
                const Bounding* pBounding = DynamicCast<const Bounding*>(pPane);
                if (pBounding)
                {
                    pNewPane = Layout::AllocateAndConstruct<Bounding, const Bounding&>(*pBounding, Layout::AlignmentForPaneInstance);
                }
                else
                {
                    const Capture* pCapture = DynamicCast<const Capture*>(pPane);
                    if (pCapture)
                    {
                        pNewPane = Layout::AllocateAndConstruct<Capture, const Capture&>(*pCapture, Layout::AlignmentForPaneInstance);
                    }
                    else
                    {
                        const Scissor* pScissor = DynamicCast<const Scissor*>(pPane);
                        if (pScissor)
                        {
                            pNewPane = Layout::AllocateAndConstruct<Scissor, const Scissor&>(*pScissor, Layout::AlignmentForPaneInstance);
                        }
                        else
                        {
                            const Parts* pParts = DynamicCast<const Parts*>(pPane);
                            if (pParts)
                            {
                                if (pPartsLayout)
                                {
                                    // 部品レイアウトが指定されている場合は、既に構築済みのレイアウトがあるので、
                                    // 部品ペインを作成して、レイアウトをセットする。
                                    Parts* pNewParts = Layout::AllocateAndConstruct<Parts, const Parts&>(*pParts, Layout::AlignmentForPaneInstance);
                                    pNewParts->SetLayout(pPartsLayout);
                                    pNewPane = pNewParts;
                                }
                                else
                                {
                                    // 部品レイアウトが指定されていない場合は、まだ部品レイアウトが作られていないので、この場でコピーする。
                                    // コピーしたレイアウトはルートペインが部品ペインになる。レイアウトをコピーするとペインツリーごとコピー
                                    // されるので、子供をコピーする必要はない。
                                    Layout* pCopiedPartsLayout = Layout::AllocateAndConstruct<Layout, nn::gfx::Device*, const Layout&, Layout*, const char*, const char*>(pDevice, *pParts->GetLayout(), pPartsParentLayout, NULL, pNewRootName, Layout::AlignmentForPaneInstance);
                                    static_cast<Parts*>(pCopiedPartsLayout->GetRootPane())->SetLayout(pCopiedPartsLayout);
                                    pNewPane = pCopiedPartsLayout->GetRootPane();
                                    needCloneChild = false;
                                }
                            }
                            else
                            {
                                const Alignment* pAlignment = DynamicCast<const Alignment*>(pPane);
                                if (pAlignment)
                                {
                                    pNewPane = Layout::AllocateAndConstruct<Alignment, const Alignment&>(*pAlignment, Layout::AlignmentForPaneInstance);
                                }
                                else
                                {
                                    pNewPane = Layout::AllocateAndConstruct<Pane, const Pane&>(*pPane, Layout::AlignmentForPaneInstance);
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    if (needCloneChild) {
        for (PaneList::const_iterator iter = pPane->GetChildList().begin(); iter != pPane->GetChildList().end(); ++iter)
        {
            Pane* pChildTree = ClonePaneTreeWithPartsLayout(&(*iter), NULL, pDevice, pPartsParentLayout, pResAccessor, pNewRootName, pLayout);
            pNewPane->AppendChild(pChildTree);
        }
    }

    return pNewPane;
}

bool
ComparePaneTreeTest(const Pane* pPaneLhs, const Pane* pPaneRhs)
{
    const Picture* pPictureLhs = DynamicCast<const Picture*>(pPaneLhs);
    if (pPictureLhs)
    {
        const Picture* pPictureRhs = DynamicCast<const Picture*>(pPaneRhs);
        if (pPictureRhs == NULL ||
            pPictureLhs->CompareCopiedInstanceTest(*pPictureRhs) == false)
        {
            return false;
        }
    }
    else
    {
        const TextBox* pTextBoxLhs = DynamicCast<const TextBox*>(pPaneLhs);
        if (pTextBoxLhs)
        {
            const TextBox* pTextBoxRhs = DynamicCast<const TextBox*>(pPaneRhs);
            if (pTextBoxRhs == NULL ||
                pTextBoxLhs->CompareCopiedInstanceTest(*pTextBoxRhs) == false)
            {
                return false;
            }
        }
        else
        {
            const Window* pWindowLhs = DynamicCast<const Window*>(pPaneLhs);
            if (pWindowLhs)
            {
                const Window* pWindowRhs = DynamicCast<const Window*>(pPaneRhs);
                if (pWindowRhs == NULL ||
                    pWindowLhs->CompareCopiedInstanceTest(*pWindowRhs) == false)
                {
                    return false;
                }
            }
            else
            {
                const Bounding* pBoundingLhs = DynamicCast<const Bounding*>(pPaneLhs);
                if (pBoundingLhs)
                {
                    const Bounding* pBoundingRhs = DynamicCast<const Bounding*>(pPaneRhs);
                    if (pBoundingRhs == NULL ||
                        pBoundingLhs->CompareCopiedInstanceTest(*pBoundingRhs) == false)
                    {
                        return false;
                    }
                }
                else
                {
                    const Capture* pCaptureLhs = DynamicCast<const Capture*>(pPaneLhs);
                    if (pCaptureLhs)
                    {
                        const Capture* pCaptureRhs = DynamicCast<const Capture*>(pPaneRhs);
                        if (pCaptureRhs == NULL ||
                            pCaptureLhs->CompareCopiedInstanceTest(*pCaptureRhs) == false)
                        {
                            return false;
                        }
                    }
                    else
                    {

                        const Parts* pPartsLhs = DynamicCast<const Parts*>(pPaneLhs);
                        if (pPartsLhs)
                        {
                            const Parts* pPartsRhs = DynamicCast<const Parts*>(pPaneRhs);
                            if (pPartsRhs == NULL ||
                                pPartsLhs->CompareCopiedInstanceTest(*pPartsRhs) == false)
                            {
                                return false;
                            }
                        }
                        else
                        {
                            const Alignment* pAlignmentLhs = DynamicCast<const Alignment*>(pPaneLhs);
                            if (pAlignmentLhs)
                            {
                                const Alignment* pAlignmentRhs = DynamicCast<const Alignment*>(pPaneRhs);
                                if (pAlignmentRhs == NULL ||
                                    pAlignmentLhs->CompareCopiedInstanceTest(*pAlignmentRhs) == false)
                                {
                                    return false;
                                }
                            }
                            else
                            {
                                const Pane* pLhs = DynamicCast<const Pane*>(pPaneLhs);
                                if (pLhs)
                                {
                                    const Pane* pRhs = DynamicCast<const Pane*>(pPaneRhs);
                                    if (pRhs == NULL ||
                                        pLhs->CompareCopiedInstanceTest(*pRhs) == false)
                                    {
                                        return false;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    if (pPaneLhs->GetChildList().size() != pPaneRhs->GetChildList().size())
    {
        return false;
    }

    PaneList::const_iterator iterLhs = pPaneLhs->GetChildList().begin();
    PaneList::const_iterator iterRhs = pPaneRhs->GetChildList().begin();
    PaneList::const_iterator endIterLhs = pPaneLhs->GetChildList().end();
    for (; iterLhs != endIterLhs; ++iterLhs, ++iterRhs)
    {
        const Pane*   pLhs = &(*iterLhs);
        const Pane*   pRhs = &(*iterRhs);
        if (ComparePaneTreeTest(pLhs, pRhs) == false)
        {
            return false;
        }
    }

    return true;
}// NOLINT(impl/function_size)

void DrawNullAndBoundingPane(nn::gfx::CommandBuffer& commandBuffer, DrawInfo& drawInfo, Material& material, const Pane* pPane, const nn::util::Unorm8x4& nullColor, const nn::util::Unorm8x4& boundingColor)
{
    // ルートペインかどうかをチェック
    if (pPane->GetParent())
    {
        if (IsTypeOf<Pane>(pPane))
        {
            drawInfo.SetModelViewMtx(pPane->GetGlobalMtx());

            nn::font::Rectangle rect = pPane->GetPaneRect();
            const nn::util::Float2 vector = NN_UTIL_FLOAT_2_INITIALIZER(rect.left, rect.top);
            detail::DrawBox(commandBuffer, drawInfo, material, vector, pPane->GetSize(), nullColor);

            drawInfo.ResetVertexBufferState();
        }
        else if (IsTypeOf<Bounding>(pPane))
        {
            drawInfo.SetModelViewMtx(pPane->GetGlobalMtx());

            nn::font::Rectangle rect = pPane->GetPaneRect();
            const nn::util::Float2 vector = NN_UTIL_FLOAT_2_INITIALIZER(rect.left, rect.top);
            detail::DrawBox(commandBuffer, drawInfo, material, vector, pPane->GetSize(), boundingColor);

            drawInfo.ResetVertexBufferState();
        }
    }

    PaneList::const_iterator endIter = pPane->GetChildList().end();
    for (PaneList::const_iterator iter = pPane->GetChildList().begin(); iter != endIter; ++iter)
    {
        DrawNullAndBoundingPane(commandBuffer, drawInfo, material, &(*iter), nullColor, boundingColor);
    }
}

float GetHermiteCurveValue(float frame, const ResHermiteKey*  pKeys, int  keySize)
{
    NN_SDK_ASSERT(keySize > 0, "out of bounds: keySize[%u] > 0", keySize);

    // outside of curve
    if (keySize == 1 || frame <= pKeys[0].frame)
    {
        return pKeys[0].value;
    }
    else if (frame >= pKeys[keySize - 1].frame)
    {
        return pKeys[keySize - 1].value;
    }

    // find 2 keys
    const uint32_t  ikeyEnd = keySize - 1;
    uint32_t  ikeyL = 0;
    uint32_t  ikeyR = ikeyEnd;
    while (ikeyL != ikeyR - 1 && ikeyL != ikeyR)
    {
        int ikeyCenter = static_cast<int>(ikeyL + ikeyR) / 2;
        if (frame <= pKeys[ikeyCenter].frame)
        {
            ikeyR = static_cast<uint32_t >(ikeyCenter);
        }
        else
        {
            ikeyL = static_cast<uint32_t >(ikeyCenter);
        }
    }

    // calculate hermite interpolation
    const ResHermiteKey& key0 = pKeys[ikeyL];
    const ResHermiteKey& key1 = pKeys[ikeyR];
    const float R_FRAME_TOLERANCE = 0.001F;
    const float diff = frame - key1.frame;
    if (-R_FRAME_TOLERANCE < diff && diff < R_FRAME_TOLERANCE)
    {
        if (ikeyR < ikeyEnd &&
            key1.frame == pKeys[ikeyR + 1].frame)
        {
            // 同一フレームに値の異なるキーがある場合
            return pKeys[ikeyR + 1].value;
        }
        else
        {
            return key1.value;
        }
    }
    float t1 = frame - key0.frame;
    float t2 = 1.0F / (key1.frame - key0.frame);
    float v0 = key0.value;
    float v1 = key1.value;
    float s0 = key0.slope;
    float s1 = key1.slope;

    float t1t1t2 = t1 * t1 * t2;
    float t1t1t2t2 = t1t1t2 * t2;
    float t1t1t1t2t2 = t1 * t1t1t2t2;
    float t1t1t1t2t2t2 = t1t1t1t2t2 * t2;

    return v0 * (2.0F * t1t1t1t2t2t2 - 3.0F * t1t1t2t2 + 1.0F)
        + v1 * (-2.0F * t1t1t1t2t2t2 + 3.0F * t1t1t2t2)
        + s0 * (t1t1t1t2t2 - 2.0F * t1t1t2 + t1)
        + s1 * (t1t1t1t2t2 - t1t1t2);
}

//-----------------------------------------------------------------------------
float GetParameterizedAnimValue(float frame, float current, const ResParameterizedAnim* pParameterizedAnim)
{
    const uint16_t count = pParameterizedAnim->parameterizedAnimCount;
    for (uint16_t i = 0; i < count; i++)
    {
        const uint32_t* pOffset = nn::util::ConstBytePtr(pParameterizedAnim, sizeof(*pParameterizedAnim) + sizeof(uint32_t) * i).Get<uint32_t>();
        const nn::ui2d::ResParameterizedAnimParameter* pParameter = nn::util::ConstBytePtr(pParameterizedAnim, *pOffset).Get<nn::ui2d::ResParameterizedAnimParameter>();
        if (!IsFrameInRange(frame, pParameter->offset, pParameter->duration))
        {
            continue;
        }

        return GetParameterizedAnimValueAtFrame(frame, pParameter);
    }

    // アニメーションの範囲外なら現在の値をそのまま返す
    return current;
}

//-----------------------------------------------------------------------------
float GetParameterizedAnimValueAtFrame_(float normalizedFrame, const ResParameterizedAnimParameter* pParameter)
{
    switch (pParameter->parameterizedAnimType)
    {
    case ParameterizedAnimType_Linear: // y = ax
        return pParameter->value.startValue + (pParameter->value.targetValue - pParameter->value.startValue) * normalizedFrame;
    case ParameterizedAnimType_Square: // y = a(1 - (1 - x) ^ 2)
        return pParameter->value.startValue + (pParameter->value.targetValue - pParameter->value.startValue) * (1.0f - (1.0f - normalizedFrame) * (1.0f - normalizedFrame));
    case ParameterizedAnimType_Smooth: // y = a * (1 - cos(πx)) / 2
        return pParameter->value.startValue + (pParameter->value.targetValue - pParameter->value.startValue) * ((1.0f - cos(util::FloatPi * normalizedFrame)) / 2.0f);
    case ParameterizedAnimType_SinOffset: //
        return pParameter->value.startValue + (pParameter->value.targetValue * sin(util::FloatPi * normalizedFrame));
    case ParameterizedAnimType_RandomOffset: //
    {
        std::random_device rd;
        std::default_random_engine generator(rd());
        std::uniform_real_distribution<float> distribution(-pParameter->value.targetValue, pParameter->value.targetValue);
        return pParameter->value.startValue + distribution(generator);
    }
    case ParameterizedAnimType_LinearOffset: //
        return pParameter->value.startValue + pParameter->value.targetValue * normalizedFrame;
    case ParameterizedAnimType_Event:
        // TODO
        return 0.0f;
        break;;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

//-----------------------------------------------------------------------------
float GetParameterizedAnimValueAtFrameClamped(float frame, const ResParameterizedAnimParameter* pParameter)
{
    const float normalizedFrame = (frame - pParameter->offset) / pParameter->duration; // [0.0, 1.0] に正規化したフレーム
    return GetParameterizedAnimValueAtFrame_(std::min(std::max(0.0f, normalizedFrame), 1.0f), pParameter);
}

//-----------------------------------------------------------------------------
float GetParameterizedAnimValueAtFrame(float frame, const ResParameterizedAnimParameter* pParameter)
{
    const float normalizedFrame = (frame - pParameter->offset) / pParameter->duration; // [0.0, 1.0] に正規化したフレーム
    return GetParameterizedAnimValueAtFrame_(normalizedFrame, pParameter);
}

//-----------------------------------------------------------------------------
//  ユーザー拡張データ取得サポート関数
//-----------------------------------------------------------------------------
const ResExtUserData* GetExtUserData(const ResExtUserDataList* pExtUserDataList, const char* pName)
{
    const ResExtUserData* pExtUserData = nn::util::ConstBytePtr(pExtUserDataList, sizeof(*pExtUserDataList)).Get<const ResExtUserData>();

    if (pExtUserDataList != NULL)
    {
        for (int i = 0; i < pExtUserDataList->count; ++i, ++pExtUserData)
        {
            const char* pUserDataName = pExtUserData->GetName();
            if (0 == std::strcmp(pName, pUserDataName))
            {
                return pExtUserData;
            }
        }
    }

    return NULL;
}

//-----------------------------------------------------------------------------
//  バッファのアライメントを考慮したサイズを計算します。
//-----------------------------------------------------------------------------
size_t GetAlignedBufferSize(nn::gfx::Device* pDevice,nn::gfx::GpuAccess gpuAccess, const size_t size)
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(gpuAccess);
    info.SetSize(size);

    //  Buffer のアライメントを計算する。
    return nn::util::align_up(size, nn::gfx::Buffer::GetBufferAlignment(pDevice, info));
}

//-----------------------------------------------------------------------------
//  デフォルトコンストラクタで初期化したペインのマテリアルのシェーダ ID を設定します。
//-----------------------------------------------------------------------------
void SetDefaultShaderId(nn::ui2d::Material* pMaterial, int texCount)
{
    // 2 枚ブレンド以上の場合は一意にシェーダ ID が確定できないため、
    // ShaderId_Undefined のままにしてユーザ側で明示的に SetShaderId() で設定してもらう。
    if (texCount == 0)
    {
        pMaterial->SetShaderId(ShaderId_NullTexture);
    }
    else if (texCount == 1)
    {
        pMaterial->SetShaderId(ShaderId_SingleTexture);
    }
}

struct GfxShaderPool
{
    nn::gfx::MemoryPoolInfoData memoryPoolInfo;
    nn::util::BinTPtr< void > pMemoryPool;
    nn::util::BinTPtr< void > pBuffer;

    char reserved[32];
};

//---------------------------------------------------------------------------
//  nn::gfx::ResShaderContainer で使用されているメモリプールが初期化済みかどうかをチェックします。
//---------------------------------------------------------------------------
bool IsResShaderContainerInitialized(nn::gfx::ResShaderContainer* pContainer)
{
    GfxShaderPool* pShaderPool = reinterpret_cast<GfxShaderPool*>(pContainer->ToData().pShaderBinaryPool.Get());
    if (pShaderPool == NULL)
    {
        return false;
    }

    const nn::gfx::MemoryPool* pMemPool = reinterpret_cast<nn::gfx::MemoryPool*>(pShaderPool->pMemoryPool.Get());

    return pMemPool != NULL && pMemPool->ToData()->state == nn::gfx::MemoryPool::DataType::State_Initialized;
}

//---------------------------------------------------------------------------
//  nn::gfx::ResShaderProgram が初期化済みかどうかをチェックします。
//---------------------------------------------------------------------------
bool IsResShaderProgramInitialized(nn::gfx::ResShaderProgram* pResShaderProgram)
{
    if (pResShaderProgram == NULL)
    {
        return false;
    }

    nn::gfx::Shader* pShader = pResShaderProgram->GetShader();
    return nn::font::IsInitialized(*pShader);
}

//---------------------------------------------------------------------------
//  キャプチャテクスチャの階層構造を含めた名前を作成します。
//---------------------------------------------------------------------------
void MakeCaptureTextureName(char* pNewName, size_t resultSize, const char* pPrefix, const char* pName)
{
    const size_t prefixLength = std::strlen(pPrefix);
    const size_t nameLength = std::strlen(pName);

    // 名前の長さチェック
    // 結合後のサイズ + セパレータ + 終端文字がバッファに収まっているか。
    NN_SDK_ASSERT(prefixLength + nameLength + 1 + 1 <= resultSize, "Lenght of capture texture name exceeded maximum size.\nPrefix = %s, Name = %s\n", pPrefix, pName);
    NN_UNUSED(resultSize);

    memcpy(pNewName, pPrefix, prefixLength);
    *(pNewName + prefixLength) = CaptureTexturePathSeparator;
    memcpy(pNewName + prefixLength + 1, pName, nameLength);
    *(pNewName + prefixLength + 1 + nameLength) = 0;
}

//---------------------------------------------------------------------------
//  BuildArgSet に設定されているキャプチャテクスチャ修飾用文字列スタックの特定の階層までを接続した場合の文字列の長さ(null 文字含む)を計算します。
//---------------------------------------------------------------------------
size_t CalcCaptureTexturePrefixLength(const BuildArgSet& buildArgSet, int concatenateDepth)
{
    size_t  length = 0;
    for (int i = 0; i < concatenateDepth; ++i)
    {
        length += std::strlen(buildArgSet.pCaptureTexturePrefixStack[i]);
        // Separator or null
        ++length;
    }

    return length;
}

//---------------------------------------------------------------------------
// BuildArgSet に設定されているキャプチャテクスチャ修飾用文字列を接続した文字列を作成します。
//---------------------------------------------------------------------------
void ConcatCaptureTexturePrefixString(char* pResult, size_t resultSize, const BuildArgSet& buildArgSet, int concatenateDepth)
{
    size_t resultSizeTotal = 0;
    char* pWrite = pResult;
    for (int i = 0; i < concatenateDepth; ++i)
    {
        const size_t length = std::strlen(buildArgSet.pCaptureTexturePrefixStack[i]);
        memcpy(pWrite, buildArgSet.pCaptureTexturePrefixStack[i], length);
        pWrite += length;
        *pWrite = CaptureTexturePathSeparator;
        ++pWrite;

        resultSizeTotal += (length + 1);
        NN_SDK_ASSERT(resultSize >= resultSizeTotal);
    }
    NN_UNUSED(resultSize);
    NN_UNUSED(resultSizeTotal);

    // 最後に一つ戻して末尾に null 文字を設定する
    --pWrite;
    *pWrite = 0;
}

//---------------------------------------------------------------------------
// BuildArgSet のパーツ階層修飾情報と名前からキャプチャテクスチャ名を生成して、リソースアクセッサから TextureInfo を取得します。
//---------------------------------------------------------------------------
const TextureInfo* AcquireCaptureTextureWithResolvePrefix(char* pNewNameCopy, int newNameCopyBufferSize, const BuildArgSet& buildArgSet, bool overrided, nn::gfx::Device* pDevice, ResourceAccessor* pResAccessor, const char* pName)
{
    const char* pPrefix = NULL;

    int concatenatedStackPosition = buildArgSet.captureTexturePrefixStackPosition;

    if (overrided)
    {
        concatenatedStackPosition = buildArgSet.captureTextureOverridePosition;
        if (concatenatedStackPosition < 0)
        {
            concatenatedStackPosition = 0;
        }
    }

    // キャプチャテクスチャの名前修飾用のプリフィックス文字列を作成します。
    // 一時的なもののためメモリはスタックから確保します。
    if (concatenatedStackPosition > 0)
    {
        size_t  fullTextureNameLength = CalcCaptureTexturePrefixLength(buildArgSet, concatenatedStackPosition);
        NN_SDK_ASSERT(fullTextureNameLength < CaptureTexturePathMax);

        char* pDynamicPrefix = static_cast<char*>(alloca(fullTextureNameLength));
        pPrefix = pDynamicPrefix;

        ConcatCaptureTexturePrefixString(pDynamicPrefix, fullTextureNameLength, buildArgSet, concatenatedStackPosition);
    }

    return AcquireCaptureTexture(pNewNameCopy, newNameCopyBufferSize, pDevice, pResAccessor, pPrefix, pName);
}

//---------------------------------------------------------------------------
// プリフィックスとテクスチャ名からキャプチャテクスチャ名を生成して、リソースアクセッサから TextureInfo を取得します。
//---------------------------------------------------------------------------
const TextureInfo* AcquireCaptureTexture(char* pNewNameCopy, int newNameCopyBufferSize, nn::gfx::Device* pDevice, ResourceAccessor* pResAccessor, const char* pPrefix, const char* pName)
{
    const char* pNewName = pName;
    const size_t prefixLength = pPrefix != NULL ? std::strlen(pPrefix) : 0;

    if (prefixLength > 0)
    {
        const size_t nameLength = std::strlen(pName);
        const size_t bufferSize = prefixLength + nameLength + 1 + 1;

        NN_SDK_ASSERT(bufferSize < CaptureTexturePathMax);

        // 接続した名前用のバッファをスタックから確保する。
        char* pConcatName = static_cast<char*>(alloca(bufferSize));

        // Prefix と Name を接続
        memcpy(pConcatName, pPrefix, prefixLength);
        *(pConcatName + prefixLength) = CaptureTexturePathSeparator;
        memcpy(pConcatName + prefixLength + 1, pName, nameLength);
        *(pConcatName + prefixLength + 1 + nameLength) = 0;

        pNewName = pConcatName;
    }

    const TextureInfo* texInfo = pResAccessor->AcquireTexture(pDevice, pNewName);

    // 必要があれば作成した名前をコピーして返す
    if (pNewNameCopy != NULL)
    {
        strncpy(pNewNameCopy, pNewName, newNameCopyBufferSize);
        pNewNameCopy[newNameCopyBufferSize - 1] = 0;
    }

    return texInfo;
}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
nn::gfx::ShaderCodeType TryInitializeAndGetShaderCodeType(nn::gfx::Device* pDevice, nn::gfx::ResShaderVariation* pVariation)
{
    NN_SDK_ASSERT_NOT_NULL(pVariation);

    nn::gfx::ResShaderProgram* pBinaryProgram = pVariation->GetResShaderProgram(nn::gfx::ShaderCodeType_Binary);
    nn::gfx::ShaderCodeType codeType = nn::gfx::ShaderCodeType_Binary;

    nn::gfx::ShaderInitializeResult shaderResult;
    if (IsResShaderProgramInitialized(pBinaryProgram))
    {
        shaderResult = nn::gfx::ShaderInitializeResult_Success;
    }
    else {
        shaderResult = (pBinaryProgram != NULL) ? pBinaryProgram->Initialize(pDevice) : nn::gfx::ShaderInitializeResult_SetupFailed;
    }

    if (shaderResult != nn::gfx::ShaderInitializeResult_Success)
    {
        // 失敗したら中間表現で初期化する
        nn::gfx::ResShaderProgram* pIrProgram = pVariation->GetResShaderProgram(nn::gfx::ShaderCodeType_Ir);

        if (pIrProgram)
        {
            codeType = nn::gfx::ShaderCodeType_Ir;

            const bool isIrShaderInitialize = IsResShaderProgramInitialized(pIrProgram);
            shaderResult = isIrShaderInitialize ? nn::gfx::ShaderInitializeResult_Success : pIrProgram->Initialize(pDevice);
        }
    }

    if (shaderResult != nn::gfx::ShaderInitializeResult_Success)
    {
        // 失敗したらソースで初期化する
        nn::gfx::ResShaderProgram* pSourceProgram = pVariation->GetResShaderProgram(nn::gfx::ShaderCodeType_Source);
        NN_SDK_ASSERT_NOT_NULL(pSourceProgram);
        codeType = nn::gfx::ShaderCodeType_Source;

        const bool isSrcShaderInitialize = IsResShaderProgramInitialized(pSourceProgram);
        shaderResult = isSrcShaderInitialize ? nn::gfx::ShaderInitializeResult_Success : pSourceProgram->Initialize(pDevice);
    }
    NN_SDK_ASSERT(shaderResult == nn::gfx::ShaderInitializeResult_Success);

    return codeType;
}

namespace detail
{
//---------------------------------------------------------------------------
//  キャプチャする際のルートグローバル行列を取得します。
//---------------------------------------------------------------------------
void CalculateCaptureRootMatrix(nn::util::MatrixT4x3fType& mtx, const DrawInfo& drawInfo)
{
    nn::util::MatrixIdentity(&mtx);

    // 左上がウインドウ原点のシステムでは描画結果が反転するためペインエフェクトのキャプチャ用描画も反転しておく。
    if (drawInfo.IsLeftTopWindowOrigin())
    {
        nn::util::Vector3fType inverseY
            = NN_UTIL_VECTOR_3F_INITIALIZER(1.0f, -1.0f, 1.0f);
        nn::util::MatrixSetScale(&mtx, inverseY);
    }
}

//---------------------------------------------------------------------------
// 値をクランプします。
//---------------------------------------------------------------------------
void ClampValue(float& value, float min, float max)
{
    if (value < min)
    {
        value = min;
    }
    if (value > max)
    {
        value = max;
    }
}

}



//---------------------------------------------------------------------------
// 複合フォントのヘルパークラスです。
//---------------------------------------------------------------------------

int ComplexFontHelper::SetupTextureCacheArg(
    nn::font::TextureCache::InitializeArg* pTextureCacheArg,
    AcquireFontFunction pAcquireFontFunction,
    void* pAcquireFontFunctionUserData,
    const void* pCpxData)
{
    const nn::font::detail::BinaryFileHeader* pHeader = static_cast<const nn::font::detail::BinaryFileHeader*>(pCpxData);

    bool isOldResource8_0 = pHeader->version <= 0x08000000;

    if (!isOldResource8_0)
    {
        if (!nn::font::detail::IsValidBinaryFile(pHeader, BinFileSignatureFcpx, BinaryCpxFileFormatVersion))
        {
            NN_SDK_ASSERT(false, "Invalid bfcpx file.");
        }
    }

    const nn::font::ResComplexFontDescription* pComplexFont = reinterpret_cast<const nn::font::ResComplexFontDescription*>(pHeader + 1);
    const void* pRes = nn::util::ConstBytePtr(pComplexFont, pComplexFont->subFont).Get<void>();

    // TextureCacheArg の初期化

    int fontFaceCount = pTextureCacheArg->fontCount;

    // TextureCacheArg が初期状態であれば font face の数を 0 とみなす
    if (pTextureCacheArg->fontCount == 1)
    {
        if (pTextureCacheArg->fontListCount[0] == 0 ||
            (pTextureCacheArg->fontListCount[0] == 1 && pTextureCacheArg->pFontDatas[0][0] == NULL))
        {
            fontFaceCount = 0;
        }
    }

    int fontFaceHead = fontFaceCount;
    if (isOldResource8_0)
    {
        BuildTextureCacheArg<nn::font::ResMultiScalableFontOld, nn::font::ResScalableFontDescriptionOld>(
            pRes,
            &fontFaceCount,
            pTextureCacheArg,
            pAcquireFontFunction,
            pAcquireFontFunctionUserData,
            pHeader->version);
    }
    else
    {
        BuildTextureCacheArg<nn::font::ResMultiScalableFont, nn::font::ResScalableFontDescription>(
            pRes,
            &fontFaceCount,
            pTextureCacheArg,
            pAcquireFontFunction,
            pAcquireFontFunctionUserData,
            pHeader->version);
    }
    pTextureCacheArg->fontCount = fontFaceCount;
    return fontFaceHead;
}

nn::font::Font* ComplexFontHelper::InitializeComplexFontTree(
    nn::gfx::Device* pDevice,
    nn::font::RegisterTextureViewSlot pRegisterTextureViewSlot,
    void* pUserDataForDescriptorSlotAllocator,
    nn::font::TextureCache* pTextureCache,
    int fontFaceHead,
    AcquireFontFunction pAcquireFontFunction,
    void* pAcquireFontFunctionUserData,
    const void* pCpxData)
{
    const nn::font::detail::BinaryFileHeader* pHeader = static_cast<const nn::font::detail::BinaryFileHeader*>(pCpxData);

    bool isOldResource8_0 = pHeader->version == 0x08000000;

    if (!isOldResource8_0)
    {
        if (!nn::font::detail::IsValidBinaryFile(pHeader, BinFileSignatureFcpx, BinaryCpxFileFormatVersion))
        {
            NN_SDK_ASSERT(false, "Invalid bfcpx file.");
        }
    }

    const nn::font::ResComplexFontDescription* pComplexFont = reinterpret_cast<const nn::font::ResComplexFontDescription*>(pHeader + 1);
    const void* pRes = nn::util::ConstBytePtr(pComplexFont, pComplexFont->subFont).Get<void>();

    // フォントの木構造の構築
    int fontFaceCount = fontFaceHead;
    nn::font::Font* pFont;
    if (isOldResource8_0)
    {
        pFont = BuildFontTree<nn::font::ResMultiScalableFontOld>(
            pDevice,
            pRegisterTextureViewSlot,
            pUserDataForDescriptorSlotAllocator,
            pRes,
            &fontFaceCount,
            pTextureCache,
            pAcquireFontFunction,
            pAcquireFontFunctionUserData,
            pHeader->version);
    }
    else
    {
        pFont = BuildFontTree<nn::font::ResMultiScalableFont>(
            pDevice,
            pRegisterTextureViewSlot,
            pUserDataForDescriptorSlotAllocator,
            pRes,
            &fontFaceCount,
            pTextureCache,
            pAcquireFontFunction,
            pAcquireFontFunctionUserData,
            pHeader->version);
    }
    return pFont;
}

void ComplexFontHelper::FinalizeComplexFontTree(
    nn::gfx::Device* pDevice,
    nn::font::Font* pComplexFontTree,
    nn::font::UnregisterTextureViewSlot pUnregisterTextureViewSlot,
    void* pUserDataForDescriptorSlotDeallocator)
{
    if (pComplexFontTree == NULL)
    {
        return;
    }

    DestroyFontTree(
        pDevice,
        pComplexFontTree,
        pUnregisterTextureViewSlot,
        pUserDataForDescriptorSlotDeallocator);
    Layout::DeleteObj(pComplexFontTree);
}

bool ComplexFontHelper::CheckExt(const char* pName, const char* ext)
{
    const char* pPos = strstr(pName, ext);
    return pPos != NULL && strlen(pPos) == strlen(ext);
}

template <typename T1, typename T2>
void ComplexFontHelper::BuildTextureCacheArg(
    const void* pRes,
    int* pFontFaceCount,
    nn::font::TextureCache::InitializeArg* pTextureCacheInitArg,
    AcquireFontFunction pAcquireFontFunction,
    void* pAcquireFontFunctionUserData,
    uint32_t version)
{
    const nn::font::ResSubFontType type = *static_cast<const nn::font::ResSubFontType*>(pRes);
    switch (type)
    {
    case nn::font::ResSubFontType_BitmapFont:
        // 何もしない
        break;
    case nn::font::ResSubFontType_MultiScalableFont:
        {
            NN_SDK_ASSERT(*pFontFaceCount < nn::font::TextureCache::FontFaceCountMax, "The number of font faces must not be more than %d.", nn::font::TextureCache::FontFaceCountMax);

            const T1* pMultiScFont = static_cast<const T1*>(pRes);

            // TextureCache の設定
            {
                const T2* pDesc = nn::util::ConstBytePtr(pMultiScFont, pMultiScFont->scalableFontDescription).Get<T2>();
                for (uint32_t i = 0; i < pMultiScFont->scalableFontDescriptionCount; i++)
                {
                    const char* pName = nn::util::ConstBytePtr(pMultiScFont, pDesc[i].fontName).Get<char>();

                    size_t fontDataSize = 0;
                    void* pFontData = pAcquireFontFunction(&fontDataSize, pName, nn::ui2d::ResourceTypeScalableFont, pAcquireFontFunctionUserData);
                    NN_SDK_ASSERT(pFontData != NULL, "The font resource [%s] was not found.\n", pName);

                    pTextureCacheInitArg->pFontDatas[*pFontFaceCount][i] = pFontData;
                    pTextureCacheInitArg->pFontDataSizes[*pFontFaceCount][i] = fontDataSize;
                    pTextureCacheInitArg->pBoldWeights[*pFontFaceCount][i] = pDesc[i].boldWeight;
                    pTextureCacheInitArg->pBorderWidths[*pFontFaceCount][i] = pDesc[i].borderWidth;
                    pTextureCacheInitArg->fontIndex[*pFontFaceCount][i] = pDesc[i].ttcFontIndex;
                    SetResScalableFontDescriptionToArg(pTextureCacheInitArg, *pFontFaceCount, i, &pDesc[i], version);

                    nn::font::TextureCache::FontDataType dataType = nn::font::TextureCache::FontDataType_Ttf;
                    if (CheckExt(pName, ".bfttf"))
                    {
                        if (IsRawFont(pFontData))
                        {
                            // 生のデータが渡された場合は .ttf として解釈する
                            dataType = nn::font::TextureCache::FontDataType_Ttf;
                        }
                        else
                        {
                            dataType = nn::font::TextureCache::FontDataType_Bfttf;
                        }
                    }
                    else if (CheckExt(pName, ".bfotf"))
                    {
                        if (IsRawFont(pFontData))
                        {
                            // 生のデータが渡された場合は .otf として解釈する
                            dataType = nn::font::TextureCache::FontDataType_Otf;
                        }
                        else
                        {
                            dataType = nn::font::TextureCache::FontDataType_Bfotf;
                        }
                    }
                    else if (CheckExt(pName, ".otf"))
                    {
                        dataType = nn::font::TextureCache::FontDataType_Otf;
                    }
                    pTextureCacheInitArg->pFontDataTypes[*pFontFaceCount][i] = dataType;

                    const nn::font::ResCharCodeRangeSet* pCharCodeRangeSet = &pDesc[i].charCodeRangeSet;
                    pTextureCacheInitArg->charCodeRangeCount[*pFontFaceCount][i] = pCharCodeRangeSet->charCodeRangeCount;
                    const nn::font::ResCharCodeRange* pCharCodeRange = nn::util::ConstBytePtr(pMultiScFont, pCharCodeRangeSet->charCodeRange).Get<nn::font::ResCharCodeRange>();
                    for (int j = 0; j < nn::font::TextureCache::CharCodeRangeCountMax; j++)
                    {
                        if (j < pCharCodeRangeSet->charCodeRangeCount)
                        {
                            pTextureCacheInitArg->charCodeRangeFirst[*pFontFaceCount][i][j] = pCharCodeRange[j].first;
                            pTextureCacheInitArg->charCodeRangeLast[*pFontFaceCount][i][j] = pCharCodeRange[j].last;
                        }
                        else
                        {
                            pTextureCacheInitArg->charCodeRangeFirst[*pFontFaceCount][i][j] = 0;
                            pTextureCacheInitArg->charCodeRangeLast[*pFontFaceCount][i][j] = 0;
                        }
                    }
                }

                pTextureCacheInitArg->fontListCount[*pFontFaceCount] = pMultiScFont->scalableFontDescriptionCount;
            }

            (*pFontFaceCount)++;
        }
        break;
    case nn::font::ResSubFontType_PairFont:
        {
            const nn::font::ResPairFont* pResPairFont = static_cast<const nn::font::ResPairFont*>(pRes);

            const void* pResForFirst = nn::util::ConstBytePtr(pResPairFont, pResPairFont->firstFont).Get<void>();
            BuildTextureCacheArg<T1, T2>(
                pResForFirst,
                pFontFaceCount,
                pTextureCacheInitArg,
                pAcquireFontFunction,
                pAcquireFontFunctionUserData,
                version);

            const void* pResForSecond = nn::util::ConstBytePtr(pResPairFont, pResPairFont->secondFont).Get<void>();
            BuildTextureCacheArg<T1, T2>(
                pResForSecond,
                pFontFaceCount,
                pTextureCacheInitArg,
                pAcquireFontFunction,
                pAcquireFontFunctionUserData,
                version);
        }
        break;
    default:
        NN_SDK_ASSERT(false, "Unsupported complex font type.");
    }
}

template <typename T>
nn::font::Font* ComplexFontHelper::BuildFontTree(
    nn::gfx::Device* pDevice,
    nn::font::RegisterTextureViewSlot pRegisterTextureViewSlot,
    void* pUserDataForDescriptorSlotAllocator,
    const void* pRes,
    int* pFontFaceCount,
    nn::font::TextureCache* pTextureCache,
    AcquireFontFunction pAcquireFontFunction,
    void* pAcquireFontFunctionUserData,
    uint32_t version)
{
    const nn::font::ResSubFontType type = *static_cast<const nn::font::ResSubFontType*>(pRes);
    switch (type)
    {
    case nn::font::ResSubFontType_BitmapFont:
        {
            const nn::font::ResBitmapFont* pResBitmapFont = static_cast<const nn::font::ResBitmapFont*>(pRes);

            const char* pName = nn::util::ConstBytePtr(pResBitmapFont, pResBitmapFont->fontName).Get<char>();

            size_t fontDataSize = 0;
            void* pFontData = pAcquireFontFunction(&fontDataSize, pName, nn::ui2d::ResourceTypeFont, pAcquireFontFunctionUserData);
            NN_SDK_ASSERT(pFontData != NULL, "The font resource [%s] was not found.\n", pName);

            nn::font::ResFont* pResFont = static_cast<nn::font::ResFont*>(Layout::AllocateAndConstruct<nn::font::ResFont>());
            pResFont->SetResource(pDevice, pFontData);
            pResFont->RegisterTextureViewToDescriptorPool(pRegisterTextureViewSlot, pUserDataForDescriptorSlotAllocator);
            if (pResBitmapFont->charCodeRangeSet.charCodeRangeCount > 0)
            {
                uint32_t first[nn::font::ResFontBase::CharCodeRangeCountMax];
                uint32_t last[nn::font::ResFontBase::CharCodeRangeCountMax];
                const nn::font::ResCharCodeRange* pCharCodeRange = nn::util::ConstBytePtr(pResBitmapFont, pResBitmapFont->charCodeRangeSet.charCodeRange).Get<nn::font::ResCharCodeRange>();
                for (int i = 0; i < pResBitmapFont->charCodeRangeSet.charCodeRangeCount; i++)
                {
                    first[i] = pCharCodeRange[i].first;
                    last[i] = pCharCodeRange[i].last;
                }
                pResFont->SetCharCodeRange(pResBitmapFont->charCodeRangeSet.charCodeRangeCount, first, last);
            }
            return pResFont;
        }
        break;
    case nn::font::ResSubFontType_MultiScalableFont:
        {
            NN_SDK_ASSERT(*pFontFaceCount < nn::font::TextureCache::FontFaceCountMax, "The number of font faces must not be more than %d.", nn::font::TextureCache::FontFaceCountMax);

            const T* pMultiScFont = static_cast<const T*>(pRes);

            nn::font::ScalableFont::InitializeArg arg;
            arg.SetDefault();
            arg.pTextureCache = pTextureCache;
            arg.fontSize = static_cast<int>(pMultiScFont->size); // float型を整数に切り捨てて設定しています。
            arg.fontFace = static_cast<uint16_t>(*pFontFaceCount);
            SetResMultiScalableFontToArg(&arg, pMultiScFont);
            arg.isAlternateCharSpaceWithOriginalWidthForNotReadyChar = true;

            nn::font::ScalableFont* pScFont = static_cast<nn::font::ScalableFont*>(Layout::AllocateAndConstruct<nn::font::ScalableFont>());
            pScFont->Initialize(arg);

            (*pFontFaceCount)++;

            return pScFont;
        }
        break;
    case nn::font::ResSubFontType_PairFont:
        {
            const nn::font::ResPairFont* pResPairFont = static_cast<const nn::font::ResPairFont*>(pRes);

            const void* pResForFirst = nn::util::ConstBytePtr(pResPairFont, pResPairFont->firstFont).Get<void>();
            nn::font::Font* pFirst = BuildFontTree<T>(
                pDevice,
                pRegisterTextureViewSlot,
                pUserDataForDescriptorSlotAllocator,
                pResForFirst,
                pFontFaceCount,
                pTextureCache,
                pAcquireFontFunction,
                pAcquireFontFunctionUserData,
                version);

            const void* pResForSecond = nn::util::ConstBytePtr(pResPairFont, pResPairFont->secondFont).Get<void>();
            nn::font::Font* pSecond = BuildFontTree<T>(
                pDevice,
                pRegisterTextureViewSlot,
                pUserDataForDescriptorSlotAllocator,
                pResForSecond,
                pFontFaceCount,
                pTextureCache,
                pAcquireFontFunction,
                pAcquireFontFunctionUserData,
                version);

            nn::font::PairFont* pPairFont = static_cast<nn::font::PairFont*>(Layout::AllocateAndConstruct<nn::font::PairFont>());
            pPairFont->SetFont(pFirst, pSecond);
            return pPairFont;
        }
        break;
    default:
        NN_SDK_ASSERT(false, "Unsupported complex font type.");
        return NULL;
    }
}

void ComplexFontHelper::DestroyFontTree(
    nn::gfx::Device* pDevice,
    nn::font::Font* pFont,
    nn::font::UnregisterTextureViewSlot pUnregisterTextureViewSlot,
    void* pUserDataForDescriptorSlotDeallocator)
{
    // フォントの種類に従って解放する

    // ペアフォント
    nn::font::PairFont* pPairFont = nn::font::DynamicCast<nn::font::PairFont*>(pFont);
    if (pPairFont != NULL)
    {
        DestroyFontTree(
            pDevice,
            pPairFont->GetFontFirst(),
            pUnregisterTextureViewSlot,
            pUserDataForDescriptorSlotDeallocator);
        Layout::DeleteObj(pPairFont->GetFontFirst());
        DestroyFontTree(
            pDevice,
            pPairFont->GetFontSecond(),
            pUnregisterTextureViewSlot,
            pUserDataForDescriptorSlotDeallocator);
        Layout::DeleteObj(pPairFont->GetFontSecond());
        pPairFont->Finalize(pDevice);
        return;
    }

    // ビットマップフォント
    nn::font::ResFont* pResFont = nn::font::DynamicCast<nn::font::ResFont*>(pFont);
    if (pResFont != NULL)
    {
        pResFont->UnregisterTextureViewFromDescriptorPool(pUnregisterTextureViewSlot, pUserDataForDescriptorSlotDeallocator);
        pResFont->Finalize(pDevice);
        return;
    }

    // その他のフォント
    pFont->Finalize(pDevice);
}



RoundRectShape::RoundRectShape(
    nn::gfx::util::PrimitiveShapeFormat       vertexFormat,
    nn::gfx::PrimitiveTopology primitiveTopology,
    float cornerRadiusH,
    float cornerRadiusV,
    uint32_t cornerSliceCount)
    : PrimitiveShape(vertexFormat, primitiveTopology)
    , m_CornerRadiusH(cornerRadiusH * 2.0f)
    , m_CornerRadiusV(cornerRadiusV * 2.0f)
    , m_CornerSliceCount(cornerSliceCount)
{
    // プリミティブトポロジーが対応しているものかチェック
    NN_SDK_REQUIRES(GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList);

    // 頂点数を計算する
    SetVertexCount(CalculateVertexCount());

    // インデックス数を計算する
    SetIndexCount(CalculateIndexCount());

    const size_t stride = GetStride();

    // 頂点バッファのサイズを計算する。
    SetVertexBufferSize(stride * GetVertexCount());

    // インデックスバッファのサイズを計算する。
    size_t  indexBufferStride = 1;
    switch (GetIndexBufferFormat())
    {
    case nn::gfx::IndexFormat_Uint8:
        indexBufferStride = 1;
        break;
    case nn::gfx::IndexFormat_Uint16:
        indexBufferStride = 2;
        break;
    case nn::gfx::IndexFormat_Uint32:
        indexBufferStride = 4;
        break;
    default:
        NN_SDK_ASSERT(false);
    }
    SetIndexBufferSize(indexBufferStride * GetIndexCount());
}

RoundRectShape::RoundRectShape(
    nn::gfx::util::PrimitiveShapeFormat       vertexFormat,
    nn::gfx::PrimitiveTopology primitiveTopology)
    : PrimitiveShape(vertexFormat, primitiveTopology)
{
    // プリミティブトポロジーが対応しているものかチェック
    NN_SDK_REQUIRES(GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList);

    m_CornerRadiusH = 0.0f;
    m_CornerRadiusV = 0.0f;
    m_CornerSliceCount = 1;
}

RoundRectShape::~RoundRectShape()
{
}

void RoundRectShape::CopyParams(const RoundRectShape& src)
{
    m_CornerRadiusH = src.m_CornerRadiusH;
    m_CornerRadiusV = src.m_CornerRadiusV;
    m_CornerSliceCount = src.m_CornerSliceCount;

    // PrimitiveShape のパラメータコピー

    SetVertexBufferSize(src.GetVertexBufferSize());
    SetIndexBufferSize(src.GetIndexBufferSize());
    SetVertexCount(src.GetVertexFormat());
    SetIndexCount(src.GetIndexCount());

    void*   pVertexBuffer = Layout::AllocateMemory(src.GetVertexBufferSize());
    memcpy(pVertexBuffer, src.GetVertexBuffer(), src.GetVertexBufferSize());
    SetVertexBuffer(pVertexBuffer);

    void*   pIndexBuffer = Layout::AllocateMemory(src.GetIndexBufferSize());
    memcpy(pIndexBuffer, src.GetIndexBuffer(), src.GetIndexBufferSize());
    SetIndexBuffer(pIndexBuffer);
}

int RoundRectShape::CalculateVertexCount()
{
    // 角 + 左右の角間矩形 + 中央の矩形
    return ((m_CornerSliceCount + 2) * 4) + (2 * 4) + (4);
}

int RoundRectShape::CalculateIndexCount()
{
    // 角 + 左右の角間矩形 + 中央の矩形
    return (m_CornerSliceCount * 3 * 4) + (3 * 4) + (6);
}

void* RoundRectShape::CalculateVertexBuffer()
{
    float* pVertexBuffer = static_cast<float*>(GetVertexBuffer());

    float   halfWidth = 1.0f;
    float   halfHeight = 1.0f;
    float   cornerWidth = halfWidth < m_CornerRadiusH ? halfWidth : m_CornerRadiusH;
    float   cornerHeight = halfHeight < m_CornerRadiusV ? halfHeight : m_CornerRadiusV;

    float   quadrantSign[4][2] =
    {
        { 1.0f,  1.0f }, //  第一象限
        { -1.0f,  1.0f }, //  第二象限
        { -1.0f, -1.0f }, //  第三象限
        { 1.0f, -1.0f }, //  第四象限
    };

    uint32_t    roundVertexCount = m_CornerSliceCount + 1;

    // 象限分ループ
    for (int quadrantIndex = 0; quadrantIndex < 4; ++quadrantIndex)
    {
        // 角丸の中心座標
        {
            float   x = (halfWidth - cornerWidth) * quadrantSign[quadrantIndex][0];
            float   y = (halfHeight - cornerHeight) * quadrantSign[quadrantIndex][1];

            if (GetVertexFormat() & nn::gfx::util::PrimitiveShapeFormat_Pos)
            {
                pVertexBuffer[0] = x;
                pVertexBuffer[1] = y;
                pVertexBuffer[2] = 0.0f;
                pVertexBuffer += 3;
            }

            if (GetVertexFormat() & nn::gfx::util::PrimitiveShapeFormat_Uv)
            {
                pVertexBuffer[0] = CalculateTexMapU(x, halfWidth);
                pVertexBuffer[1] = CalculateTexMapV(y, halfHeight);
                pVertexBuffer += 2;
            }
        }

        float angle = 0.0f;
        //  円周側頂点
        // sin へ 2pi を渡すと精度が悪く 0 にならないため、そのまま計算すると頂点位置が微妙にずれて隙間が発生することがある。
        // 精度のよい第一象限の値と符号で生成する座標を作成するようにしています。
        for (uint32_t i = 0; i < roundVertexCount; ++i)
        {
            float cosValue = cosf(angle);
            float sinValue = sinf(angle);

            float   x = (halfWidth - cornerWidth + cosValue * cornerWidth) * quadrantSign[quadrantIndex][0];
            float   y = (halfHeight - cornerHeight + sinValue * cornerHeight) * quadrantSign[quadrantIndex][1];

            if (GetVertexFormat() & nn::gfx::util::PrimitiveShapeFormat_Pos)
            {
                pVertexBuffer[0] = x;
                pVertexBuffer[1] = y;
                pVertexBuffer[2] = 0.0f;
                pVertexBuffer += 3;
            }

            if (GetVertexFormat() & nn::gfx::util::PrimitiveShapeFormat_Uv)
            {
                pVertexBuffer[0] = CalculateTexMapU(x, halfWidth);
                pVertexBuffer[1] = CalculateTexMapV(y, halfHeight);
                pVertexBuffer += 2;
            }

            angle += ((nn::util::FloatPi * 0.5f) * (1.0f / (float)m_CornerSliceCount));
        }
    }

    // 左右の角間の矩形
    if (cornerWidth * 2.0f < 2.0f)
    {
        // 象限分ループ
        for (int quadrantIndex = 0; quadrantIndex < 4; ++quadrantIndex)
        {
            {
                float x = (halfWidth - cornerWidth) * quadrantSign[quadrantIndex][0];
                float y = halfHeight * quadrantSign[quadrantIndex][1];

                if (GetVertexFormat() & nn::gfx::util::PrimitiveShapeFormat_Pos)
                {
                    pVertexBuffer[0] = x;
                    pVertexBuffer[1] = y;
                    pVertexBuffer[2] = 0.0f;
                    pVertexBuffer += 3;
                }

                if (GetVertexFormat() & nn::gfx::util::PrimitiveShapeFormat_Uv)
                {
                    pVertexBuffer[0] = CalculateTexMapU(x, halfWidth);
                    pVertexBuffer[1] = CalculateTexMapV(y, halfHeight);
                    pVertexBuffer += 2;
                }
            }

            {
                float x = (halfWidth - cornerWidth) * quadrantSign[quadrantIndex][0];
                float y = (halfHeight - cornerHeight) * quadrantSign[quadrantIndex][1];

                if (GetVertexFormat() & nn::gfx::util::PrimitiveShapeFormat_Pos)
                {
                    pVertexBuffer[0] = x;
                    pVertexBuffer[1] = y;
                    pVertexBuffer[2] = 0.0f;
                    pVertexBuffer += 3;
                }

                if (GetVertexFormat() & nn::gfx::util::PrimitiveShapeFormat_Uv)
                {
                    pVertexBuffer[0] = CalculateTexMapU(x, halfWidth);
                    pVertexBuffer[1] = CalculateTexMapV(y, halfHeight);
                    pVertexBuffer += 2;
                }
            }
        }
    }

    // 上下の角間の矩形
    // 真ん中の領域もつなげて大きな矩形にする。
    if (cornerHeight * 2.0f < 2.0f)
    {
        // 象限分ループ
        for (int quadrantIndex = 0; quadrantIndex < 4; ++quadrantIndex)
        {
            float x = halfWidth * quadrantSign[quadrantIndex][0];
            float y = (halfHeight - cornerHeight) * quadrantSign[quadrantIndex][1];

            if (GetVertexFormat() & nn::gfx::util::PrimitiveShapeFormat_Pos)
            {
                pVertexBuffer[0] = x;
                pVertexBuffer[1] = y;
                pVertexBuffer[2] = 0.0f;
                pVertexBuffer += 3;
            }

            if (GetVertexFormat() & nn::gfx::util::PrimitiveShapeFormat_Uv)
            {
                pVertexBuffer[0] = CalculateTexMapU(x, halfWidth);
                pVertexBuffer[1] = CalculateTexMapV(y, halfHeight);
                pVertexBuffer += 2;
            }
        }
    }

    // 頂点バッファの末端アドレスを返す。
    return pVertexBuffer;
}// NOLINT(impl/function_size)

template <typename T>
void RoundRectShape::CalculateIndexBuffer()
{
    T* pIndexData = static_cast<T*>(GetIndexBuffer());

    float   halfWidth = 1.0f;
    float   halfHeight = 1.0f;
    float   cornerWidth = halfWidth < m_CornerRadiusH ? halfWidth : m_CornerRadiusH;
    float   cornerHeight = halfHeight < m_CornerRadiusV ? halfHeight : m_CornerRadiusV;

    uint32_t    roundVertexCount = m_CornerSliceCount + 1;

    // 象限分ループ
    for (int quadrantIndex = 0; quadrantIndex < 4; ++quadrantIndex)
    {
        //  円周側頂点
        for (uint32_t i = 0; i < roundVertexCount; ++i)
        {
            if (i > 0)
            {
                int indexOffsetBase = (roundVertexCount + 1) * quadrantIndex;

                pIndexData[0] = static_cast<T>(indexOffsetBase);
                // ひとつ前の頂点のインデックスだが、原点頂点のオフセットが常に 1 加算されているため、オフセットがなしになる。
                pIndexData[1] = static_cast<T>(indexOffsetBase + i);
                pIndexData[2] = static_cast<T>(indexOffsetBase + i + 1);
                pIndexData += 3;
            }
        }
    }

    int indexOffsetBase = (roundVertexCount + 1) * 4;

    // 左右の角間の矩形
    if (cornerWidth * 2.0f < 2.0f)
    {
        pIndexData[0] = static_cast<T>(indexOffsetBase);
        pIndexData[1] = static_cast<T>(indexOffsetBase + 2);
        pIndexData[2] = static_cast<T>(indexOffsetBase + 3);
        pIndexData += 3;

        pIndexData[0] = static_cast<T>(indexOffsetBase);
        pIndexData[1] = static_cast<T>(indexOffsetBase + 3);
        pIndexData[2] = static_cast<T>(indexOffsetBase + 1);
        pIndexData += 3;

        indexOffsetBase += 4;

        pIndexData[0] = static_cast<T>(indexOffsetBase);
        pIndexData[1] = static_cast<T>(indexOffsetBase + 3);
        pIndexData[2] = static_cast<T>(indexOffsetBase + 2);
        pIndexData += 3;

        pIndexData[0] = static_cast<T>(indexOffsetBase);
        pIndexData[1] = static_cast<T>(indexOffsetBase + 1);
        pIndexData[2] = static_cast<T>(indexOffsetBase + 3);
        pIndexData += 3;

        indexOffsetBase += 4;
    }

    // 上下の角間の矩形
    // 真ん中の領域もつなげて大きな矩形にする。
    if (cornerHeight * 2.0f < 2.0f)
    {
        for (int i = 0; i < 2; ++i)
        {
            pIndexData[0] = static_cast<T>(indexOffsetBase);
            pIndexData[1] = static_cast<T>(indexOffsetBase + i + 1);
            pIndexData[2] = static_cast<T>(indexOffsetBase + i + 2);
            pIndexData += 3;
        }
        indexOffsetBase += 4;
    }
}

void RoundRectShape::CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize)
{
    NN_UNUSED(vertexSize);
    NN_UNUSED(indexSize);

    // バッファサイズが足りているかチェックします。
    NN_SDK_ASSERT((vertexSize >= GetVertexBufferSize()) || (indexSize >= GetIndexBufferSize()));

    // 頂点バッファのセット
    SetVertexBuffer(pVertexMemory);

    // 頂点バッファの計算
    CalculateVertexBuffer();

    // インデックスバッファのセット
    SetIndexBuffer(pIndexMemory);

    // インデックスバッファの計算
    switch (GetIndexBufferFormat())
    {
    case nn::gfx::IndexFormat_Uint8:
        CalculateIndexBuffer<uint8_t>();
        break;
    case nn::gfx::IndexFormat_Uint16:
        CalculateIndexBuffer<uint16_t>();
        break;
    case nn::gfx::IndexFormat_Uint32:
        CalculateIndexBuffer<uint32_t>();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

void Ui2dCircleShape::CopyParams(const Ui2dCircleShape& src)
{
    // PrimitiveShape のパラメータコピー

    SetVertexBufferSize(src.GetVertexBufferSize());
    SetIndexBufferSize(src.GetIndexBufferSize());
    SetVertexCount(src.GetVertexFormat());
    SetIndexCount(src.GetIndexCount());

    void*   pVertexBuffer = Layout::AllocateMemory(src.GetVertexBufferSize());
    memcpy(pVertexBuffer, src.GetVertexBuffer(), src.GetVertexBufferSize());
    SetVertexBuffer(pVertexBuffer);

    void*   pIndexBuffer = Layout::AllocateMemory(src.GetIndexBufferSize());
    memcpy(pIndexBuffer, src.GetIndexBuffer(), src.GetIndexBufferSize());
    SetIndexBuffer(pIndexBuffer);
}

//---------------------------------------------------------------------------
//  レイアウトがフレームバッファテクスチャを要求するかを取得します。
//---------------------------------------------------------------------------
namespace
{

bool FindRecursivelyFrameBufferTextureRequired(nn::ui2d::Pane* pPane)
{
    NN_SDK_ASSERT_NOT_NULL(pPane);
    const uint8_t materialCount = pPane->GetMaterialCount();
    for (uint8_t j = 0; j < materialCount; j++)
    {
        nn::ui2d::Material* pMaterial = pPane->GetMaterial(j);
        if (pMaterial == NULL)
        {
            continue;
        }
        if (!pMaterial->IsUseFramebufferTexture())
        {
            continue;
        }
        const nn::ui2d::ShaderInfo* pShaderInfo = pMaterial->GetShaderInfo();
        NN_SDK_ASSERT_NOT_NULL(pShaderInfo);
        NN_SDK_ASSERT(pShaderInfo->GetTextureSlotCount() == nn::ui2d::CombinerUserShaderTexMapMax);
        const int slotTex = pShaderInfo->GetTextureSlot(pMaterial->GetShaderVariation(), nn::ui2d::CombinerUserShaderTexMapMax - 1);
        if (slotTex < 0)
        {
            continue;
        }

        // フレームバッファ用のテクスチャスロットが最適化されていない場合（存在する場合）はフレームバッファテクスチャが利用されています。
        return true;
    }

    nn::ui2d::PaneList::iterator endIter = pPane->GetChildList().end();
    for (nn::ui2d::PaneList::iterator iter = pPane->GetChildList().begin(); iter != endIter; ++iter)
    {
        if (FindRecursivelyFrameBufferTextureRequired(&(*iter)))
        {
            return true;
        }
    }

    return false;
}

} // namespace {no-name}

bool CheckFrameBufferTextureDescriptorSlotRequired(const Layout* pLayout)
{
    NN_SDK_ASSERT_NOT_NULL(pLayout);
    return FindRecursivelyFrameBufferTextureRequired(pLayout->GetRootPane());
}

} // namespace nn::ui2d
} // namespace nn
