﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_Layout.h>
#include <nn/ui2d/ui2d_ControlCreator.h>
#include <nn/ui2d/ui2d_TouchDragButton.h>

namespace nn
{
namespace ui2d
{

TouchDragButton::TouchDragButton()
: DragButton()
{
    // ドラッグの設定
    SetDragMode(DragMode_OnHit);
    // offとdownは受け付けない
    SetAcceptAction(Action_Off, false);
    SetAcceptAction(Action_Down, false);
}

void TouchDragButton::Build(nn::gfx::Device* pDevice, ui2d::Layout* pLayout, const ui2d::ControlSrc& controlSrc)
{
    // Touchアニメーションの取得(実際にはDownに格納)
    nn::ui2d::Animator* pDownAnimator = pLayout->CreateGroupAnimatorAuto(pDevice, controlSrc.FindFunctionalAnimName("Touch"), true);
    NN_SDK_ASSERT(pDownAnimator, "cannot create TouchAnimator[%s] for Layout[%s]", controlSrc.FindFunctionalAnimName("Touch"), pLayout->GetName());
    SetDownAnimator(pDownAnimator);
    pDownAnimator->StopAtStartFrame();

    // Releaseアニメーションの取得(実際にはCancelに格納)
    nn::ui2d::Animator* pCancelAnimator = pLayout->CreateGroupAnimatorAuto(pDevice, controlSrc.FindFunctionalAnimName("Release"), false);
    NN_SDK_ASSERT(pCancelAnimator, "cannot create CancelAnimator[%s] for Layout[%s]", controlSrc.FindFunctionalAnimName("Release"), pLayout->GetName());
    SetCancelAnimator(pCancelAnimator);

    // Disableアニメーションがあれば作成
    {
        const char* pAnimName = controlSrc.FindFunctionalAnimName("Disable");
        if (pAnimName && pAnimName[0]) {
            nn::ui2d::Animator* pDisableAnimator = pLayout->CreateGroupAnimatorAuto(pDevice, pAnimName, true);
            SetDisableAnimator(pDisableAnimator);
        }
    }
    // 当たりのペイン
    Pane* pHitPane = pLayout->GetRootPane()->FindPaneByName(controlSrc.FindFunctionalPaneName("Hit"));
    NN_SDK_ASSERT(pHitPane != NULL, "cannot get HitPane[%s] for Layout[%s]", controlSrc.FindFunctionalPaneName("Hit"), pLayout->GetName());
    SetHitPane(pHitPane);

    // ボタンの名前は、レイアウトが部品であれば部品ペイン名（ルートペインとなっている）、
    // 部品でなければレイアウト名とする
    if (pLayout->IsPartsLayout()) {
        SetName(pLayout->GetRootPane()->GetName());
    } else {
        SetName(pLayout->GetName());
    }

    SetRootPane(pLayout->GetRootPane());
}

bool TouchDragButton::ProcessOn()
{
    const State currentState = GetState();
    switch (currentState) {
    case State_OffIdle:
        ChangeState(State_Down);
        StartDown();
        return true;
    case State_On:
        return true;
    case State_Off:
        return true;
    case State_OnIdle:
        return true;
    case State_Down:
        return true;
    case State_DownIdle:
        return true;
    case State_Cancel:
        // Releaseアニメーション中にタッチされた場合は
        // キューに入れて待つ
        return false;
    default:
        NN_SDK_ASSERT(false, "Unknown State.");
        return true;
    };
}

bool TouchDragButton::ProcessCancel()
{
    const State currentState = GetState();
    switch (currentState) {
    case State_OffIdle:
        return true;
    case State_On:
        return true;
    case State_Off:
        return true;
    case State_OnIdle:
        return true;
    case State_Down:
        // Touchアニメーション中に離した場合は
        // キューに入れて待つ
        return false;
    case State_DownIdle:
        ChangeState(State_Cancel);
        StartCancel();
        return true;
    case State_Cancel:
        return true;
    default:
        NN_SDK_ASSERT(false, "Unknown State.");
        return true;
    };
}

void TouchDragButton::FinishCancel()
{
    // ReleaseアニメーションのあとはOFF_IDLEに戻る
    ChangeState(State_OffIdle);
}

} // namespace nn::ui2d
} // namespace nn
