﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_Scissor.h>

namespace nn
{
namespace ui2d
{

//----------------------------------------
Scissor::Scissor()
: Base()
{
}

//----------------------------------------
Scissor::~Scissor()
{
}

//----------------------------------------
Scissor::Scissor(const ResScissor* pBlock, const BuildArgSet& buildArgSet)
: Base(pBlock, buildArgSet)
{
}

//----------------------------------------
void Scissor::Draw(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    if (IsVisible() && GetGlobalAlpha() > 0)
    {
        NN_SDK_ASSERT(
            drawInfo.IsValidDefaultViewportInfoSet() && drawInfo.IsValidDefaultScissorInfoSet(),
            "You must set default viewport and scissor information by DrawInfo::SetDefaultViewportScissorInfo() to use scissor pane function.");

        float viewportWidth = drawInfo.GetDefaultViewportInfo().GetWidth();
        float viewportHeight = drawInfo.GetDefaultViewportInfo().GetHeight();

        nn::util::Vector3f  vX, vY, vW;
        nn::util::MatrixGetAxisX(&vX, GetGlobalMtx());
        nn::util::MatrixGetAxisY(&vY, GetGlobalMtx());
        nn::util::MatrixGetAxisW(&vW, GetGlobalMtx());

        float centerX = vW.GetX();
        float centerY = vW.GetY();
        float halfWidthSigned = (GetSize().width * vX.GetX()) * 0.5f;
        float halfHeightSigned = (GetSize().height * vY.GetY()) * 0.5f;
        float width = fabsf(halfWidthSigned) * 2.0f;
        float height = fabsf(halfHeightSigned) * 2.0f;

        // 本来頂点シェーダーで適用される処理である負のスケールを考慮して領域を調整する。
        switch (GetBasePositionX())
        {
        case HorizontalPosition_Left:
            centerX += halfWidthSigned;
            break;
        case HorizontalPosition_Right:
            centerX -= halfWidthSigned;
            break;
        default:
            break;
        }

        switch (GetBasePositionY())
        {
        case VerticalPosition_Top:
            centerY -= halfHeightSigned;
            break;
        case VerticalPosition_Bottom:
            centerY += halfHeightSigned;
            break;
        default:
            break;
        }

        float originX = centerX - (width * 0.5f) + viewportWidth * 0.5f;
        float originY = centerY - (height * 0.5f) + viewportHeight * 0.5f;

        // シザリングに設定される領域で画面がになる部分をクリップする。
        if (originX < 0.0f)
        {
            width += originX;
            originX = 0.0f;
        }
        if (originX + width > viewportWidth)
        {
            width -= ((originX + width) - viewportWidth);
        }

        if (originY < 0.0f)
        {
            height += originY;
            originY = 0.0f;
        }
        if (originY + height > viewportHeight)
        {
            height -= ((originY + height) - viewportHeight);
        }

        // ウインドウ原点が左上にあるシステムの場合は Y 軸を反転する
        if (drawInfo.IsLeftTopWindowOrigin())
        {
            originY = viewportHeight - (originY + height);
        }

        // ペイン全体が画面外にある場合は描画範囲が画面内にないので描画処理を呼び出さない。
        if (originX < viewportWidth &&
            originY < viewportHeight &&
            width > 0.0f &&
            height > 0.0f)
        {
            nn::gfx::ScissorStateInfo newScissorStateInfo;

            newScissorStateInfo.SetOriginX(static_cast<int>(originX));
            newScissorStateInfo.SetOriginY(static_cast<int>(originY));
            newScissorStateInfo.SetWidth(static_cast<int>(width));
            newScissorStateInfo.SetHeight(static_cast<int>(height));

            // シザーペイン以下でシザー設定が変更されるケースの対応。
            // 古いデフォルトシザー設定を保存して、デフォルトシザー設定を新しい設定に変更する。
            nn::gfx::ScissorStateInfo  oldDefaultScissorStateInfo = drawInfo.GetDefaultScissorInfo();

            commandBuffer.SetScissors(0, 1, &newScissorStateInfo);
            drawInfo.SetDefaultViewportScissorInfo(drawInfo.GetDefaultViewportInfo(), newScissorStateInfo);

            // 子ペインの描画
            Pane::Draw(drawInfo, commandBuffer);

            // シザリング設定の復帰
            commandBuffer.SetScissors(0, 1, &oldDefaultScissorStateInfo);
            drawInfo.SetDefaultViewportScissorInfo(drawInfo.GetDefaultViewportInfo(), oldDefaultScissorStateInfo);
        }
    }
    else
    {
        // 自分自身が見えない状態でも子階層では表示されるケースでの描画処理。
        Pane::Draw(drawInfo, commandBuffer);
    }

}

} // namespace nn::ui2d
} // namespace nn

