﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_Common.h>
#include <nn/ui2d/ui2d_Group.h>
#include <nn/ui2d/ui2d_Layout.h>
#include <nn/ui2d/ui2d_Resources.h>

namespace nn
{
namespace ui2d
{

Group::Group()
: m_pName("")
, m_UserAllocated(false)
{
}

Group::Group(const char* pName)
: m_pName(pName)
, m_UserAllocated(false)
{
    NN_SDK_ASSERT_NOT_NULL(pName);
    NN_SDK_ASSERT(std::strlen(pName) <= GroupNameStrMax);
}

Group::Group(
    const ResGroup* pResGroup,
    Pane* pRootPane
)
: m_pName(pResGroup->name)
, m_UserAllocated(false)
{
    const char *const pPaneNameBase = nn::util::ConstBytePtr(pResGroup, sizeof(ResGroup)).Get<char>();

    for (int i = 0; i < pResGroup->paneCount; ++i)
    {
        if (Pane* pFindPane = pRootPane->FindPaneByName(pPaneNameBase + i * ResourceNameStrMax, true))
        {
            AppendPane(pFindPane);
        }
    }
}

Group::Group(
    const Group& group,
    Pane* pRootPane
)
: m_pName(group.m_pName)
, m_UserAllocated(false)
{
    const PaneLinkList& list = group.m_PaneLinkList;
    if (pRootPane) {
        for (nn::ui2d::PaneLinkList::const_iterator iter = list.begin(), endIter = list.end(); iter != endIter; ++iter) {
            AppendPane(pRootPane->FindPaneByName(iter->pTarget->GetName()));
        }
    } else {
        for (nn::ui2d::PaneLinkList::const_iterator iter = list.begin(), endIter = list.end(); iter != endIter; ++iter) {
            AppendPane(iter->pTarget);
        }
    }
}

Group::~Group()
{
    // OSReport("Group::~Group()\n");

    for (PaneLinkList::iterator iter = m_PaneLinkList.begin(); iter != m_PaneLinkList.end();)
    {
        PaneLinkList::iterator currIter = iter++;
        m_PaneLinkList.erase(currIter);
        Layout::DeleteObj(&(*currIter));
    }
}

void
Group::AppendPane(Pane* pPane)
{
    if (PaneLink* pPaneLink = Layout::AllocateAndConstruct<PaneLink>())
    {
        pPaneLink->pTarget = pPane;
        m_PaneLinkList.push_back(*pPaneLink);
    }
}

bool
Group::CompareCopiedInstanceTest(const Group& target) const
{
    NN_UNUSED(target);
    return true;
}

GroupContainer::~GroupContainer()
{
    // OSReport("GroupContainer::~GroupContainer()\n");

    // Groupの後処理
    for (GroupList::iterator iter = m_GroupList.begin(); iter != m_GroupList.end();)
    {
        GroupList::iterator currIter = iter++;
        m_GroupList.erase(currIter);
        if (! currIter->IsUserAllocated())
        {
            Layout::DeleteObj(&(*currIter));
        }
    }
}

void
GroupContainer::AppendGroup(Group* pGroup)
{
    m_GroupList.push_back(*pGroup);
}

Group*
GroupContainer::FindGroupByName(const char* pFindName)
{
    // 子供に一致するものが無いか検索
    GroupList::iterator endIter = m_GroupList.end();
    for (GroupList::iterator iter = m_GroupList.begin(); iter != endIter; ++iter)
    {
        if (detail::EqualsGroupName(iter->GetName(), pFindName))
        {
            return &(*iter);
        }
    }

    return 0;
}

const Group*
GroupContainer::FindGroupByName(const char* pFindName) const
{
    return const_cast<GroupContainer*>(this)->FindGroupByName(pFindName);
}

} // namespace nn::ui2d
} // namespace nn
