﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_GraphicsResource.h>
#include <nn/ui2d/ui2d_Common.h>
#include <nn/ui2d/ui2d_Layout.h>

namespace nn
{
namespace ui2d
{

static const ptrdiff_t VertexBufferStride = sizeof(float) * 2;
static const int VertexBufferSize = VertexBufferStride * 4;

static const ptrdiff_t IndexBufferStride = sizeof(uint16_t) * 1;
static const int IndexBufferSize = IndexBufferStride * GraphicsResource::VertexBufferIndexCount;

PresetBlendStateId GraphicsResource::DefalutPresetBlendStateId = PresetBlendStateId_Default;

namespace
{
#if defined(NN_BUILD_CONFIG_SPEC_NX)

#if (NN_GFX_IS_TARGET_VK)
    #include "..\..\..\Resources\Ui2dShaders\ui2d_BuildinShader_Vk_Ir_Source.h"
#elif (NN_GFX_IS_TARGET_GL)
    #define NN_DETAIL_UI2D_INCLUDE_GL_BUILDINSHADER
#elif defined(NN_BUILD_CONFIG_OS_WIN)
    #include "..\..\..\Resources\Ui2dShaders\ui2d_BuildinShader_Nvn_Binary_Ir.h"
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "..\..\..\Resources\Ui2dShaders\ui2d_BuildinShader_Nvn_Binary.h"
#endif

#elif defined(NN_BUILD_CONFIG_OS_WIN)
    #define NN_DETAIL_UI2D_INCLUDE_GL_BUILDINSHADER
#else
    static uint8_t Ui2dBuildinShader[1] = { 0 };
#endif

#if defined(NN_DETAIL_UI2D_INCLUDE_GL_BUILDINSHADER)
    // 同じヘッダファイルの include を複数書くと cpplint に引っかかるためマクロを立てて回避
    #include "..\..\..\Resources\Ui2dShaders\ui2d_BuildinShader_Gl_Source.h"
#endif

    static const uint64_t InvalidDescriptorSlotValue = 0xFFFFFFFFFFFFFFFF;

    void ResetDescriptorSlot(nn::gfx::DescriptorSlot& slot)
    {
        slot.ToData()->value = InvalidDescriptorSlotValue;
    }

    const int MinMagFilterCount = TexFilter_MaxTexFilter * TexFilter_MaxTexFilter;

    size_t GetSizeOfGraphicsResourceBuffers_(nn::gfx::Device* pDevice)
    {
        const size_t IndexBufferAlignment = font::detail::GetIndexBufferAlignment(pDevice);

        const size_t PaddingSizeForIndexBuffer = nn::util::align_up(VertexBufferSize, IndexBufferAlignment) - VertexBufferSize;
        const size_t TotalMemoryPoolSize = nn::util::align_up(VertexBufferSize + PaddingSizeForIndexBuffer + IndexBufferSize, font::detail::GetPoolMemorySizeGranularity(pDevice));

        return TotalMemoryPoolSize;
    }

    static void SetSamplerInfo_(nn::gfx::SamplerInfo* pInfo, int presetSamplerId, float lodBias)
    {
        pInfo->SetDefault();
        pInfo->SetLodBias(lodBias);

        if (presetSamplerId >= PresetSamplerId_ClampToTransparentBorderColor)
        {
            // 特殊なサンプラー
            switch (presetSamplerId)
            {
            case PresetSamplerId_ClampToTransparentBorderColor:
                pInfo->SetAddressU(nn::gfx::TextureAddressMode_ClampToBorder);
                pInfo->SetAddressV(nn::gfx::TextureAddressMode_ClampToBorder);
                pInfo->SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
                pInfo->SetBorderColorType(nn::gfx::TextureBorderColorType_TransparentBlack);
                break;
            default:
                NN_SDK_ASSERT(false, "PresetSamplerId(%d) is not recognize.\n", presetSamplerId);
                break;
            }
        }
        else
        {
            // 通常の描画に使用するサンプラー
            static const nn::gfx::TextureAddressMode wrapModeTable[TexWrap_MaxTexWrap] =
            {
                nn::gfx::TextureAddressMode_ClampToEdge,
                nn::gfx::TextureAddressMode_Repeat,
                nn::gfx::TextureAddressMode_Mirror,
            };

            static const nn::gfx::FilterMode filterModeTable[MinMagFilterCount] =
            {
                nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint,
                nn::gfx::FilterMode_MinLinear_MagPoint_MipPoint,
                nn::gfx::FilterMode_MinPoint_MagLinear_MipPoint,
                nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint,
            };

            // PresetSamplerId の並びの規則性から各パラメータを決定します。
            int wrapModeU = presetSamplerId / (TexWrap_MaxTexWrap * MinMagFilterCount);
            int wrapModeV = (presetSamplerId / MinMagFilterCount) % TexWrap_MaxTexWrap;
            int filterMode = presetSamplerId % MinMagFilterCount;

            pInfo->SetAddressU(wrapModeTable[wrapModeU]);
            pInfo->SetAddressV(wrapModeTable[wrapModeV]);
            pInfo->SetFilterMode(filterModeTable[filterMode]);
        }
    }
}

//----------------------------------------
size_t GraphicsResource::CalculateMemoryPoolSize(nn::gfx::Device* pDevice, uint32_t charMax)
{
    size_t sizeOfRectDrawerPool = nn::font::RectDrawer::CalculateMemoryPoolSize(pDevice, charMax);

    return sizeOfRectDrawerPool + GetSizeOfGraphicsResourceBuffers_(pDevice);
}

//----------------------------------------
size_t GraphicsResource::CalculateMemoryPoolAlignment(nn::gfx::Device* pDevice)
{
    // メモリープールの先頭には、RectDrawer のメモリープールを配置する。
    return nn::font::RectDrawer::CalculateMemoryPoolAlignment(pDevice);
}

size_t CalculateCombinedBufferMemoryPoolSize(nn::gfx::Device* pDevice, size_t vertexBufferSize, size_t indexBufferSize)
{
    const size_t indexBufferAlignment = font::detail::GetIndexBufferAlignment(pDevice);
    const size_t paddingSizeForIndexBuffer = nn::util::align_up(vertexBufferSize, indexBufferAlignment) - vertexBufferSize;

    return nn::util::align_up(vertexBufferSize + paddingSizeForIndexBuffer + indexBufferSize, font::detail::GetPoolMemorySizeGranularity(pDevice));
}

//----------------------------------------
void GraphicsResource::Setup(nn::gfx::Device* pDevice, int charMax,
    nn::gfx::MemoryPool* pExternalMemoryPool, ptrdiff_t externalMemoryPoolOffset, size_t externalMemoryPoolSize,
    nn::font::RectDrawer *pFontDrawer, float lodBias
)
{
    NN_SDK_ASSERT( !m_Initialized );

    // シェーダーの初期化
    {
        NN_SDK_ASSERT(sizeof(Ui2dBuildinShader) / sizeof(Ui2dBuildinShader[0]) > 1);

        // アライメント調整をおこなったバッファにコピーして初期化を行います。

        // 通常シェーダー
        {

            nn::util::BinaryFileHeader* pHeader = reinterpret_cast<nn::util::BinaryFileHeader*>(Ui2dBuildinShader);
            const size_t sdrSize = sizeof(Ui2dBuildinShader);

            m_pUi2dBuildinShader = Layout::AllocateMemory(sdrSize, pHeader->GetAlignment());
            std::memcpy(m_pUi2dBuildinShader, Ui2dBuildinShader, sizeof(Ui2dBuildinShader));

            m_CommonShaderInfo.Initialize(pDevice, m_pUi2dBuildinShader);
        }
    }

    // 描画用バッファの確保と設定
    const size_t sizeOfRectDrawerPool = pExternalMemoryPool == NULL ? 0 :
        nn::font::RectDrawer::CalculateMemoryPoolSize(pDevice, charMax);
    if (pFontDrawer == NULL)
    {
        // デフォルトの FontDrawer を初期化する
        m_pFontDrawer = Layout::AllocateAndConstruct<nn::font::RectDrawer>();
        m_IsDefaultRectDrawerUsed = true;
        if (pExternalMemoryPool == NULL)
        {
            const size_t workBufferAlignment = nn::font::RectDrawer::GetWorkBufferAlignment();
            const size_t vtxBufferSize = nn::font::RectDrawer::GetWorkBufferSize(pDevice, charMax);
            m_pFontDrawer->Initialize(pDevice, Layout::AllocateMemory(vtxBufferSize, workBufferAlignment), charMax);
        }
        else
        {
            const size_t workBufferAlignment = nn::font::RectDrawer::GetWorkBufferAlignment();
            const size_t vtxBufferSize = nn::font::RectDrawer::GetWorkBufferSize(pDevice, charMax) - sizeOfRectDrawerPool;
            m_pFontDrawer->Initialize(pDevice, Layout::AllocateMemory(vtxBufferSize, workBufferAlignment), charMax,
               pExternalMemoryPool, externalMemoryPoolOffset, sizeOfRectDrawerPool);
        }
    }
    else
    {
        // 外部から FontDrawer を設定する
        m_pFontDrawer = pFontDrawer;
        m_IsDefaultRectDrawerUsed = false;
    }
    if (pExternalMemoryPool == NULL)
    {
        InitializeVertexBuffer(pDevice, NULL, 0, 0);
    }
    else
    {
        InitializeVertexBuffer(pDevice,
            pExternalMemoryPool, externalMemoryPoolOffset + sizeOfRectDrawerPool, externalMemoryPoolSize);
    }

    {
        NN_SDK_ASSERT(m_pSamplerTable == NULL);


        m_pSamplerTable = reinterpret_cast<nn::gfx::Sampler*>(Layout::AllocateMemory( sizeof(nn::gfx::Sampler) * PresetSamplerId_Max));
        std::memset(m_pSamplerTable, 0, sizeof(nn::gfx::Sampler) * PresetSamplerId_Max);

        m_pSamplerDescriptorSlotTable = reinterpret_cast<nn::gfx::DescriptorSlot*>(Layout::AllocateMemory(sizeof(nn::gfx::DescriptorSlot) * PresetSamplerId_Max));
        std::memset(m_pSamplerDescriptorSlotTable, 0, sizeof(nn::gfx::DescriptorSlot) * PresetSamplerId_Max);

        m_SamplerLodBias = lodBias;

        nn::gfx::SamplerInfo info;

        for(int i = 0; i < PresetSamplerId_Max;++i)
        {
            SetSamplerInfo_(&info, i, lodBias);
            nn::gfx::Sampler* pSampler = m_pSamplerTable + i;
            pSampler->Initialize(pDevice, info);

            nn::gfx::DescriptorSlot* pSlot = m_pSamplerDescriptorSlotTable + i;
            ResetDescriptorSlot(*pSlot);
        }
    }

    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);

        // 各種バッファのアライメント値を保存。
        m_ConstantBufferAlignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, info);
        m_VertexBufferAlignment = nn::font::detail::GetVertexBufferAlignment(pDevice);
        m_IndexBufferAlignment = nn::font::detail::GetIndexBufferAlignment(pDevice);
    }

    // プリセットの BlendState の初期化
    const ResBlendMode* pPresetBlendMode;
    const ResBlendMode* pPresetBlendModeAlpha;
    GetPresetBlendModeArray(&pPresetBlendMode, &pPresetBlendModeAlpha);
    for (int i = 0; i < PresetBlendStateId_MaxPresetBlendStateId; i++)
    {
        nn::gfx::BlendState::InfoType blendStateInfo;
        nn::gfx::BlendTargetStateInfo blendTargetStateInfo;
        size_t size = SetupBlendStateInfo(&blendStateInfo, &blendTargetStateInfo, &pPresetBlendMode[i], &pPresetBlendModeAlpha[i]);
        m_PresetBlendState[i].SetMemory(Layout::AllocateMemory(size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment), size);
        m_PresetBlendState[i].Initialize(pDevice, blendStateInfo);
    }

    m_Initialized = true;
}// NOLINT(impl/function_size)

//----------------------------------------
nn::gfx::BlendState* GraphicsResource::GetPresetBlendState(PresetBlendStateId presetBlendStateId)
{
    NN_SDK_ASSERT(0 <= presetBlendStateId && presetBlendStateId < PresetBlendStateId_MaxPresetBlendStateId);
    return &m_PresetBlendState[presetBlendStateId];
}

//----------------------------------------
PresetBlendStateId GraphicsResource::GetPresetBlendStateId(const ResBlendMode* pBlendMode, const ResBlendMode* pBlendModeAlpha)
{
    const ResBlendMode* pPresetBlendMode;
    const ResBlendMode* pPresetBlendModeAlpha;
    GetPresetBlendModeArray(&pPresetBlendMode, &pPresetBlendModeAlpha);
    for (int i = 0; i < PresetBlendStateId_MaxPresetBlendStateId; i++)
    {
        if (pPresetBlendMode[i].Equals(pBlendMode) && pPresetBlendModeAlpha[i].Equals(pBlendModeAlpha))
        {
            return static_cast<PresetBlendStateId>(i);
        }
    }
    return PresetBlendStateId_None;
}

//----------------------------------------
size_t GraphicsResource::SetupBlendStateInfo(nn::gfx::BlendState::InfoType* pBlendStateInfo, nn::gfx::BlendTargetStateInfo* pBlendTargetStateInfo, const ResBlendMode* pBlendMode, const ResBlendMode* pBlendModeAlpha)
{
    pBlendStateInfo->SetDefault();
    pBlendTargetStateInfo->SetDefault();
    pBlendStateInfo->SetBlendTargetStateInfoArray(pBlendTargetStateInfo, 1);

    if (pBlendMode->IsLogicOpDisable())
    {
        if (pBlendMode->IsBlendOpDisable())
        {
            // ブレンド無効
            pBlendTargetStateInfo->SetBlendEnabled(false);
        }
        else
        {
            // ブレンド有効
            pBlendStateInfo->SetLogicOperationEnabled(false);
            pBlendTargetStateInfo->SetBlendEnabled(true);
            pBlendTargetStateInfo->SetSourceColorBlendFactor(pBlendMode->GetGfxSrcFactor());
            pBlendTargetStateInfo->SetDestinationColorBlendFactor(pBlendMode->GetGfxDstFactor());
            pBlendTargetStateInfo->SetSourceAlphaBlendFactor(pBlendModeAlpha->GetGfxSrcFactor());
            pBlendTargetStateInfo->SetDestinationAlphaBlendFactor(pBlendModeAlpha->GetGfxDstFactor());
            pBlendTargetStateInfo->SetColorBlendFunction(pBlendMode->GetGfxBlendOp());
            pBlendTargetStateInfo->SetAlphaBlendFunction(pBlendModeAlpha->GetGfxBlendOp());
        }
    }
    else
    {
        // COLOR_LOGIC_OP が Enable だと、BLEND は無効扱いになる。
        pBlendStateInfo->SetLogicOperationEnabled(true);
        pBlendTargetStateInfo->SetBlendEnabled(false);
        pBlendStateInfo->SetLogicOperation(pBlendMode->GetGfxLogicOp());
    }

    size_t size = nn::gfx::BlendState::GetRequiredMemorySize(*pBlendStateInfo);
    return size;
}

//----------------------------------------
void
GraphicsResource::InitializeVertexBuffer(
    nn::gfx::Device* pDevice,
    nn::gfx::MemoryPool* pExternalMemoryPool, ptrdiff_t externalMemoryPoolOffset, size_t externalMemoryPoolSize)
{
    NN_SDK_ASSERT(pExternalMemoryPool == NULL || nn::util::is_aligned(externalMemoryPoolOffset, font::detail::GetVertexBufferAlignment(pDevice)));

    const size_t indexBufferAlignment = font::detail::GetIndexBufferAlignment(pDevice);
    const size_t paddingSizeForIndexBuffer = nn::util::align_up(VertexBufferSize, indexBufferAlignment) - VertexBufferSize;

    const ptrdiff_t offsetToVertexBuffer = pExternalMemoryPool != NULL ? externalMemoryPoolOffset : 0;
    const ptrdiff_t offsetToIndexBuffer = offsetToVertexBuffer + VertexBufferSize + paddingSizeForIndexBuffer;

    const size_t totalMemoryPoolSize = nn::util::align_up(VertexBufferSize + paddingSizeForIndexBuffer + IndexBufferSize, font::detail::GetPoolMemorySizeGranularity(pDevice));
    const size_t memoryPoolAlignment = std::max(font::detail::GetVertexBufferAlignment(pDevice), font::detail::GetMemoryPoolAlignment(pDevice));

    NN_SDK_ASSERT(pExternalMemoryPool == NULL || (totalMemoryPoolSize <= externalMemoryPoolSize));
    NN_UNUSED(externalMemoryPoolSize);

    nn::gfx::MemoryPool* pMemoryPoolForBuffers = pExternalMemoryPool;
    if (pMemoryPoolForBuffers == NULL)
    {
        m_pBufferMemory = Layout::AllocateMemory(totalMemoryPoolSize, memoryPoolAlignment);

        nn::gfx::MemoryPoolInfo memPoolInfo;
        memPoolInfo.SetDefault();
        memPoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached);
        memPoolInfo.SetPoolMemory(m_pBufferMemory, totalMemoryPoolSize);

        m_MemoryPoolForBuffers.Initialize(pDevice, memPoolInfo);
        pMemoryPoolForBuffers = &m_MemoryPoolForBuffers;
    }

    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);
        info.SetSize(VertexBufferSize);

        m_VertexBuffer.Initialize(pDevice, info, pMemoryPoolForBuffers, offsetToVertexBuffer, VertexBufferSize);
        m_VertexBuffer.GetGpuAddress(&m_VertexBufferGpuAddress); // GetGpuAddress は高頻度で呼ばれる処理なのであらかじめ取得しておく

        float* pVertexBufferMemory = m_VertexBuffer.Map<float>();
        {
            pVertexBufferMemory[0] = 0.0f;
            pVertexBufferMemory[1] = 0.0f;
            pVertexBufferMemory[2] = 1.0f;
            pVertexBufferMemory[3] = 0.0f;
            pVertexBufferMemory[4] = 0.0f;
            pVertexBufferMemory[5] = 1.0f;
            pVertexBufferMemory[6] = 1.0f;
            pVertexBufferMemory[7] = 1.0f;
        }
        m_VertexBuffer.Unmap();
    }

    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_IndexBuffer);
        info.SetSize(IndexBufferSize);

        m_IndexBuffer.Initialize(pDevice, info, pMemoryPoolForBuffers, offsetToIndexBuffer, IndexBufferSize);
        m_IndexBuffer.GetGpuAddress(&m_IndexBufferGpuAddress); // GetGpuAddress は高頻度で呼ばれる処理なのであらかじめ取得しておく

        uint16_t* pIndexBufferMemory = m_IndexBuffer.Map<uint16_t>();
        {
            pIndexBufferMemory[0] = 1;
            pIndexBufferMemory[1] = 0;
            pIndexBufferMemory[2] = 3;
            pIndexBufferMemory[3] = 0;
            pIndexBufferMemory[4] = 2;
            pIndexBufferMemory[5] = 3;
        }
        m_IndexBuffer.Unmap();
    }
}

//----------------------------------------
void
GraphicsResource::ActivateVertexBuffer(nn::gfx::CommandBuffer* pCommandBuffer) const
{
    pCommandBuffer->SetVertexBuffer(0, m_VertexBufferGpuAddress, VertexBufferStride, VertexBufferSize);
}

//----------------------------------------
void GraphicsResource::GetPresetBlendModeArray(const ResBlendMode** ppPresetBlendModeArray, const ResBlendMode** ppPresetBlendModeAlphaArray)
{
    static ResBlendMode presetBlendMode[PresetBlendStateId_MaxPresetBlendStateId] =
    {
        ResBlendMode(BlendOp_Add, BlendFactor_SrcAlpha, BlendFactor_InvSrcAlpha, LogicOp_Disable), // PresetBlendStateId_Default
        ResBlendMode(BlendOp_Disable, BlendFactor_0, BlendFactor_0, LogicOp_Disable), // PresetBlendStateId_OpaqueOrAlphaTest
        ResBlendMode(BlendOp_Add, BlendFactor_SrcAlpha, BlendFactor_1, LogicOp_Disable), // PresetBlendStateId_Addition
        ResBlendMode(BlendOp_ReverseSubtract, BlendFactor_SrcAlpha, BlendFactor_1, LogicOp_Disable), // PresetBlendStateId_Subtraction
        ResBlendMode(BlendOp_Add, BlendFactor_0, BlendFactor_SrcColor, LogicOp_Disable), // PresetBlendStateId_Multiplication
        ResBlendMode(BlendOp_Add, BlendFactor_SrcAlpha, BlendFactor_InvSrcAlpha, LogicOp_Disable), // PresetBlendStateId_SemitransparencyMaxAlpha
    };

    static ResBlendMode presetBlendModeAlpha[PresetBlendStateId_MaxPresetBlendStateId] =
    {
        ResBlendMode(BlendOp_Add, BlendFactor_SrcAlpha, BlendFactor_InvSrcAlpha, LogicOp_Disable), // PresetBlendStateId_Default
        ResBlendMode(BlendOp_Disable, BlendFactor_0, BlendFactor_0, LogicOp_Disable), // PresetBlendStateId_OpaqueOrAlphaTest
        ResBlendMode(BlendOp_Add, BlendFactor_SrcAlpha, BlendFactor_1, LogicOp_Disable), // PresetBlendStateId_Addition
        ResBlendMode(BlendOp_ReverseSubtract, BlendFactor_SrcAlpha, BlendFactor_1, LogicOp_Disable), // PresetBlendStateId_Subtraction
        ResBlendMode(BlendOp_Add, BlendFactor_0, BlendFactor_SrcColor, LogicOp_Disable), // PresetBlendStateId_Multiplication
        ResBlendMode(BlendOp_SelectMax, BlendFactor_1, BlendFactor_1, LogicOp_Disable), // PresetBlendStateId_SemitransparencyMaxAlpha
    };

    (*ppPresetBlendModeArray) = presetBlendMode;
    (*ppPresetBlendModeAlphaArray) = presetBlendModeAlpha;
}

//----------------------------------------
GraphicsResource::GraphicsResource()
: m_pUi2dBuildinShader(NULL)
, m_pConstantBufferMemories(NULL)
, m_pBufferMemory(NULL)
, m_pRectShaderBinary(NULL)
, m_RectShaderBinarySize(0)
, m_ConstantBufferAlignment(4)
, m_VertexBufferAlignment(4)
, m_IndexBufferAlignment(4)
, m_pFontDrawer(NULL)
, m_pSamplerTable(NULL)
, m_pSamplerDescriptorSlotTable(NULL)
, m_SamplerLodBias(0.0f)
, m_Initialized(false)
, m_IsDefaultRectDrawerUsed(false)
{
}

//----------------------------------------
GraphicsResource::~GraphicsResource()
{
    NN_SDK_ASSERT(!m_Initialized);
}

//----------------------------------------
void
GraphicsResource::Finalize(nn::gfx::Device* pDevice)
{
    if (!m_Initialized)
    {
        return;
    }

    m_Initialized = false;

    m_IndexBuffer.Finalize(pDevice);
    m_VertexBuffer.Finalize(pDevice);

    nn::font::FinalizeIfNecessary(m_MemoryPoolForBuffers, pDevice);

    m_CommonShaderInfo.Finalize(pDevice, false);

    if (NULL != m_pRectShaderBinary)
    {
        Layout::FreeMemory(m_pRectShaderBinary);
    }
    m_pRectShaderBinary = NULL;
    m_RectShaderBinarySize = 0;

    if (m_IsDefaultRectDrawerUsed && m_pFontDrawer != NULL)
    {
        m_pFontDrawer->Finalize(pDevice);
        Layout::FreeMemory(m_pFontDrawer->GetBuffer());
        Layout::DeleteObj(m_pFontDrawer);
        m_pFontDrawer = NULL;
    }

    if (m_pBufferMemory != NULL)
    {
        Layout::FreeMemory(m_pBufferMemory);
        m_pBufferMemory = NULL;
    }
    if (m_pUi2dBuildinShader != NULL)
    {
        Layout::FreeMemory(m_pUi2dBuildinShader);
        m_pUi2dBuildinShader = NULL;
    }

    if(m_pSamplerTable != NULL)
    {
        for(int i = 0; i < PresetSamplerId_Max;++i)
        {
            nn::gfx::Sampler* pSampler = m_pSamplerTable + i;
            pSampler->Finalize(pDevice);

            nn::gfx::DescriptorSlot* pSlot = m_pSamplerDescriptorSlotTable + i;
            NN_SDK_ASSERT(pSlot->ToData()->value == InvalidDescriptorSlotValue, "You must call UnregisterCommonSamplerSlot() before Finalize().");
            NN_UNUSED(pSlot);
        }

        Layout::FreeMemory(m_pSamplerTable);
        m_pSamplerTable = NULL;
    }

    if(m_pSamplerDescriptorSlotTable != NULL)
    {
        Layout::FreeMemory(m_pSamplerDescriptorSlotTable);
        m_pSamplerDescriptorSlotTable = NULL;
    }

    for (int i = 0; i < PresetBlendStateId_MaxPresetBlendStateId; i++)
    {
        nn::gfx::BlendState* pBlendState = &m_PresetBlendState[i];
        void* pBuffer = pBlendState->GetMemory();
        pBlendState->Finalize(pDevice);
        Layout::FreeMemory(pBuffer);
    }
}

//----------------------------------------

void GraphicsResource::RegisterCommonSamplerSlot(RegisterSamplerSlot pRegisterSamplerSlot, void* pUserData)
{
    m_pFontDrawer->RegisterSamplerToDescriptorPool(pRegisterSamplerSlot, pUserData);

    {
        NN_SDK_ASSERT(m_pSamplerDescriptorSlotTable != NULL, "You must call Setup() before you use.");

        for(int i = 0; i < PresetSamplerId_Max;++i)
        {
            nn::gfx::Sampler* pSampler = m_pSamplerTable + i;
            nn::gfx::DescriptorSlot* pSlot = m_pSamplerDescriptorSlotTable + i;

            pRegisterSamplerSlot(pSlot, *pSampler, pUserData);
        }
    }
}

//----------------------------------------

void GraphicsResource::AcquireCommonSamplerSlot(AcquireSamplerSlot pAcquireSamplerSlot, void* pUserData)
{
    m_pFontDrawer->AcquireCommonSamplerSlot(pAcquireSamplerSlot, pUserData);

    {
        NN_SDK_ASSERT(m_pSamplerDescriptorSlotTable != NULL, "You must call Setup() before you use.");

        nn::gfx::SamplerInfo info;
        for (int i = 0; i < PresetSamplerId_Max; i++)
        {
            SetSamplerInfo_(&info, i, m_SamplerLodBias);
            nn::gfx::DescriptorSlot* pSlot = m_pSamplerDescriptorSlotTable + i;

            pAcquireSamplerSlot(pSlot, info, pUserData);
        }
    }
}

//----------------------------------------

void GraphicsResource::UnregisterCommonSamplerSlot(UnregisterSamplerSlot pUnregisterSamplerSlot, void* pUserData)
{
    m_pFontDrawer->UnregisterSamplerFromDescriptorPool(pUnregisterSamplerSlot, pUserData);

    {
        NN_SDK_ASSERT(m_pSamplerDescriptorSlotTable != NULL, "You must call Setup() before you use.");

        for(int i = 0; i < PresetSamplerId_Max;++i)
        {
            nn::gfx::Sampler* pSampler = m_pSamplerTable + i;
            nn::gfx::DescriptorSlot* pSlot = m_pSamplerDescriptorSlotTable + i;

            pUnregisterSamplerSlot(pSlot, *pSampler, pUserData);
            ResetDescriptorSlot(*pSlot);
        }
    }
}

//----------------------------------------

void GraphicsResource::ReleaseCommonSamplerSlot(ReleaseSamplerSlot pReleaseSamplerSlot, void* pUserData)
{
    m_pFontDrawer->ReleaseCommonSamplerSlot(pReleaseSamplerSlot, pUserData);

    {
        NN_SDK_ASSERT(m_pSamplerDescriptorSlotTable != NULL, "You must call Setup() before you use.");

        nn::gfx::SamplerInfo info;
        for (int i = 0; i < PresetSamplerId_Max; i++)
        {
            SetSamplerInfo_(&info, i, m_SamplerLodBias);
            nn::gfx::DescriptorSlot* pSlot = m_pSamplerDescriptorSlotTable + i;

            pReleaseSamplerSlot(pSlot, info, pUserData);
            ResetDescriptorSlot(*pSlot);
        }
    }
}

//----------------------------------------

nn::gfx::DescriptorSlot& GraphicsResource::GetSamplerDescriptorSlot(TexWrap wrapS, TexWrap wrapT, TexFilter minFilter, TexFilter magFilter) const
{
    NN_SDK_ASSERT(wrapS < TexWrap_MaxTexWrap);
    NN_SDK_ASSERT(wrapT < TexWrap_MaxTexWrap);
    NN_SDK_ASSERT(minFilter <= TexFilter_MaxTexFilter);
    NN_SDK_ASSERT(magFilter <= TexFilter_MaxTexFilter);

    NN_SDK_ASSERT(m_pSamplerDescriptorSlotTable != NULL, "You must call Setup() before you use.");

    const int filterIdx =  minFilter + magFilter * TexFilter_MaxTexFilter;
    const int index = wrapS * TexWrap_MaxTexWrap * MinMagFilterCount + wrapT * MinMagFilterCount + filterIdx;

    NN_SDK_ASSERT(index <= PresetSamplerId_MirrorU_MirrorV_MinLinear_MagLinaer_MipPoint);

    nn::gfx::DescriptorSlot* pSlot = m_pSamplerDescriptorSlotTable + index;

    NN_SDK_ASSERT(pSlot->ToData()->value != InvalidDescriptorSlotValue, "You must call RegisterCommonSamplerSlot() before you use.");

    return *pSlot;
}

//----------------------------------------

nn::gfx::DescriptorSlot& GraphicsResource::GetSamplerDescriptorSlot(PresetSamplerId id) const
{
    NN_SDK_ASSERT(id < PresetSamplerId_Max);

    nn::gfx::DescriptorSlot* pSlot = m_pSamplerDescriptorSlotTable + id;

    NN_SDK_ASSERT(pSlot->ToData()->value != InvalidDescriptorSlotValue, "You must call RegisterCommonSamplerSlot() before you use.");

    return *pSlot;
}



} // namespace nn::ui2d
} // namespace nn
