﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_ButtonBase.h>

namespace nn
{
namespace ui2d
{

ButtonBase::ButtonBase()
: m_State(State_OffIdle)
{
    m_Flag.Clear();
    m_Flag.SetAllBitOn(FlagMask_AcceptOn | FlagMask_AcceptOff | FlagMask_AcceptDown | FlagMask_AcceptCancel | FlagMask_Active);
}

ButtonBase::~ButtonBase()
{
}

void ButtonBase::On()
{
    if (IsActive() && IsAcceptAction(Action_On)) {
        m_ActionQueue.PushWithOmit(Action_On);
    }
}

void ButtonBase::Off()
{
    if (IsAcceptAction(Action_Off)) {
        m_ActionQueue.PushWithOmit(Action_Off);
    }
}

void ButtonBase::Down()
{
    if (IsActive() && IsAcceptAction(Action_Down)) {
        m_ActionQueue.PushWithOmit(Action_Down);
    }
}

void ButtonBase::Cancel()
{
    if (IsAcceptAction(Action_Cancel)) {
        m_ActionQueue.PushWithOmit(Action_Cancel);
    }
}

void ButtonBase::ForceOff()
{
    m_ActionQueue.Clear();
    ForceChangeState(State_OffIdle);
}

void ButtonBase::ForceOn()
{
    m_ActionQueue.Clear();
    ForceChangeState(State_OnIdle);
}

void ButtonBase::ForceDown()
{
    m_ActionQueue.Clear();
    ForceChangeState(State_DownIdle);
}

void ButtonBase::Update()
{
    ProcessActionFromQueue();

    switch (m_State) {
    case State_OffIdle:
        break;
    case State_On:
        if (UpdateOn()) {
            FinishOn();
            ProcessActionFromQueue();
        }
        break;
    case State_Off:
        if (UpdateOff()) {
            FinishOff();
            ProcessActionFromQueue();
        }
        break;
    case State_OnIdle:
        break;
    case State_Down:
        if (UpdateDown()) {
            FinishDown();
            ProcessActionFromQueue();
        }
        break;
    case State_DownIdle:
        break;
    case State_Cancel:
        if (UpdateCancel()) {
            FinishCancel();
            ProcessActionFromQueue();
        }
        break;
    default:
        NN_SDK_ASSERT(false, "Unknown State.");
        break;
    };
}

void ButtonBase::SetActive(bool active)
{
    m_Flag.SetMaskedBits(FlagMask_Active, active ? FlagMask_Active : 0);
}

bool ButtonBase::ProcessOn()
{
    switch (m_State) {
    case State_OffIdle:
        ChangeState(State_On);
        StartOn();
        return true;
    case State_On:
        return true;
    case State_Off:
        ChangeState(State_On);
        StartOn();
        return true;
    case State_OnIdle:
        return true;
    case State_Down:
        return false;
    case State_DownIdle:
        return true;
    case State_Cancel:
        return false;
    default:
        NN_SDK_ASSERT(false, "Unknown State.");
        return true;
    };
}

bool ButtonBase::ProcessOff()
{
    switch (m_State) {
    case State_OffIdle:
        return true;
    case State_On:
        ChangeState(State_Off);
        StartOff();
        return true;
    case State_Off:
        return true;
    case State_OnIdle:
        ChangeState(State_Off);
        StartOff();
        return true;
    case State_Down:
        return false;
    case State_DownIdle:
        ChangeState(State_Off);
        StartOff();
        return true;
    case State_Cancel:
        return true;
    default:
        NN_SDK_ASSERT(false, "Unknown State.");
        return true;
    };
}

bool ButtonBase::ProcessDown()
{
    switch (m_State) {
    case State_OffIdle:
        ChangeState(State_On);
        StartOn();
        return false;
    case State_On:
        return false;
    case State_Off:
        ChangeState(State_On);
        StartOn();
        return false;
    case State_OnIdle:
        ChangeState(State_Down);
        StartDown();
        return true;
    case State_Down:
        return true;
    case State_DownIdle:
        return true;
    case State_Cancel:
        return false;
    default:
        NN_SDK_ASSERT(false, "Unknown State.");
        return true;
    };
}

bool ButtonBase::ProcessCancel()
{
    switch (m_State) {
    case State_OffIdle:
        return true;
    case State_On:
        return true;
    case State_Off:
        return true;
    case State_OnIdle:
        return true;
    case State_Down:
        return false;
    case State_DownIdle:
        ChangeState(State_Cancel);
        StartCancel();
        return true;
    case State_Cancel:
        return true;
    default:
        NN_SDK_ASSERT(false, "Unknown State.");
        return true;
    };
}

bool ButtonBase::UpdateOn()
{
    return true;
}

bool ButtonBase::UpdateOff()
{
    return true;
}

bool ButtonBase::UpdateDown()
{
    return true;
}

bool ButtonBase::UpdateCancel()
{
    return true;
}

void ButtonBase::StartOn()
{
}

void ButtonBase::StartOff()
{
}

void ButtonBase::StartDown()
{
}

void ButtonBase::StartCancel()
{
}

void ButtonBase::FinishOn()
{
    ChangeState(State_OnIdle);
}

void ButtonBase::FinishOff()
{
    ChangeState(State_OffIdle);
}

void ButtonBase::FinishDown()
{
    ChangeState(State_DownIdle);
}

void ButtonBase::FinishCancel()
{
    ChangeState(State_OffIdle);
}

void ButtonBase::ChangeState(State state)
{
    m_State = state;
}

void ButtonBase::ForceChangeState(State state)
{
    m_State = state;
}

void ButtonBase::ProcessActionFromQueue()
{
    if ( ! m_ActionQueue.IsEmpty()) {
        bool isPop = false;
        switch (m_ActionQueue.Peek()) {
        case Action_On:
            isPop = ProcessOn();
            break;
        case Action_Off:
            isPop = ProcessOff();
            break;
        case Action_Down:
            isPop = ProcessDown();
            break;
        case Action_Cancel:
            isPop = ProcessCancel();
            break;
        default:
            NN_SDK_ASSERT(false, "Unknown Action.");
            break;
        }
        if (isPop) {
            m_ActionQueue.Pop();
        }
    }
}

bool ButtonBase::IsDowning() const
{
    if (m_State == State_Down || m_State == State_DownIdle) {
        return true;
    } else {
        return m_ActionQueue.HasDownAction();
    }
}

//! キューの中にDOWNアクションがあるか調査します。
bool ButtonBase::ActionQueue::HasDownAction() const
{
    for (int i = 0; i < m_ActionCount; i++) {
        if (m_Actions[i] == Action_Down) {
            return true;
        }
    }
    return false;
}

void ButtonBase::ActionQueue::PushWithOmit(Action action)
{
    /*
    アクションキューの中に追加されるのと同じアクションがないか調べ、あるときは
    アクションを追加する代わりに、そのアクションより後のアクションを削除する。
    ないときは、単純に一番後ろに追加する。

    例えば

    ┌────┐
    │         │
    ├────┤
    │   ON    │
    ├────┤
    │  OFF   │
    ├────┤
    │   DOWN  │ <= 先頭のアクション
    └────┘

    の状態でDOWNアクションが入ってきた時は、OFFとONを削除して

    ┌────┐
    │         │
    ├────┤
    │         │
    ├────┤
    │        │
    ├────┤
    │   DOWN  │
    └────┘

    とする。これは、感覚的には「ＡをしてＢをしてＣをしてＡをする」というのを
    結局「Ａをする」というものに置き換えるということである。
    この仕様により、アクションキューには同じアクションは１つまでしか存在しない。
    さらに、アクションキューの最大要素数がアクションの数（現在は４つ）だけ必要である理由にもなっている。
    */
    for (int i = 0; i < m_ActionCount; i++) {
        if (m_Actions[i] == action) {
            m_ActionCount = i + 1;
            return;
        }
    }

    // 同じアクションはアクションキューには一つしか存在しない、ということはこの時点でキューがいっぱい
    // ということはないはずだが、念のため確認する。
    if (m_ActionCount < QueueSize) {
        m_Actions[m_ActionCount] = action;
        m_ActionCount++;
    } else {
        NN_SDK_ASSERT(false, "action queue is full");
    }
}

void ButtonBase::ActionQueue::Pop()
{
    if (m_ActionCount > 0) {
        for (int i = 0; i < m_ActionCount - 1; i++) {
            m_Actions[i] = m_Actions[i + 1];
        }
        m_ActionCount--;
    } else {
        NN_SDK_ASSERT(false, "action queue is empty");
    }
}

} // namespace nn::ui2d
} // namespace nn
