﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>

#include <nn/nn_SdkLog.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>

#include <nn/font/font_ResFont.h>

#include <nn/ui2d/ui2d_ArcResourceAccessor.h>
#include <nn/ui2d/ui2d_Resources.h>
#include <nn/ui2d/detail/ui2d_Log.h>

namespace nn
{
namespace ui2d
{

namespace
{

const int ResourcePathLengthMax = 255;

//----------------------------------------
void*
GetResourceSub(
    ArchiveHandle* pArcHandle,
    const char* resRootDir,
    nn::ui2d::ResType resType,
    const char* pName,
    size_t* pSize
)
{
    NN_SDK_ASSERT(resType != 0, "please specify resource type. name = %s", pName);

    int32_t entryNum = -1;

    {
        char resTypeStr[5];
        resTypeStr[0] = uint8_t (resType >> 24);
        resTypeStr[1] = uint8_t (resType >> 16);
        resTypeStr[2] = uint8_t (resType >>  8);
        resTypeStr[3] = uint8_t (resType >>  0);
        resTypeStr[4] = 0;

        char pathName[ResourcePathLengthMax + 1];
        if (resRootDir[0] == '.' && resRootDir[1] == '\0')
        {
            // ルート指定がトップになっている場合
            nn::util::SNPrintf(pathName, ResourcePathLengthMax, "%s/%s", resTypeStr, pName);
        }
        else
        {
            // ルート指定がトップ以外の場合
            nn::util::SNPrintf(pathName, ResourcePathLengthMax,  "%s/%s/%s", resRootDir, resTypeStr, pName);
        }
        entryNum = pArcHandle->GetArcExtractor()->ConvertPathToEntryId(pathName);
    }

    if (entryNum != -1)
    {
        ArcFileInfo fileInfo;
        void* pRes = pArcHandle->GetArcExtractor()->GetFileFast(&fileInfo, entryNum);
        NN_SDK_ASSERT_NOT_NULL(pRes);
        if (pSize)
        {
            *pSize = fileInfo.GetLength();
        }

        return pRes;
    }

    return NULL;
}

//----------------------------------------
void
SearchAllResource(
    const ArchiveHandle* pArcHandle,
    const char* pResRootDir,
    nn::ui2d::ResType resType,
    void(*callback)(const void* pRes, size_t size, const char* pName, void* pParam), void* pParam
)
{
    // リソースの種類をパスに直しておく
    char pathName[ResourcePathLengthMax + 1];
    {
        char resTypeStr[5];
        resTypeStr[0] = uint8_t (resType >> 24);
        resTypeStr[1] = uint8_t (resType >> 16);
        resTypeStr[2] = uint8_t (resType >>  8);
        resTypeStr[3] = uint8_t (resType >>  0);
        resTypeStr[4] = 0;

        if (pResRootDir[0] == '.' && pResRootDir[1] == '\0')
        {
            // ルート指定がトップになっている場合
            nn::util::SNPrintf(pathName, ResourcePathLengthMax, "%s/", resTypeStr);
        }
        else
        {
            // ルート指定がトップ以外の場合
            nn::util::SNPrintf(pathName, ResourcePathLengthMax,  "%s/%s/", pResRootDir, resTypeStr);
        }
    }

    // すべてのリソースを検索
    const nn::ui2d::ArcExtractor* pArcExtractor = pArcHandle->GetArcExtractor();
    int handle = 0;
    for (; ; )
    {
        nn::ui2d::ArcEntry entry;
        const int entryId = handle;
        int result = pArcExtractor->ReadEntry(&handle, &entry, 1);
        if (result == 0)
        {
            break;
        }
        // ファイル名を除くパスが一致しているものは検索対象のファイルタイプ。
        if (strstr(entry.name, pathName) == entry.name)
        {
            ArcFileInfo fileInfo;
            const void* pRes = pArcHandle->GetArcExtractor()->GetFileFast(&fileInfo, entryId);
            callback(pRes, fileInfo.GetLength(), entry.name, pParam);
        }
    }
}

//----------------------------------------

bool IsResTextureFileInitialized(nn::gfx::ResTextureFile* pResTextureFile)
{
    if(pResTextureFile == NULL)
    {
        return false;
    }

    const nn::gfx::MemoryPool* pMemPool = reinterpret_cast<nn::gfx::MemoryPool*>(pResTextureFile->ToData().textureContainerData.pTextureMemoryPool.Get());
    return pMemPool->ToData()->state == nn::gfx::MemoryPool::DataType::State_Initialized;
}

//----------------------------------------
bool IsIndividualArchiveShaderVariationTable(int firstBlend)
{
    if (firstBlend == IndividualArchiveShaderVariationTableFirstBlendWithDetailedCombiner ||
        firstBlend == IndividualArchiveShaderVariationTableFirstBlendWithCombinerUserShader)
    {
        return true;
    }
    return false;
}

//----------------------------------------
const char* GetArchiveShaderRegistrationName(int firstBlend, bool isCombined, const char* pName)
{
    if (isCombined)
    {
        if (firstBlend == IndividualArchiveShaderVariationTableFirstBlendWithDetailedCombiner)
        {
            return ArchiveShaderRegistrationNameWithDetailedCombiner;
        }
        else if (firstBlend == IndividualArchiveShaderVariationTableFirstBlendWithCombinerUserShader)
        {
            return ArchiveShaderRegistrationNameWithCombinerUserShader;
        }
        else
        {
            return ArchiveShaderRegistrationName;
        }
    }
    else
    {
        return pName;
    }
}


}   // namespace

//----------------------------------------
ArchiveHandle::ArchiveHandle()
    : m_ArcExtractor()
    , m_FontList()
    , m_TextureList()
    , m_ShaderList()
    , m_pResTextureFile(NULL)
    , m_pResArchiveShaderFile(NULL)
    , m_pResArchiveShaderVariationTableFile(NULL)
    , m_pArchiveBinaryTop(NULL)
    , m_pMemoryPool(NULL)
    , m_MemoryPoolOffset(0)
    , m_MemoryPoolSize(0)
{
}

//----------------------------------------
ArchiveHandle::~ArchiveHandle()
{
}

//----------------------------------------
bool ArchiveHandle::Initialize(void* archiveStart, const char* pResourceRootDirectory, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize)
{
    NN_SDK_ASSERT_NOT_NULL(archiveStart);
    NN_SDK_ASSERT_NOT_NULL(pResourceRootDirectory);

    bool bSuccess = m_ArcExtractor.PrepareArchive(archiveStart);

    if (! bSuccess)
    {
        return false;
    }

    // ルートディレクトリ文字列のコピー
    nn::util::Strlcpy(m_ResRootDir, pResourceRootDirectory, RootPathLengthMax);

    m_pArchiveBinaryTop = archiveStart;
    m_pMemoryPool = pMemoryPool;
    m_MemoryPoolOffset = memoryPoolOffset;
    m_MemoryPoolSize = memoryPoolSize;

    return true;
}

//----------------------------------------
void ArchiveHandle::Finalize(nn::gfx::Device* pDevice)
{
    m_pResArchiveShaderVariationTableFile = NULL;
    m_pResArchiveShaderFile = NULL;

    m_FontList.Finalize(pDevice);
    m_TextureList.Finalize(pDevice);
    m_ShaderList.Finalize(pDevice);

    if(m_pResTextureFile != NULL)
    {
        m_pResTextureFile->Finalize(pDevice);
        m_pResTextureFile = NULL;
    }
}

//----------------------------------------
void ArchiveHandle::InitializeBntxIfNeeded(nn::gfx::Device* pDevice)
{
    if(m_pResTextureFile != NULL)
    {
        return;
    }

    void* pTexRes = NULL;
    pTexRes = GetResourceSub(this, this->GetResRootDir(), ResourceTypeTexture, "__Combined.bntx", NULL);
    if (pTexRes != NULL)
    {
        m_pResTextureFile = nn::gfx::ResTextureFile::ResCast(pTexRes);
        if(!IsResTextureFileInitialized(m_pResTextureFile))
        {
            if (m_pMemoryPool == NULL)
            {
                m_pResTextureFile->Initialize(pDevice);
            }
            else
            {
                nn::util::BytePtr binTop(m_pArchiveBinaryTop);
                m_pResTextureFile->Initialize(pDevice, m_pMemoryPool, binTop.Distance(m_pResTextureFile) + m_MemoryPoolOffset, m_MemoryPoolSize);
            }
        }
    }
}

//----------------------------------------
void ArchiveHandle::LoadTextureAll(nn::gfx::Device* pDevice)
{
    InitializeBntxIfNeeded(pDevice);

    // bntx の テクスチャをすべて初期化します。
    if (m_pResTextureFile != NULL)
    {
        const int TexCount = m_pResTextureFile->GetTextureDic()->GetCount();
        for(int index = 0; index < TexCount; index++)
        {
            nn::gfx::ResTexture* pTexRes = m_pResTextureFile->GetResTexture(index);
            if(pTexRes == NULL)
            {
                continue;
            }

            const char* pTexName = pTexRes->GetName();
            ResourceTextureInfo* pTexInfo = m_TextureList.RegisterResourceTexture(pTexName);
            if (!ui2d::LoadTexture(pTexInfo, pDevice, pTexRes))
            {
                NN_DETAIL_UI2D_ERROR("texture object unavailable. - %s", pTexName);
            }
        }
    }
}

//----------------------------------------
void ArchiveHandle::LoadShaderAll(nn::gfx::Device* pDevice)
{
    nn::ui2d::ResType resType = nn::ui2d::ResourceTypeShader;
    char resTypeStr[6];
    resTypeStr[0] = uint8_t (resType >> 24);
    resTypeStr[1] = uint8_t (resType >> 16);
    resTypeStr[2] = uint8_t (resType >>  8);
    resTypeStr[3] = uint8_t (resType >>  0);
    resTypeStr[4] = '/';
    resTypeStr[5] = 0;

    int  dir = 0;
    nn::ui2d::ArcEntry dirEntry;
    char temp[128];
    char shaderName[8];
    ArcExtractor* pArcExt = GetArcExtractor();

    while (pArcExt->ReadEntry(&dir, &dirEntry, 1))
    {
        // ResourceTypeShaderファイルをさがす
        const char* pSubstr = std::strstr(dirEntry.name, resTypeStr);
        if (pSubstr)
        {
            {
                nn::util::Strlcpy(temp, pSubstr + strlen(resTypeStr), sizeof(temp));
                const char* pShaderName = std::strstr(temp, ResourceAccessor::ArchiveShaderPrefix);
                if(pShaderName == NULL)
                {
                    continue;
                }

                pShaderName = pShaderName + std::strlen(ResourceAccessor::ArchiveShaderPrefix);

                const char* pShaderNameEnd = std::strstr(pShaderName, ResourceAccessor::ArchiveShaderSuffix);
                if(pShaderNameEnd == NULL)
                {
                    continue;
                }

                // 3 枚ブレンド用シェーダのシェーダ名(__ArchiveShader.bnsh)にはハイフンが含まれていないためマッチングを「ArchiveShader*.bnsh」で行っているが、
                // ユーザシェーダの場合のシェーダ名の切り出しは「ArchiveShader-*.bnsh」とする必要があるため、1 文字進める。
                if (strcmp(temp, ArchiveShaderFileName) != 0)
                {
                    pShaderName++;
                }
                std::memset(shaderName, 0, sizeof(shaderName));
                nn::util::Strlcpy(shaderName, pShaderName, static_cast<int>(pShaderNameEnd - pShaderName + 1));
            }

            ShaderInfo* pShaderInfo;

            void* pShaderRes = pArcExt->FindFile(dirEntry.name);
            void* pVariationTable = NULL;
            if (strcmp(temp, ArchiveShaderFileName) == 0)
            {
                // 統合済みシェーダアーカイブを m_ShaderList に登録する。
                // 統合済みシェーダアーカイブのファイル名を temp に構築して登録。
                pShaderInfo = m_ShaderList.RegisterShader(ArchiveShaderRegistrationName, true);
                std::strcpy(temp, resTypeStr);
                std::strncat(temp, ArchiveShaderVariationTableFileName, 128 - strlen(temp) - 1);

                pVariationTable = pArcExt->FindFile(temp);
            }
            else
            {
                // ArchiveShader-%s.bnsh という命名ルールで、%s のところだけを取り出して m_ShaderList に登録する際のリソース名とする。
                pShaderInfo = m_ShaderList.RegisterShader(shaderName, true);
            }

            nn::util::BytePtr binTop(m_pArchiveBinaryTop);
            ui2d::LoadArchiveShader(pShaderInfo, pDevice, pShaderRes, pVariationTable, m_pMemoryPool, binTop.Distance(pShaderRes) + m_MemoryPoolOffset, m_MemoryPoolSize);
        }
    }
}

//----------------------------------------
bool ArchiveHandle::LoadTexture(ResourceTextureInfo* pTexInfo, nn::gfx::Device* pDevice, const char *pName)
{
    // まず、bntx を探してその中からテクスチャが無いか探します。
    InitializeBntxIfNeeded(pDevice);

    void*   pTexRes = NULL;

    // bntx から テクスチャを取得します。
    if (m_pResTextureFile != NULL)
    {
        int resTextureIndex = m_pResTextureFile->GetTextureDic()->FindIndex(pName);
        if(resTextureIndex != -1)
        {
            pTexRes = m_pResTextureFile->GetResTexture(resTextureIndex);
        }
    }

    // bntx から発見できない場合、指定された名前そのままで検索する
    if(pTexRes == NULL)
    {
        pTexRes = GetResourceSub(this, this->GetResRootDir(), ResourceTypeTexture, pName, NULL);
        return false;
    }

    if (!ui2d::LoadTexture(pTexInfo, pDevice, pTexRes))
    {
        NN_DETAIL_UI2D_ERROR("texture object unavailable. - %s", pName);
        return false;
    }

    return true;
}


//----------------------------------------
nn::font::Font* ArchiveHandle::LoadFont(nn::gfx::Device* pDevice, const char *pName)
{
    void* pFontRes = GetResourceSub(this, this->GetResRootDir(), ResourceTypeFont, pName, NULL);
    if (!pFontRes)
    {
        NN_DETAIL_UI2D_ERROR("font resource not found. - %s", pName);
        return NULL;
    }

    nn::font::ResFont* pResFont = Layout::AllocateAndConstruct<nn::font::ResFont>();
    if (pResFont == NULL)
    {
        NN_DETAIL_UI2D_ERROR("font object creation failed.");
        return NULL;
    }

    nn::util::BytePtr binTop(m_pArchiveBinaryTop);
    bool bSuccess = pResFont->SetResource(pDevice, pFontRes,
        m_pMemoryPool, binTop.Distance(pFontRes) + m_MemoryPoolOffset, m_MemoryPoolSize);
    if (!bSuccess)
    {
        NN_DETAIL_UI2D_ERROR("Fail to load ResFont.");
        Layout::DeleteObj(pResFont);
        return NULL;
    }

    return pResFont;
}

//----------------------------------------
bool ArchiveHandle::LoadShader(ShaderInfo* pShaderInfo, nn::gfx::Device* pDevice, const char *pName)
{
    int firstBlend;
    int secondBlend;
    bool isCombined = ConvertArchiveShaderNameToBlends(&firstBlend, &secondBlend, pName);

    if (isCombined)
    {
        // 統合済みアーカイブシェーダから探します。
        if (m_pResArchiveShaderFile == NULL)
        {
            m_pResArchiveShaderFile = GetResourceSub(this, this->GetResRootDir(), ResourceTypeShader, ArchiveShaderFileName, NULL);
        }

        if (!IsIndividualArchiveShaderVariationTable(firstBlend))
        {
            void* pShaderRes = m_pResArchiveShaderFile;
            if (pShaderRes != NULL)
            {
                if (m_pResArchiveShaderVariationTableFile == NULL)
                {
                    m_pResArchiveShaderVariationTableFile = GetResourceSub(this, this->GetResRootDir(), ResourceTypeShader, ArchiveShaderVariationTableFileName, NULL);
                }
                void* pVariationTable = m_pResArchiveShaderVariationTableFile;
                if (pVariationTable != NULL)
                {
                    const int variationIndex = SearchShaderVariationIndexFromTable(pVariationTable, firstBlend, secondBlend);
                    if (variationIndex != -1)
                    {
                        nn::util::BytePtr binTop(m_pArchiveBinaryTop);
                        ui2d::LoadArchiveShader(pShaderInfo, pDevice, pShaderRes, pVariationTable, m_pMemoryPool, binTop.Distance(pShaderRes) + m_MemoryPoolOffset, m_MemoryPoolSize);

                        return true;
                    }
                }
            }
            // MultiArcResourceAccessor 内から探索する場合は、ある ArchiveHandle に存在していないことも
            // 正常にありえるため、見つからなくても警告は出さない。
            // NN_DETAIL_UI2D_ERROR("shader resource not found. - %s\n", pName);
        }
        else
        {
            // 詳細コンバイナ（バリエーション定数版）
            int stageBits[DetailedCombinerStageBitsCountWithVariationTable];
            ConvertArchiveShaderDetailedCombinerNameToStageBits(stageBits, pName);

            void* pShaderRes = m_pResArchiveShaderFile;
            if (pShaderRes != NULL)
            {
                //! バリエーション変数版の詳細コンバイナ用バリエーションテーブルのファイル名
                char individualArchiveShaderVariationTableName[ResourcePathLengthMax];
                std::sprintf(individualArchiveShaderVariationTableName, "%s_%d.%s",
                    IndividualArchiveShaderVariationTableBaseName,
                    firstBlend,
                    IndividualArchiveShaderVariationTableExtensionName);

                void* pDetailedCombinerVariationTable = GetResourceSub(this, this->GetResRootDir(), ResourceTypeShader, individualArchiveShaderVariationTableName, NULL);
                if (pDetailedCombinerVariationTable)
                {
                    const int variationIndex = SearchShaderVariationDetailedCombinerIndexFromTable(pDetailedCombinerVariationTable, stageBits);
                    if (variationIndex != -1)
                    {
                        const int texMapCount = firstBlend == IndividualArchiveShaderVariationTableFirstBlendWithCombinerUserShader ? CombinerUserShaderTexMapMax : TexMapMax;
                        // 拡張用のテーブル利用時は pVariationExTable が GetVariationTable() から得られる事に注意してください。
                        nn::util::BytePtr binTop(m_pArchiveBinaryTop);
                        ui2d::LoadArchiveShader(pShaderInfo, pDevice, pShaderRes, pDetailedCombinerVariationTable, m_pMemoryPool, binTop.Distance(pShaderRes) + m_MemoryPoolOffset, m_MemoryPoolSize, texMapCount);
                        return true;
                    }
                }
            }
        }
    }
    else
    {
        // 指定された名前そのままで検索します。
        char str[256];
        ResourceAccessor::GetArchiveShaderResourceName(str, pName, sizeof(str));
        void* pShaderRes = GetResourceSub(this, this->GetResRootDir(), ResourceTypeShader, str, NULL);
        if (pShaderRes != NULL)
        {
            ui2d::LoadArchiveShader(pShaderInfo, pDevice, pShaderRes, NULL);
            return true;
        }
        NN_DETAIL_UI2D_ERROR("shader resource not found. - %s\n", str);
    }

    return false;
}

//----------------------------------------
ArcExtractor* ArchiveHandle::GetArcExtractor()
{
    return &m_ArcExtractor;
}

//----------------------------------------
const ArcExtractor* ArchiveHandle::GetArcExtractor() const
{
    return &m_ArcExtractor;
}

//----------------------------------------
FontContainer* ArchiveHandle::GetFontList()
{
    return &m_FontList;
}

//----------------------------------------
TextureContainer* ArchiveHandle::GetTextureList()
{
    return &m_TextureList;
}

//----------------------------------------
ShaderContainer* ArchiveHandle::GetShaderList()
{
    return &m_ShaderList;
}

//----------------------------------------
void ArchiveHandle::RegisterFont(const char* pName, nn::font::Font* pFont)
{
    m_FontList.RegisterFont(pName, pFont, true);
}

//----------------------------------------
ResourceTextureInfo* ArchiveHandle::RegisterTexture(const char* pName)
{
    return m_TextureList.RegisterResourceTexture(pName);
}

//----------------------------------------
ShaderInfo* ArchiveHandle::RegisterShader(const char* pName)
{
    return m_ShaderList.RegisterShader(pName, true);
}

//----------------------------------------
void ArchiveHandle::UnregisterAll()
{
    m_FontList.clear();
    m_TextureList.clear();
    m_ShaderList.clear();
}

//----------------------------------------
void ArchiveHandle::RegisterTextureViewToDescriptorPool(RegisterTextureViewSlot pRegisterTextureViewSlot, void* pUserData)
{
    m_FontList.RegisterTextureViewToDescriptorPool(pRegisterTextureViewSlot, pUserData);
    m_TextureList.RegisterTextureViewToDescriptorPool(pRegisterTextureViewSlot, pUserData);
}

//----------------------------------------
void ArchiveHandle::UnregisterTextureViewFromDescriptorPool(UnregisterTextureViewSlot pUnregisterTextureViewSlot, void* pUserData)
{
    m_FontList.UnregisterTextureViewFromDescriptorPool(pUnregisterTextureViewSlot, pUserData);
    m_TextureList.UnregisterTextureViewFromDescriptorPool(pUnregisterTextureViewSlot, pUserData);
}

//----------------------------------------
const void* ArchiveHandle::GetArchiveDataStart() const
{
    return m_ArcExtractor.GetArchiveDataStart();
}

//----------------------------------------
const char* ArchiveHandle::GetResRootDir() const
{
    return m_ResRootDir;
}

//----------------------------------------
ArcResourceAccessor::ArcResourceAccessor()
:   m_ArcBuf(0)
{
    m_ResRootDir[0] = 0;
}

//----------------------------------------
ArcResourceAccessor::~ArcResourceAccessor()
{
    NN_SDK_ASSERT(m_UserRegisteredFontList.empty() && m_UserRegisteredTextureList.empty() && m_UserRegisteredShaderList.empty(), "ArcResourceAccessor::Finalize() must be called before destructor.");
}

//----------------------------------------
void ArcResourceAccessor::Finalize(nn::gfx::Device* pDevice)
{
    m_ArcHandle.Finalize(pDevice);

    m_UserRegisteredFontList.Finalize(pDevice);
    m_UserRegisteredTextureList.Finalize(pDevice);
    m_UserRegisteredShaderList.Finalize(pDevice);

    ResourceAccessor::Finalize(pDevice);
}

//----------------------------------------
bool
ArcResourceAccessor::Attach(
    void*       archiveStart,
    const char* pResourceRootDirectory,
    nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize
)
{
    NN_SDK_ASSERT(! IsAttached());
    NN_SDK_ASSERT_NOT_NULL(archiveStart);
    NN_SDK_ASSERT_NOT_NULL(pResourceRootDirectory);

    bool bSuccess = m_ArcHandle.Initialize(archiveStart, pResourceRootDirectory, pMemoryPool, memoryPoolOffset, memoryPoolSize);
    if (! bSuccess)
    {
        return false;
    }

    m_ArcBuf = archiveStart;
    // ルートディレクトリ文字列のコピー
    nn::util::Strlcpy(m_ResRootDir, pResourceRootDirectory, RootPathLengthMax);

    return true;
}

//----------------------------------------
void*
ArcResourceAccessor::Detach()
{
    NN_SDK_ASSERT(IsAttached());

    void* ret = m_ArcBuf;
    m_ArcBuf = 0;
    return ret;
}

//----------------------------------------
void*
ArcResourceAccessor::FindResourceByName(
    size_t *  pSize,
    ResType     resType,
    const char* pName
)
{
    return GetResourceSub(&m_ArcHandle, m_ResRootDir, resType, pName, pSize);
}

//----------------------------------------
void
ArcResourceAccessor::FindResourceByType(
    ResType resType,
    FindResourceByTypeCallback pCallback,
    void* pParam
) const
{
    SearchAllResource(&m_ArcHandle, m_ResRootDir, resType, pCallback, pParam);
}

//----------------------------------------
bool ArcResourceAccessor::LoadTexture(ResourceTextureInfo* pTexInfo, nn::gfx::Device* pDevice, const char *pName)
{
    bool result = m_ArcHandle.LoadTexture(pTexInfo, pDevice, pName);
    if (!result)
    {
        NN_DETAIL_UI2D_ERROR("texture resource not found. - %s", pName);
    }

    return result;
}

//----------------------------------------
nn::font::Font* ArcResourceAccessor::LoadFont(nn::gfx::Device* pDevice, const char* pName)
{
    nn::font::Font* pFont = m_ArcHandle.LoadFont(pDevice, pName);
    if (pFont == NULL)
    {
        NN_DETAIL_UI2D_ERROR("font resource not found. - %s", pName);
    }

    return pFont;
}

//----------------------------------------
bool ArcResourceAccessor::LoadShader(ShaderInfo* pShaderInfo, nn::gfx::Device* pDevice, const char *pName)
{
    const bool result = m_ArcHandle.LoadShader(pShaderInfo, pDevice, pName);
    if (!result)
    {
        NN_DETAIL_UI2D_ERROR("shader resource not found. - %s", pName);
    }
    return result;
}

//----------------------------------------
nn::font::Font*
ArcResourceAccessor::AcquireFont(nn::gfx::Device* pDevice, const char *pName)
{
    // ユーザがリソースをロードしていれば使う
    nn::font::Font* pFont = m_UserRegisteredFontList.FindFontByName(pName);
    if (pFont != NULL)
    {
        return pFont;
    }

    // アーカイブ内のリソースがロードされていれば使う
    pFont = m_ArcHandle.GetFontList()->FindFontByName(pName);
    if (pFont != NULL)
    {
        return pFont;
    }

    // アーカイブ内のリソースをロードする
    pFont = this->LoadFont(pDevice, pName);
    if (pFont != NULL)
    {
        m_ArcHandle.RegisterFont(pName, pFont); // アーカイブ内のリソース用リスト
    }

    return pFont;
}

//----------------------------------------
FontKey
ArcResourceAccessor::RegisterFont(const char* pName, nn::font::Font* pFont)
{
    return m_UserRegisteredFontList.RegisterFont(pName, pFont, false); // ユーザがロードしたリソース用リスト
}

//----------------------------------------
void
ArcResourceAccessor::UnregisterFont(FontKey key)
{
    m_UserRegisteredFontList.UnregisterFont(key);
}

//----------------------------------------
const TextureInfo*
ArcResourceAccessor::AcquireTexture(nn::gfx::Device* pDevice, const char *pName)
{
    // ユーザがリソースをロードしていれば使う
    const TextureInfo* pTexInfo = m_UserRegisteredTextureList.FindTextureByName(pName);
    if (pTexInfo != NULL)
    {
        return pTexInfo;
    }

    // アーカイブ内のリソースがロードされていれば使う
    pTexInfo = m_ArcHandle.GetTextureList()->FindTextureByName(pName);
    if (pTexInfo != NULL)
    {
        return pTexInfo;
    }

    // アーカイブ内のリソースをロードする
    {
        ResourceTextureInfo* pNewTextureInfo = m_ArcHandle.RegisterTexture(pName); // アーカイブ内のリソース用リスト

        if (pNewTextureInfo)
        {
            this->LoadTexture(pNewTextureInfo, pDevice, pName);

            if (!pNewTextureInfo->IsValid())
            {
                NN_DETAIL_UI2D_ERROR("Can't load texture : %s\n", pName);
            }
        }
        else
        {
            NN_DETAIL_UI2D_ERROR("Can't register texture : %s\n", pName);
        }

        return pNewTextureInfo;
    }
}

//----------------------------------------
PlacementTextureInfo*
ArcResourceAccessor::RegisterTexture(const char* pName)
{
    return m_UserRegisteredTextureList.RegisterPlacementTexture(pName, false); // ユーザがロードしたリソース用リスト
}

//----------------------------------------
void
ArcResourceAccessor::UnregisterTexture(TextureInfo* pTexInfo)
{
    m_UserRegisteredTextureList.UnregisterTexture(pTexInfo);
}

//----------------------------------------
const ShaderInfo*
ArcResourceAccessor::AcquireShader(nn::gfx::Device* pDevice, const char* pName)
{
    // 統合済みアーカイブシェーダの場合は特別な名前で登録する
    int firstBlend;
    int secondBlend;
    bool isCombined = ConvertArchiveShaderNameToBlends(&firstBlend, &secondBlend, pName);
    const char* pRegistrationName = GetArchiveShaderRegistrationName(firstBlend, isCombined, pName);

    // ユーザがリソースをロードしていれば使う
    const ShaderInfo* pShaderInfo = m_UserRegisteredShaderList.FindShaderByName(pRegistrationName);
    if (pShaderInfo != NULL)
    {
        return pShaderInfo;
    }

    // アーカイブ内のリソースがロードされていれば使う
    pShaderInfo = m_ArcHandle.GetShaderList()->FindShaderByName(pRegistrationName);
    if (pShaderInfo != NULL)
    {
        return pShaderInfo;
    }

    // アーカイブ内のリソースをロードする
    {
        ShaderInfo* pNewShaderInfo = m_ArcHandle.RegisterShader(pRegistrationName); // アーカイブ内のリソース用リスト

        if (pNewShaderInfo)
        {
            this->LoadShader(pNewShaderInfo, pDevice, pName);
        }
        else
        {
            NN_DETAIL_UI2D_ERROR("Can't register shader : %s\n", pName);
        }

        return pNewShaderInfo;
    }
}

//----------------------------------------
ShaderInfo*
ArcResourceAccessor::RegisterShader(const char* pName, bool isOwned)
{
    return m_UserRegisteredShaderList.RegisterShader(pName, isOwned); // ユーザがロードしたリソース用リスト
}

//----------------------------------------
void
ArcResourceAccessor::UnregisterShader(ShaderInfo* pShaderInfo)
{
    m_UserRegisteredShaderList.UnregisterShader(pShaderInfo);
}

//----------------------------------------
void
ArcResourceAccessor::RegisterTextureViewToDescriptorPool(RegisterTextureViewSlot pRegisterTextureViewSlot, void* pUserData)
{
    m_UserRegisteredFontList.RegisterTextureViewToDescriptorPool(pRegisterTextureViewSlot, pUserData);
    m_UserRegisteredTextureList.RegisterTextureViewToDescriptorPool(pRegisterTextureViewSlot, pUserData);
    m_ArcHandle.RegisterTextureViewToDescriptorPool(pRegisterTextureViewSlot, pUserData);
}

//----------------------------------------
void
ArcResourceAccessor::UnregisterTextureViewFromDescriptorPool(UnregisterTextureViewSlot pUnregisterTextureViewSlot, void* pUserData)
{
    m_UserRegisteredFontList.UnregisterTextureViewFromDescriptorPool(pUnregisterTextureViewSlot, pUserData);
    m_UserRegisteredTextureList.UnregisterTextureViewFromDescriptorPool(pUnregisterTextureViewSlot, pUserData);
    m_ArcHandle.UnregisterTextureViewFromDescriptorPool(pUnregisterTextureViewSlot, pUserData);
}

//----------------------------------------
RenderTargetTextureInfo*
ArcResourceAccessor::RegisterRenderTargetTexture(const char* pName)
{
    // レンダーターゲットテクスチャの名前が重複すると最初に登録したものしか見つからなくなり
    // 意図したテクスチャが参照されないケースが発生するためアサートで止める。
    NN_SDK_ASSERT(m_UserRegisteredTextureList.FindTextureByName(pName) == NULL);

    // ユーザーが用意したリソースとしてキャプチャテクスチャを登録する。
    return m_UserRegisteredTextureList.RegisterRenderTargetTexture(pName, false);
}

//----------------------------------------
void
ArcResourceAccessor::UnregisterRenderTargetTexture(TextureInfo* pTexInfo)
{
    m_UserRegisteredTextureList.UnregisterTexture(pTexInfo);
}

//----------------------------------------
MultiArcResourceAccessor::MultiArcResourceAccessor()
{
}

//----------------------------------------
MultiArcResourceAccessor::~MultiArcResourceAccessor()
{
    NN_SDK_ASSERT(m_ArcList.empty() && m_UserRegisteredFontList.empty() && m_UserRegisteredTextureList.empty() && m_UserRegisteredShaderList.empty(), "MultiArcResourceAccessor::Finalize() must be called before destructor.");
}

//----------------------------------------
void MultiArcResourceAccessor::Finalize(nn::gfx::Device* pDevice)
{
    ArcResourceList::iterator endIter = m_ArcList.end();
    for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; )
    {
        ArcResourceList::iterator currIter = iter++;
        currIter->GetArcHandle()->Finalize(pDevice);
        m_ArcList.erase(currIter);
        Layout::DeleteObj(&(*currIter));
    }

    m_UserRegisteredFontList.Finalize(pDevice);
    m_UserRegisteredTextureList.Finalize(pDevice);
    m_UserRegisteredShaderList.Finalize(pDevice);

    ResourceAccessor::Finalize(pDevice);
}

//----------------------------------------
void
MultiArcResourceAccessor::Attach(ArchiveHandle* pArchiveHandle)
{
    NN_SDK_ASSERT_NOT_NULL(pArchiveHandle);

    ArcResourceLink* pLink = Layout::AllocateAndConstruct<ArcResourceLink>();
    pLink->SetArcHandle(pArchiveHandle);
    m_ArcList.push_back(*pLink);
}

//----------------------------------------
void
MultiArcResourceAccessor::Detach(const ArchiveHandle* pArchiveHandle)
{
    NN_SDK_ASSERT_NOT_NULL(pArchiveHandle);

    ArcResourceList::iterator endIter = m_ArcList.end();
    for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
    {
        if (pArchiveHandle == iter->GetArcHandle())
        {
            ArcResourceLink* pLink = &(*iter);
            m_ArcList.erase(iter);
            Layout::DeleteObj(pLink);
            return;
        }
    }
}

//----------------------------------------
void
MultiArcResourceAccessor::DetachAll()
{
    ArcResourceList::iterator endIter = m_ArcList.end();
    for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; )
    {
        ArcResourceList::iterator currIter = iter++;
        m_ArcList.erase(currIter);
        Layout::DeleteObj(&(*currIter));
    }
}

//----------------------------------------
void*
MultiArcResourceAccessor::FindResourceByName(
    size_t*   pSize,
    ResType     resType,
    const char* pName
)
{
    ArcResourceList::iterator endIter = m_ArcList.end();
    for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
    {
        ArchiveHandle* pArcHandle = iter->GetArcHandle();
        if (void* pRes = GetResourceSub(pArcHandle, pArcHandle->GetResRootDir(), resType, pName, pSize))
        {
            return pRes;
        }
    }

    return NULL;
}

//----------------------------------------
void
MultiArcResourceAccessor::FindResourceByType(
    ResType resType,
    FindResourceByTypeCallback pCallback,
    void* pParam
) const
{
    ArcResourceList::const_iterator endIter = m_ArcList.end();
    for (ArcResourceList::const_iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
    {
        const ArchiveHandle* pArcHandle = iter->GetArcHandle();
        SearchAllResource(pArcHandle, pArcHandle->GetResRootDir(), resType, pCallback, pParam);
    }
}

//----------------------------------------
nn::font::Font*
MultiArcResourceAccessor::AcquireFont(nn::gfx::Device* pDevice, const char *pName)
{
    // ユーザがリソースをロードしていれば使う
    nn::font::Font* pFont = m_UserRegisteredFontList.FindFontByName(pName);
    if (pFont != NULL)
    {
        return pFont;
    }

    // アーカイブ内のリソースがロードされていれば使う
    ArcResourceList::iterator endIter = m_ArcList.end();
    for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
    {
        pFont = iter->GetArcHandle()->GetFontList()->FindFontByName(pName);
        if (pFont != NULL)
        {
            return pFont;
        }
    }

    // アーカイブ内のリソースをロードする
    {
        ArchiveHandle* pArchiveHandle = FindFontArchive(pName);
        pFont = this->LoadFont(pDevice, pName);
        if (pFont != NULL)
        {
            pArchiveHandle->RegisterFont(pName, pFont);
        }
    }

    return pFont;
}

//----------------------------------------
FontKey
MultiArcResourceAccessor::RegisterFont(const char* pName, nn::font::Font* pFont)
{
    return m_UserRegisteredFontList.RegisterFont(pName, pFont, false);
}

//----------------------------------------
void
MultiArcResourceAccessor::UnregisterFont(FontKey key)
{
    m_UserRegisteredFontList.UnregisterFont(key);
}

//----------------------------------------
const TextureInfo*
MultiArcResourceAccessor::AcquireTexture(nn::gfx::Device* pDevice, const char *pName)
{
    // ユーザがリソースをロードしていれば使う
    const TextureInfo* pTexInfo = m_UserRegisteredTextureList.FindTextureByName(pName);
    if (pTexInfo != NULL)
    {
        return pTexInfo;
    }

    // アーカイブ内のリソースがロードされていれば使う
    ArcResourceList::iterator endIter = m_ArcList.end();
    for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
    {
        pTexInfo = iter->GetArcHandle()->GetTextureList()->FindTextureByName(pName);
        if (pTexInfo != NULL)
        {
            return pTexInfo;
        }
    }

    // アーカイブ内のリソースをロードする
    {
        ArchiveHandle* pArchiveHandle = FindTextureArchive(pName);
        ResourceTextureInfo* pNewTextureInfo = pArchiveHandle->RegisterTexture(pName);

        if (pNewTextureInfo)
        {
            this->LoadTexture(pNewTextureInfo, pDevice, pName);

            if (!pNewTextureInfo->IsValid())
            {
                NN_DETAIL_UI2D_ERROR("Can't load texture : %s\n", pName);
            }
        }
        else
        {
            NN_DETAIL_UI2D_ERROR("Can't register texture : %s\n", pName);
        }

        return pNewTextureInfo;
    }
}

//----------------------------------------
bool
MultiArcResourceAccessor::LoadTexture(ResourceTextureInfo* pTexInfo, nn::gfx::Device* pDevice, const char *pName)
{
    // アーカイブ内のリソースがロードされていれば使う
    ArcResourceList::iterator endIter = m_ArcList.end();
    for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
    {
        ArchiveHandle* pArcHandle = iter->GetArcHandle();
        if(pArcHandle->LoadTexture(pTexInfo, pDevice, pName))
        {
            return true;
        }
    }

    NN_DETAIL_UI2D_ERROR("texture resource not found. - %s", pName);

    return false;
}

//----------------------------------------
bool MultiArcResourceAccessor::LoadShader(ShaderInfo* pShaderInfo, nn::gfx::Device* pDevice, const char* pName)
{
    // アーカイブ内のリソースがロードされていれば使う
    ArcResourceList::iterator endIter = m_ArcList.end();
    for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
    {
        ArchiveHandle* pArcHandle = iter->GetArcHandle();
        if(pArcHandle->LoadShader(pShaderInfo, pDevice, pName))
        {
            return true;
        }
    }

    NN_DETAIL_UI2D_ERROR("shader resource not found. - %s", pName);

    return false;
}

//----------------------------------------
PlacementTextureInfo*
MultiArcResourceAccessor::RegisterTexture(const char* pName)
{
    return m_UserRegisteredTextureList.RegisterPlacementTexture(pName, false);
}

//----------------------------------------
void
MultiArcResourceAccessor::UnregisterTexture(TextureInfo* pTexInfo)
{
    m_UserRegisteredTextureList.UnregisterTexture(pTexInfo);
}

//----------------------------------------
const ShaderInfo*
MultiArcResourceAccessor::AcquireShader(nn::gfx::Device* pDevice, const char* pName)
{
    // 統合済みアーカイブシェーダの場合は特別な名前で登録する
    int firstBlend;
    int secondBlend;
    bool isCombined = ConvertArchiveShaderNameToBlends(&firstBlend, &secondBlend, pName);
    const char* pRegistrationName = GetArchiveShaderRegistrationName(firstBlend, isCombined, pName);

    // ユーザがリソースをロードしていれば使う
    const ShaderInfo* pShaderInfo = m_UserRegisteredShaderList.FindShaderByName(pRegistrationName);
    if (pShaderInfo != NULL)
    {
        return pShaderInfo;
    }

    // アーカイブ内のリソースがロードされていれば使う
    ArcResourceList::iterator endIter = m_ArcList.end();
    for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
    {
        pShaderInfo = iter->GetArcHandle()->GetShaderList()->FindShaderByName(pRegistrationName);
        if (pShaderInfo != NULL)
        {
            if (isCombined)
            {
                if (!IsIndividualArchiveShaderVariationTable(firstBlend))
                {
                    const void* pVariationTable = pShaderInfo->GetVariationTable();
                    int variationIndex = SearchShaderVariationIndexFromTable(pVariationTable, firstBlend, secondBlend);
                    if (variationIndex != -1)
                    {
                        return pShaderInfo;
                    }
                }
                else
                {
                    // 詳細コンバイナ（バリエーション定数版）
                    int stageBits[DetailedCombinerStageBitsCountWithVariationTable];
                    ConvertArchiveShaderDetailedCombinerNameToStageBits(stageBits, pName);

                    const void* pVariationTable = pShaderInfo->GetVariationTable();
                    const int variationIndex = SearchShaderVariationDetailedCombinerIndexFromTable(pVariationTable, stageBits);
                    if (variationIndex != -1)
                    {
                        return pShaderInfo;
                    }
                }
            }
            else
            {
                return pShaderInfo;
            }
        }
    }

    // アーカイブ内のリソースをロードする
    {
        ArchiveHandle* pArchiveHandle = FindShaderArchive(pName);
        ShaderInfo* pNewShaderInfo = pArchiveHandle->RegisterShader(pRegistrationName);

        if (pNewShaderInfo)
        {
            this->LoadShader(pNewShaderInfo, pDevice, pName);
        }
        else
        {
            NN_DETAIL_UI2D_ERROR("Can't register shader: %s\n", pName);
        }

        return pNewShaderInfo;
    }
}

//----------------------------------------
ShaderInfo*
MultiArcResourceAccessor::RegisterShader(const char* pName)
{
    return m_UserRegisteredShaderList.RegisterShader(pName, false);
}

//----------------------------------------
void
MultiArcResourceAccessor::UnregisterShader(ShaderInfo* pShaderInfo)
{
    m_UserRegisteredShaderList.UnregisterShader(pShaderInfo);
}

//----------------------------------------
void
MultiArcResourceAccessor::RegisterTextureViewToDescriptorPool(RegisterTextureViewSlot pRegisterTextureViewSlot, void* pUserData)
{
    m_UserRegisteredFontList.RegisterTextureViewToDescriptorPool(pRegisterTextureViewSlot, pUserData);
    m_UserRegisteredTextureList.RegisterTextureViewToDescriptorPool(pRegisterTextureViewSlot, pUserData);
    ArcResourceList::iterator endIter = m_ArcList.end();
    for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
    {
        iter->GetArcHandle()->RegisterTextureViewToDescriptorPool(pRegisterTextureViewSlot, pUserData);;
    }
}

//----------------------------------------
void
MultiArcResourceAccessor::UnregisterTextureViewFromDescriptorPool(UnregisterTextureViewSlot pUnregisterTextureViewSlot, void* pUserData)
{
    m_UserRegisteredFontList.UnregisterTextureViewFromDescriptorPool(pUnregisterTextureViewSlot, pUserData);
    m_UserRegisteredTextureList.UnregisterTextureViewFromDescriptorPool(pUnregisterTextureViewSlot, pUserData);
    ArcResourceList::iterator endIter = m_ArcList.end();
    for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
    {
        iter->GetArcHandle()->UnregisterTextureViewFromDescriptorPool(pUnregisterTextureViewSlot, pUserData);;
    }
}

//----------------------------------------
RenderTargetTextureInfo*
MultiArcResourceAccessor::RegisterRenderTargetTexture(const char* pName)
{
    // レンダーターゲットテクスチャの名前が重複すると最初に登録したものしか見つからなくなり
    // 意図したテクスチャが参照されないケースが発生するためアサートで止める。
    NN_SDK_ASSERT(m_UserRegisteredTextureList.FindTextureByName(pName) == NULL);

    // ユーザーが用意したリソースとしてキャプチャテクスチャを登録する。
    return m_UserRegisteredTextureList.RegisterRenderTargetTexture(pName, false);
}

//----------------------------------------
void
MultiArcResourceAccessor::UnregisterRenderTargetTexture(TextureInfo* pTexInfo)
{
    m_UserRegisteredTextureList.UnregisterTexture(pTexInfo);
}

//----------------------------------------
ArchiveHandle* MultiArcResourceAccessor::FindFontArchive(const char* pName)
{
    ArcResourceList::iterator endIter = m_ArcList.end();
    for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
    {
        ArchiveHandle* pArcHandle = iter->GetArcHandle();
        void* pFontRes = GetResourceSub(pArcHandle, pArcHandle->GetResRootDir(), ResourceTypeFont, pName, NULL);
        if (pFontRes != NULL)
        {
            return iter->GetArcHandle();
        }
    }
    NN_DETAIL_UI2D_ERROR("font resource not found. - %s\n", pName);
    return NULL;
}

//----------------------------------------
ArchiveHandle* MultiArcResourceAccessor::FindTextureArchive(const char* pName)
{
    ArcResourceList::iterator endIter = m_ArcList.end();
    for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
    {
        void* pTexRes = NULL;
        ArchiveHandle* pArcHandle = iter->GetArcHandle();

        // まず、bntx を探してその中からテクスチャが無いか探します。
        pTexRes = GetResourceSub(pArcHandle, pArcHandle->GetResRootDir(), ResourceTypeTexture, "__Combined.bntx", NULL);
        if (pTexRes != NULL)
        {
            nn::gfx::ResTextureFile* pResTextureFile = nn::gfx::ResTextureFile::ResCast(pTexRes);
            if (pResTextureFile != NULL)
            {
                if (pResTextureFile->GetTextureDic()->FindIndex(pName) != -1)
                {
                    return iter->GetArcHandle();
                }
            }
        }

        // bntx から発見できない場合、指定された名前そのままで検索する
        pTexRes = GetResourceSub(pArcHandle, pArcHandle->GetResRootDir(), ResourceTypeTexture, pName, NULL);
        if (pTexRes != NULL)
        {
            return iter->GetArcHandle();
        }
    }
    NN_DETAIL_UI2D_ERROR("texture resource not found. - %s\n", pName);
    return NULL;
}

//----------------------------------------
ArchiveHandle* MultiArcResourceAccessor::FindShaderArchive(const char* pName)
{
    int firstBlend;
    int secondBlend;
    bool isCombined = ConvertArchiveShaderNameToBlends(&firstBlend, &secondBlend, pName);

    if (isCombined)
    {
        // 統合済みアーカイブシェーダから探します。
        if (!IsIndividualArchiveShaderVariationTable(firstBlend))
        {
            ArcResourceList::iterator endIter = m_ArcList.end();
            for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
            {
                ArchiveHandle* pArcHandle = iter->GetArcHandle();
                void* pShaderRes = GetResourceSub(pArcHandle, pArcHandle->GetResRootDir(), ResourceTypeShader, ArchiveShaderFileName, NULL);
                if (pShaderRes != NULL)
                {
                    void* pVariationTable = GetResourceSub(pArcHandle, pArcHandle->GetResRootDir(), ResourceTypeShader, ArchiveShaderVariationTableFileName, NULL);
                    if (pShaderRes != NULL)
                    {
                        int variationIndex = SearchShaderVariationIndexFromTable(pVariationTable, firstBlend, secondBlend);
                        if (variationIndex != -1)
                        {
                            return iter->GetArcHandle();
                        }
                    }
                }
            }
        }
        else
        {
            // 詳細コンバイナ（バリエーション定数版）
            int stageBits[DetailedCombinerStageBitsCountWithVariationTable];
            ConvertArchiveShaderDetailedCombinerNameToStageBits(stageBits, pName);

            //! バリエーション変数版の詳細コンバイナ用バリエーションテーブルのファイル名
            char individualArchiveShaderVariationTableName[ResourcePathLengthMax];
            std::sprintf(individualArchiveShaderVariationTableName, "%s_%d.%s",
                IndividualArchiveShaderVariationTableBaseName,
                firstBlend,
                IndividualArchiveShaderVariationTableExtensionName);

            ArcResourceList::iterator endIter = m_ArcList.end();
            for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
            {
                ArchiveHandle* pArcHandle = iter->GetArcHandle();
                void* pDetailedCombinerVariationTable = GetResourceSub(pArcHandle, pArcHandle->GetResRootDir(), ResourceTypeShader, individualArchiveShaderVariationTableName, NULL);
                if (pDetailedCombinerVariationTable)
                {
                    const int variationIndex = SearchShaderVariationDetailedCombinerIndexFromTable(pDetailedCombinerVariationTable, stageBits);
                    if (variationIndex != -1)
                    {
                        return iter->GetArcHandle();
                    }
                }
            }
        }
        NN_DETAIL_UI2D_ERROR("shader resource not found. - %s\n", pName);
    }
    else
    {
        // 指定された名前そのままで検索します。
        char str[256];
        ResourceAccessor::GetArchiveShaderResourceName(str, pName, sizeof(str));
        ArcResourceList::iterator endIter = m_ArcList.end();
        for (ArcResourceList::iterator iter = m_ArcList.begin(); iter != endIter; ++iter)
        {
            ArchiveHandle* pArcHandle = iter->GetArcHandle();
            void* pShaderRes = GetResourceSub(pArcHandle, pArcHandle->GetResRootDir(), ResourceTypeShader, str, NULL);
            if (pShaderRes != NULL)
            {
                return iter->GetArcHandle();
            }
        }
        NN_DETAIL_UI2D_ERROR("shader resource not found. - %s\n", str);
    }

    return NULL;
}

}   // namespace ui2d
}   // namespace nn
