﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_Animator.h>
#include <nn/ui2d/ui2d_Util.h>
#include <nn/ui2d/ui2d_Group.h>
#include <nn/ui2d/ui2d_Pane.h>
#include <nn/ui2d/ui2d_Resources.h>

namespace nn
{
namespace ui2d
{

Animator::Animator()
: m_Speed(0.f)
, m_PlayType(PlayType_OneShot)
{
    m_Flag.Clear();
}

Animator::~Animator()
{
}

void Animator::Play(PlayType type, float speed)
{
    if ((- GetFrameMax() <= speed && speed <= GetFrameMax()) || (IsWaitData() && type == PlayType_OneShot)) {
        m_PlayType = type;
        m_Speed = speed;
        ClearPlayStatusFlag();

        if (speed >= 0) {
            SetFrame(0.f);
        } else {
            SetFrame(GetFrameMax());
        }
    } else {
        if (IsWaitData()) {
            NN_SDK_ASSERT(false, "wait animation play type must be PlayType_OneShot");
        } else {
            NN_SDK_ASSERT(false, "play speed[%f] exceeds MaxFrame", speed);
        }
    }
}

void Animator::PlayAuto(float speed)
{
    Play(IsLoopData() ? PlayType_Loop : PlayType_OneShot, speed);
}

void Animator::PlayFromCurrent(PlayType type, float speed)
{
    if ((- GetFrameMax() <= speed && speed <= GetFrameMax()) || (IsWaitData() && type == PlayType_OneShot)) {
        m_PlayType = type;
        m_Speed = speed;
        ClearPlayStatusFlag();
    } else {
        if (IsWaitData()) {
            NN_SDK_ASSERT(false, "wait animation play type must be PlayType_OneShot");
        } else {
            NN_SDK_ASSERT(false, "play speed[%f] exceeds MaxFrame", speed);
        }
    }
}

void Animator::StopAt(float frame)
{
    NN_SDK_ASSERT((0.f <= frame) && (frame <= GetFrameMax()));
    m_Speed = 0.f;
    SetFrame(frame);
    ClearPlayStatusFlag();
}

void Animator::StopAtCurrentFrame()
{
    m_Speed = 0.f;
    ClearPlayStatusFlag();
}

void Animator::StopAtStartFrame()
{
    m_Speed = 0.f;
    SetFrame(0.f);
    ClearPlayStatusFlag();
}

void Animator::StopAtEndFrame()
{
    m_Speed = 0.f;
    SetFrame(GetFrameMax());
    ClearPlayStatusFlag();
}

void Animator::UpdateFrame(float progress_frame)
{
    ClearPlayStatusFlag();

    if (m_Speed == 0.f) {
        // 再生スピードが0の場合はここで抜ける
        return;
    }

    float nextFrame = GetFrame() + m_Speed * progress_frame;

    if ( m_Speed > 0 ) {
        if (nextFrame >= GetFrameMax()) {
            switch( m_PlayType ) {
            case PlayType_OneShot:
                nextFrame = GetFrameMax();
                m_Speed = 0.f;
                m_Flag.SetAllBitOn(FlagMask_PlayStatusEnd);
                break;
            case PlayType_Loop:
                nextFrame = (nextFrame - GetFrameMax());
                m_Flag.SetAllBitOn(FlagMask_PlayStatusTurnedMax);
                break;
            case PlayType_RoundTrip:
                nextFrame = GetFrameMax() - (nextFrame - GetFrameMax());
                m_Speed *= -1;
                m_Flag.SetAllBitOn(FlagMask_PlayStatusTurnedMax);
                break;
            default:
                NN_SDK_ASSERT(false);break;
            }
        }
    } else {
        if ( nextFrame <= 0.f ) {
            switch( m_PlayType ) {
            case PlayType_OneShot:
                nextFrame = 0.f;
                m_Speed = 0.f;
                m_Flag.SetAllBitOn(FlagMask_PlayStatusEnd);
                break;
            case PlayType_Loop:
                nextFrame = nextFrame + GetFrameMax();
                m_Flag.SetAllBitOn(FlagMask_PlayStatusTurnedMin);
                break;
            case PlayType_RoundTrip:
                nextFrame = - nextFrame;
                m_Speed *= -1;
                m_Flag.SetAllBitOn(FlagMask_PlayStatusTurnedMin);
                break;
            default:
                NN_SDK_ASSERT(false);break;
            }
        }
    }

    SetFrame(nextFrame);
}

void PaneAnimator::Setup(Pane* pPane, bool enabled)
{
    BindPane(pPane, false);
    m_pPane = pPane;
    SetEnabled(enabled);
}

void PaneAnimator::Unbind()
{
    UnbindPane(m_pPane);
}

void GroupAnimator::Setup(Group* pGroup, bool enabled)
{
    BindGroup(pGroup);
    m_pGroup = pGroup;
    SetEnabled(enabled);
}

void GroupAnimator::Unbind()
{
    UnbindGroup(m_pGroup);
}

void GroupAnimator::Setup(const AnimResource& animRes, GroupContainer* pGroupContainer, int  index, bool enabled)
{
    const int bindGroupNum = animRes.GetGroupCount();
    if (index < bindGroupNum) {
        Setup(pGroupContainer->FindGroupByName(animRes.GetGroupArray()[index].GetName()), enabled);
    } else {
        // index番目のグループが存在しない
        NN_SDK_ASSERT(false, "index[%d] exceeds group num[%d] of AnimResource[%s]\n", index, bindGroupNum, animRes.GetTagName());
    }
}

void GroupArrayAnimator::Setup(const AnimResource& animRes, GroupContainer* pGroupContainer, Group** pGroupBuffer, bool enabled)
{
    m_GroupCount = animRes.GetGroupCount();
    m_pGroupArray = pGroupBuffer;
    for (int32_t i = 0; i < m_GroupCount; i++) {
        m_pGroupArray[i] = pGroupContainer->FindGroupByName(animRes.GetGroupArray()[i].GetName());
        BindGroup(m_pGroupArray[i]);
    }
    SetEnabled(enabled);
}

void GroupArrayAnimator::Unbind()
{
    for (int32_t  i = 0; i < m_GroupCount; i++) {
        UnbindGroup(m_pGroupArray[i]);
    }
}

} // namespace nn::ui2d
} // namespace nn
