﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_AnimButton.h>
#include <nn/ui2d/ui2d_Animator.h>
#include <nn/ui2d/ui2d_ControlCreator.h>
#include <nn/ui2d/ui2d_Layout.h>

namespace nn
{
namespace ui2d
{

AnimButton::AnimButton()
: m_pStateChangeCallback(NULL)
, m_pStateChangeCallbackParam(NULL)
, m_pOnOffAnimator(NULL)
, m_pDownAnimator(NULL)
, m_pCancelAnimator(NULL)
, m_pDisableAnimator(NULL)
, m_pHitPane(NULL)
, m_Tag(0)
, m_pName(NULL)
{
    m_HitBoxBottomLeft.v[0] = 0.0f;
    m_HitBoxBottomLeft.v[1] = 0.0f;
    m_HitBoxTopRight.v[0] = 0.0f;
    m_HitBoxTopRight.v[1] = 0.0f;
}

void AnimButton::Build(nn::gfx::Device* pDevice, ui2d::Layout* pLayout, const ui2d::ControlSrc& controlSrc)
{
    // 番をOnOffアニメーションとDownアニメーションを取得する。
    // また、アニメーションはすべてグループアニメーションであり、グループが一つ以上割り当てられている必要がある。
    m_pOnOffAnimator = pLayout->CreateGroupAnimatorAuto(pDevice, controlSrc.FindFunctionalAnimName("OnOff"), true);
    NN_SDK_ASSERT(m_pOnOffAnimator, "cannot create OnOffAnimator[%s] for Layout[%s]", controlSrc.FindFunctionalAnimName("OnOff"), pLayout->GetName());
    m_pOnOffAnimator->StopAtStartFrame();
    m_pDownAnimator = pLayout->CreateGroupAnimatorAuto(pDevice, controlSrc.FindFunctionalAnimName("Down"), false);
    NN_SDK_ASSERT(m_pDownAnimator, "cannot create DownAnimator[%s] for Layout[%s]", controlSrc.FindFunctionalAnimName("Down"), pLayout->GetName());
    // Disableアニメーションがあれば作成
    {
        const char* anim_name = controlSrc.FindFunctionalAnimName("Disable");
        if (anim_name && anim_name[0]) {
            m_pDisableAnimator = pLayout->CreateGroupAnimatorAuto(pDevice, anim_name, false);
        }
    }

    // 当たりのペインを取得
    m_pHitPane = pLayout->GetRootPane()->FindPaneByName(controlSrc.FindFunctionalPaneName("Hit"));
    NN_SDK_ASSERT(m_pHitPane, "cannot get HitPane[%s] for Layout[%s]", controlSrc.FindFunctionalPaneName("Hit"), pLayout->GetName());
    // ボタンの名前は、レイアウトが部品であれば部品ペイン名（ルートペインとなっている）、
    // 部品でなければレイアウト名とする
    if (pLayout->IsPartsLayout()) {
        m_pName = pLayout->GetRootPane()->GetName();
    } else {
        m_pName = pLayout->GetName();
    }
}

void AnimButton::UpdateHitBox()
{
    if (m_pHitPane == NULL) {
        return;
    }

    // グローバル行列からスケールを取得して大きさを計算する。
    nn::util::Vector3f axisX;
    nn::util::Vector3f axisY;
    nn::util::Vector3f axisZ;
    nn::util::Vector3f axisW;
    nn::util::MatrixGetAxes(&axisX, &axisY, &axisZ, &axisW, m_pHitPane->GetGlobalMtx());
    float half_width = std::abs(m_pHitPane->GetSize().width * axisX.GetX()) / 2.f;
    float half_height = std::abs(m_pHitPane->GetSize().height * axisY.GetY()) / 2.f;
    nn::util::Float2 center = NN_UTIL_FLOAT_2_INITIALIZER(axisW.GetX(), axisW.GetY());

    switch(m_pHitPane->GetBasePositionX()) {
    case nn::ui2d::HorizontalPosition_Center:
        break;
    case nn::ui2d::HorizontalPosition_Left:
        center.v[0] += half_width;
        break;
    case nn::ui2d::HorizontalPosition_Right:
        center.v[0] -= half_width;
        break;
    default:
        NN_SDK_ASSERT(false,"Unknown HorizontalPosition type.");
        break;
    }

    switch(m_pHitPane->GetBasePositionY()) {
    case nn::ui2d::VerticalPosition_Center:
        break;
    case nn::ui2d::VerticalPosition_Top:
        center.v[1] -= half_height;
        break;
    case nn::ui2d::VerticalPosition_Bottom:
        center.v[1] += half_height;
        break;
    default:
        NN_SDK_ASSERT(false, "Unknown VerticalPosition type.");
        break;
    }

    m_HitBoxBottomLeft.v[0] = center.v[0] - half_width;
    m_HitBoxBottomLeft.v[1] = center.v[1] - half_height;
    m_HitBoxTopRight.v[0] = center.v[0] + half_width;
    m_HitBoxTopRight.v[1] = center.v[1] + half_height;
}

bool AnimButton::IsHit(const nn::util::Float2& vec2) const
{
    if (m_pHitPane) {
        return (m_HitBoxBottomLeft.v[0] <= vec2.v[0] && vec2.v[0] <= m_HitBoxTopRight.v[0] &&
            m_HitBoxBottomLeft.v[1] <= vec2.v[1] && vec2.v[1] <= m_HitBoxTopRight.v[1]);
    } else {
        return false;
    }
}

void AnimButton::InitializeDragPosition(const nn::util::Float2& /* pos */)
{
}

void AnimButton::UpdateDragPosition(const nn::util::Float2* /* pPos */)
{
}

void AnimButton::SetStateChangeCallback(StateChangeCallback pCallback, void* pParam)
{
    m_pStateChangeCallback = pCallback;
    m_pStateChangeCallbackParam = pParam;
}

void AnimButton::PlayDisableAnim(bool isDisable)
{
    if (m_pDisableAnimator) {
        m_pDisableAnimator->SetEnabled(true);
        m_pDisableAnimator->PlayFromCurrent(ui2d::Animator::PlayType_OneShot, isDisable ? 1.f : - 1.f);
    }
}

void AnimButton::SetAllAnimatorDisable()
{
    if (m_pOnOffAnimator) {
        m_pOnOffAnimator->SetEnabled(false);
    }
    if (m_pDownAnimator) {
        m_pDownAnimator->SetEnabled(false);
    }
    if (m_pCancelAnimator) {
        m_pCancelAnimator->SetEnabled(false);
    }
    if (m_pDisableAnimator) {
        m_pDisableAnimator->SetEnabled(false);
    }
}

void AnimButton::ForceOff()
{
    ButtonBase::ForceOff();
    if (m_pOnOffAnimator) {
        EnableAnim(m_pOnOffAnimator);
        m_pOnOffAnimator->StopAtStartFrame();
    }
}

void AnimButton::ForceOn()
{
    ButtonBase::ForceOn();
    if (m_pOnOffAnimator) {
        EnableAnim(m_pOnOffAnimator);
        m_pOnOffAnimator->StopAtEndFrame();
    }
}

void AnimButton::ForceDown()
{
    ButtonBase::ForceDown();
    if (m_pDownAnimator) {
        EnableAnim(m_pDownAnimator);
        m_pDownAnimator->StopAtEndFrame();
    }
}

void AnimButton::StartOn()
{
    if (m_pOnOffAnimator) {
        EnableAnim(m_pOnOffAnimator);
        m_pOnOffAnimator->PlayFromCurrent(ui2d::Animator::PlayType_OneShot, 1.f);
    }
}

void AnimButton::StartOff()
{
    if (m_pOnOffAnimator) {
        EnableAnim(m_pOnOffAnimator);
        m_pOnOffAnimator->PlayFromCurrent(ui2d::Animator::PlayType_OneShot, -1.f);
    }
}

void AnimButton::StartDown()
{
    if (m_pDownAnimator) {
        EnableAnim(m_pDownAnimator);
        m_pDownAnimator->Play(ui2d::Animator::PlayType_OneShot, 1.f);
    }
}

void AnimButton::StartCancel()
{
    if (m_pCancelAnimator) {
        EnableAnim(m_pCancelAnimator);
        m_pCancelAnimator->Play(ui2d::Animator::PlayType_OneShot, 1.f);
    }
}

bool AnimButton::UpdateOn()
{
    if (m_pOnOffAnimator) {
        return m_pOnOffAnimator->IsEndTrigger();
    } else {
        return true;
    }
}

bool AnimButton::UpdateOff()
{
    if (m_pOnOffAnimator) {
        return m_pOnOffAnimator->IsEndTrigger();
    } else {
        return true;
    }
}

bool AnimButton::UpdateDown()
{
    if (m_pDownAnimator) {
        return m_pDownAnimator->IsEndTrigger();
    } else {
        return true;
    }
}

bool AnimButton::UpdateCancel()
{
    if (m_pCancelAnimator) {
        return m_pCancelAnimator->IsEndTrigger();
    } else {
        return true;
    }
}

bool AnimButton::ProcessCancel()
{
    // AnimButtonは、デフォルトではCancelイベントを受け付けても
    // 何もしないようにしておく(Cancelイベントに反応するボタンの
    // 方が少ないため)
    return true;
}

void AnimButton::ChangeState(State state)
{
    const State currentState = GetState();
    if (currentState != state) {
        if (m_pStateChangeCallback) {
            m_pStateChangeCallback(this, currentState, state, m_pStateChangeCallbackParam);
        }
        SetState(state);
    }
}

void AnimButton::EnableAnim(ui2d::Animator* pAnimator)
{
    if (m_pOnOffAnimator) {
        m_pOnOffAnimator->SetEnabled(pAnimator == m_pOnOffAnimator);
    }
    if (m_pDownAnimator) {
        m_pDownAnimator->SetEnabled(pAnimator == m_pDownAnimator);
    }
    if (m_pCancelAnimator) {
        m_pCancelAnimator->SetEnabled(pAnimator == m_pCancelAnimator);
    }
}

} // namespace nn::ui2d
} // namespace nn
