﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "precompiled.h"

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <nn/diag/text/diag_SdkTextSpy.h>

#include <nn/spy/spy_SpyModule.h>

namespace nn {
namespace spy {

//----------------------------------------------------------
SpyModule::~SpyModule() NN_NOEXCEPT
{
}

//----------------------------------------------------------
bool SpyModule::PushData(const void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(GetController());

    SpyController* pController = GetController();
    if (pController != nullptr)
    {
        return pController->PushDataAt(*this, buffer, length, nn::os::GetSystemTick());
    }
    else
    {
        return false;
    }
}

//----------------------------------------------------------
bool SpyModule::PushData(
    const void* const buffers[],
    const size_t lengths[],
    int count) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffers);
    NN_SDK_ASSERT_NOT_NULL(lengths);
    NN_SDK_ASSERT(count >= 0);
    NN_SDK_ASSERT_NOT_NULL(GetController());

    SpyController* pController = GetController();
    if (pController != nullptr)
    {
        return pController->PushDataAt(*this, buffers, lengths, count, nn::os::GetSystemTick());
    }
    else
    {
        return false;
    }
}

//----------------------------------------------------------
bool SpyModule::PushDataAt(const void* buffer, size_t length, nn::os::Tick tick) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(GetController());

    SpyController* pController = GetController();
    if (pController != nullptr)
    {
        return pController->PushDataAt(*this, buffer, length, tick);
    }
    else
    {
        return false;
    }
}

//----------------------------------------------------------
bool SpyModule::PushDataAt(
    const void* const buffers[],
    const size_t lengths[],
    int count,
    nn::os::Tick tick) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffers);
    NN_SDK_ASSERT_NOT_NULL(lengths);
    NN_SDK_ASSERT(count >= 0);
    NN_SDK_ASSERT_NOT_NULL(GetController());

    SpyController* pController = GetController();
    if (pController != nullptr)
    {
        return pController->PushDataAt(*this, buffers, lengths, count, tick);
    }
    else
    {
        return false;
    }
}

//----------------------------------------------------------
void SpyModule::Attach(SpyController& controller) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pController == NULL || m_pController == &controller, NN_TEXT_SPY("この SpyModule は、すでに他の SpyController にアタッチされています。"));

    m_pController = &controller;
    ++m_RegisterCount;

    if (m_RegisterCount > 1)
    {
        return;
    }

    // このタイミング（初期化時）で、DataId を無効化し、
    // PC との接続のタイミングで有効な ID を割り当てます。
    AllocateDataId(nn::spy::detail::SpyDataId_Invalid);

    OnRegistered();
}

//----------------------------------------------------------
void SpyModule::Detach(bool force) NN_NOEXCEPT
{
    if (m_RegisterCount == 0)
    {
        NN_SDK_ASSERT(false, NN_TEXT_SPY("アタッチされていない SpyModule に対して、Detach() が呼び出されました。"));
        return;
    }

    if (force)
    {
        m_RegisterCount = 0;
    }
    else
    {
        --m_RegisterCount;

        if (m_RegisterCount > 0)
        {
            return;
        }
    }

    AllocateDataId(nn::spy::detail::SpyDataId_Invalid);
    SetPrevious(NULL);
    SetNext(NULL);

    OnUnregistered();

    m_pController = NULL;
}

//----------------------------------------------------------
void SpyModule::AllocateDataId(nn::spy::detail::SpyDataId dataId) NN_NOEXCEPT
{
    m_DataInfo.SetDataId(dataId);
}

} // namespace nn::spy
} // namespace nn

#endif // NN_BUILD_CONFIG_SPY_ENABLED
