﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "precompiled.h"

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <nn/diag/text/diag_SdkTextSpy.h>

#include <nn/spy/detail/fnd/basis/spyfnd_Memory.h>
#include <nn/spy/detail/fnd/basis/spyfnd_Time.h>
#include <nn/spy/detail/fnd/hio/spyfnd_HioChannel.h>
#include <nn/spy/detail/fnd/string/spyfnd_String.h>
#include <nn/spy/detail/spy_Packet.h>
#include <nn/spy/spy_SpyController.h>

#include <nn/nn_Middleware.h>
#include <nn/nn_Version.h>

// ミドルウェア情報
#define NW_MIDDLEWARE_SYMBOL(buildOption) "NintendoWare_Spy-" NN_MACRO_STRINGIZE(NN_NX_ADDON_VERSION_MAJOR) "_" NN_MACRO_STRINGIZE(NN_NX_ADDON_VERSION_MINOR) "_" NN_MACRO_STRINGIZE(NN_NX_ADDON_VERSION_MICRO) "-" #buildOption

#if defined(NN_SDK_BUILD_DEBUG)
NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL(Debug));
#elif defined(NN_SDK_BUILD_DEVELOP)
NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL(Develop));
#elif defined(NN_SDK_BUILD_RELEASE)
NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL(Release));
#endif

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
//#define BUFFER_DEBUG_ENABLED
//#define MODULE_DEBUG_ENABLED
//#define STATE_DEBUG_ENABLED
#endif

namespace nn {
namespace spy {

namespace {

#if defined(NN_BUILD_CONFIG_SPEC_CAFE)
const int DataBufferAlignment = FS_IO_BUFFER_ALIGN;
#else
const int DataBufferAlignment = 1;
#endif

#if defined(NN_BUILD_CONFIG_SPEC_CAFE)
static const char* SyncPort = "NWSoundSpy_Sync";
static const char* DataPort = "NWSoundSpy_Data";
#else
static const char* SyncPort = "NnSpy_Sync";
static const char* DataPort = "NnSpy_Data";
#endif

static const char* SyncThreadName = "nn::spy::SyncThread";
static const char* DataThreadName = "nn::spy::DataThread";

static const size_t DefaultDataBufferLength = 1 * 1024;
static const size_t DefaultSyncThreadStackSize = 16 * 1024;
static const size_t DefaultDataThreadStackSize = 16 * 1024;

}

//----------------------------------------------------------
SpyController::InitializeArg::InitializeArg() NN_NOEXCEPT
    : dataBufferLength(DefaultDataBufferLength)
    , syncThreadStackSize(DefaultSyncThreadStackSize)
    , dataThreadStackSize(DefaultDataThreadStackSize)
{
}

//----------------------------------------------------------
SpyController::OpenArg::OpenArg() NN_NOEXCEPT
    : syncThreadPriority(nn::spy::detail::fnd::Thread::DefaultThreadPriority)
    , dataThreadPriority(nn::spy::detail::fnd::Thread::DefaultThreadPriority)
{
}

//----------------------------------------------------------
SpyController::SpyController() NN_NOEXCEPT
    : m_TickBase(0)
    , m_LastOutOfMemoryWarningTick(0)
    , m_State(State_NotInitialized)
    , m_pModuleTop(NULL)
    , m_pModuleLast(NULL)
    , m_pModule(NULL)
    , m_pCurrentDataBuffer(nullptr)
    , m_DataBufferUsageMax(0)
    , m_PushDataEvent(false, false)
    , m_IsSyncThreadEnabled(false)
    , m_IsDataThreadEnabled(false)
    , m_IsDataSelected(false)
    , m_SyncThreadStackSize(0)
    , m_DataThreadStackSize(0)
{
    m_SessionMsgHandlerAdaptor.Initialize(this);

    m_Session.SetStateChangedCallback(&SpyController::SessionStateChangedCallback, this);

    m_pCurrentDataBuffer = &m_DataBuffer1;
    m_SyncThreadHandler.Initialize(this, &SpyController::SyncThreadMain);
    m_DataThreadHandler.Initialize(this, &SpyController::DataThreadMain);
}

//----------------------------------------------------------
void
SpyController::Initialize(const InitializeArg& arg, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    NN_USING_MIDDLEWARE(g_MiddlewareInfo);

    if(m_State >= State_Initialized)
    {
        return;
    }

    // バッファサイズチェック
    NN_SDK_ASSERT(bufferLength >= GetRequiredMemorySize(arg));

    nn::spy::detail::fnd::FrameHeap frameHeap;
    frameHeap.Initialize(buffer, bufferLength);

    // スレッドスタック
    m_pSyncThreadStack = frameHeap.Allocate(arg.syncThreadStackSize, nn::os::ThreadStackAlignment);
    m_pDataThreadStack = frameHeap.Allocate(arg.dataThreadStackSize, nn::os::ThreadStackAlignment);
    m_SyncThreadStackSize = arg.syncThreadStackSize;
    m_DataThreadStackSize = arg.dataThreadStackSize;

    NN_SDK_ASSERT_NOT_NULL(m_pSyncThreadStack);
    NN_SDK_ASSERT_NOT_NULL(m_pDataThreadStack);

    // バッファ
    size_t alignedDataBufferSize =  nn::spy::detail::fnd::RoundUp(
        arg.dataBufferLength,
        DataBufferAlignment);

    m_DataBuffer1.address = frameHeap.Allocate(alignedDataBufferSize, DataBufferAlignment);
    m_DataBuffer2.address = frameHeap.Allocate(alignedDataBufferSize, DataBufferAlignment);
    NN_SDK_ASSERT_NOT_NULL(m_DataBuffer1.address);
    NN_SDK_ASSERT_NOT_NULL(m_DataBuffer2.address);

    m_DataBuffer1.length = alignedDataBufferSize;
    m_DataBuffer2.length = alignedDataBufferSize;

#if defined(NN_SPY_DATA_FILE_CHANNEL_AVAILABLE)
    m_SpyDataFileChannel.Initialize(frameHeap.Allocate(detail::SpyDataFileChannel::GetRequiredMemorySize()));
#endif

    size_t restLength = frameHeap.GetFreeLength();

    m_Session.Initialize(
        frameHeap.Allocate(restLength),
        restLength);

    ClearDataBuffer();

    SetState(State_Initialized);

    // モジュール
    RegisterModule(m_TimeModule);
    RegisterModule(m_LogModule);
    RegisterModule(m_PlotModule);
    RegisterModule(m_MarkerModule);
    RegisterModule(m_DebugModule);
}

//----------------------------------------------------------
void
SpyController::Finalize() NN_NOEXCEPT
{
    if(m_State <= State_NotInitialized)
    {
        return;
    }

    Close();
    ClearDataBuffer();

    m_Session.Finalize();

    while (m_pModuleTop != nullptr)
    {
        UnregisterModule(*m_pModuleTop, true /* force */);
    }

    m_DataBuffer1.address = NULL;
    m_DataBuffer2.address = NULL;
    m_DataBuffer1.length = 0;
    m_DataBuffer2.length = 0;

#if defined(NN_SPY_DATA_FILE_CHANNEL_AVAILABLE)
    m_SpyDataFileChannel.Finalize();
#endif

    SetState(State_NotInitialized);
}

//----------------------------------------------------------
size_t
SpyController::GetRequiredMemorySize(const InitializeArg& arg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(arg.dataBufferLength > 0);
    NN_SDK_REQUIRES(arg.syncThreadStackSize > 0);
    NN_SDK_REQUIRES(arg.dataThreadStackSize > 0);


    size_t result = detail::SpySession::GetRequiredMemorySize();

#if defined(NN_SPY_DATA_FILE_CHANNEL_AVAILABLE)
    result += detail::SpyDataFileChannel::GetRequiredMemorySize();
#endif

    result += nn::os::ThreadStackAlignment + arg.syncThreadStackSize;
    result += nn::os::ThreadStackAlignment + arg.dataThreadStackSize;

    // ダブルバッファにするので、× 2
    result += DataBufferAlignment;
    result += nn::spy::detail::fnd::RoundUp(arg.dataBufferLength, DataBufferAlignment) * 2;

    return result;
}

//----------------------------------------------------------
bool
SpyController::Open(const OpenArg& arg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized(), "SpySession is not initialized.\n");

    if(m_State >= State_Opening)
    {
        return true;
    }

    SetState(State_Opening);

#if defined(NN_SPY_DATA_FILE_CHANNEL_AVAILABLE)
    m_SpyDataFileChannel.Open(arg.dataFileChannelParam);
#endif

    if(!StartSyncThread(arg.syncThreadPriority))
    {
        Close();
        return false;
    }

    if(!StartDataThread(arg.dataThreadPriority))
    {
        Close();
        return false;
    }

    return true;
}

//----------------------------------------------------------
void
SpyController::Close() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized(), "SpySession is not initialized.\n");

    if(m_State < State_Opening)
    {
        return;
    }

    SetState(State_Disconnecting);

    // NOTE:
    // ここで先にワーカースレッドの停止フラグを立ててから、
    // ワーカースレッドのブロッキングを解除します。
    // そうしないと次のブロッキングに入ってしまう可能性があります。
    m_IsDataThreadEnabled = false;
    m_IsSyncThreadEnabled = false;

    // m_IsDataThreadEnabled フラグを設定後に、PushData 待ちを解消する
    m_PushDataEvent.Set();

    // 同期 Read() から抜けさせる。
    m_Session.Close();

    m_DataThread.WaitForExit();
    m_SyncThread.WaitForExit();

    m_DataThread.Release();
    m_SyncThread.Release();

#if defined(NN_SPY_DATA_FILE_CHANNEL_AVAILABLE)
    m_SpyDataFileChannel.Close();
#endif

    SetState(State_Initialized);
}

//----------------------------------------------------------
void
SpyController::SetCurrentApplicationFrame(int value) NN_NOEXCEPT
{
    if(GetCurrentApplicationFrame() == value)
    {
        return;
    }

    m_TimeModule.SetCurrentApplicationFrame(value);
}

//----------------------------------------------------------
void
SpyController::SetCurrentAudioFrame(int value) NN_NOEXCEPT
{
    if(GetCurrentAudioFrame() == value)
    {
        return;
    }

    m_TimeModule.SetCurrentAudioFrame(value);
}

//----------------------------------------------------------
bool
SpyController::RegisterModule(SpyModule& module) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized() && !IsOpening() && !IsOpened());

    if (!module.IsRegistered())
    {
        // バージョン番号のチェック。
        NN_SDK_ASSERT(module.GetDataInfo().GetDataVersion() != 0);
        if (module.GetDataInfo().GetDataVersion() == 0)
        {
            return false;
        }

        // データタイプ名のチェック。
        {
            const char* dataName = module.GetDataInfo().GetDataName();
            NN_SDK_ASSERT_NOT_NULL(dataName);
            if (dataName == NULL)
            {
                return false;
            }
            else
            {
                size_t dataNameLength = strlen(dataName);
                NN_SDK_ASSERT_MINMAX(dataNameLength, 1u, 128u);
                if (dataNameLength < 1 || 128 < dataNameLength)
                {
                    return false;
                }
            }
        }

        // リンク
        NN_SDK_ASSERT(module.GetPrevious() == NULL && module.GetNext() == NULL, NN_TEXT_SPY("登録する SpyModule は、リンクが切れている必要があります。"));

        if(m_pModuleTop == NULL)
        {
            m_pModuleTop = &module;
            m_pModuleLast = &module;
        }
        else
        {
            NN_SDK_ASSERT(m_pModuleLast != NULL);
            m_pModuleLast->SetNext(&module);
            module.SetPrevious(m_pModuleLast);
            m_pModuleLast = &module;
        }
    }
    else
    {
        if(module.GetController() != this)
        {
            NN_DETAIL_SPY_WARN(NN_TEXT_SPY("[%s] 複数の SpyController には登録できません。\n"), NN_CURRENT_FUNCTION_NAME);
            return false;
        }
    }

    module.Attach(*this);
    DumpAllModules();

    return true;
}

//----------------------------------------------------------
void
SpyController::UnregisterModule(SpyModule& module) NN_NOEXCEPT
{
    UnregisterModule(module, false /* force */);
}

//----------------------------------------------------------
void
SpyController::UnregisterModule(SpyModule& module, bool force) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized() && !IsOpening() && !IsOpened());

    if(module.GetController() != this)
    {
        NN_SDK_ASSERT(false, NN_TEXT_SPY("この SpyController に登録されている SpyModule を指定してください。"));
        return;
    }

    if (force || module.GetRegisterCount() == 1)
    {
        // Previous -> Next をつなぐ
        // Top を更新する
        if (module.GetPrevious() != NULL)
        {
            module.GetPrevious()->SetNext(module.GetNext());
        }
        else
        {
            NN_SDK_ASSERT(m_pModuleTop == &module);
            m_pModuleTop = module.GetNext();
        }

        // Previous <- Next をつなぐ
        // Last を更新する
        if (module.GetNext() != NULL)
        {
            module.GetNext()->SetPrevious(module.GetPrevious());
        }
        else
        {
            NN_SDK_ASSERT(m_pModuleLast == &module);
            m_pModuleLast = module.GetPrevious();
        }

        // Current を更新する
        if (m_pModule == &module)
        {
            m_pModule = m_pModuleTop;
        }
    }


    module.Detach(force);

    DumpAllModules();
}

//----------------------------------------------------------
bool
SpyController::PushDataAt(
    SpyModule& module,
    const void* buffer,
    size_t length,
    nn::os::Tick tick) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(length > 0);

    // Push 対象でない場合は失敗させます。
    if(!module.IsRegistered() || !module.IsRequested())
    {
        return false;
    }

    return PushDataPacketAt(
        module.GetDataInfo().GetDataId(),
        buffer,
        length,
        tick);
}

//----------------------------------------------------------
bool
SpyController::PushDataAt(
    SpyModule& module,
    const void* const buffers[],
    const size_t lengths[],
    int count,
    nn::os::Tick tick) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(buffers);
    NN_SDK_REQUIRES(lengths);
    NN_SDK_REQUIRES(count >= 0);

    // Push 対象でない場合は失敗させます。
    if(!module.IsRegistered() || !module.IsRequested())
    {
        return false;
    }

    return PushDataPacketAt(
        module.GetDataInfo().GetDataId(),
        buffers,
        lengths,
        count,
        tick);
}

//----------------------------------------------------------
void
SpyController::ClearDataBuffer() NN_NOEXCEPT
{
    m_DataBuffer1.currentPosition = 0;
    m_DataBuffer2.currentPosition = 0;
}

//----------------------------------------------------------
void
SpyController::OnSessionStateChanged() NN_NOEXCEPT
{
    // セッションの準備が完了したら、接続を確立させます。
    if(m_Session.IsPrepared())
    {
        EstablishConnection();
    }

    if(!m_Session.IsConnected())
    {
        DismissConnection();
    }
}

//----------------------------------------------------------
void
SpyController::ResetSpyFrameBase() NN_NOEXCEPT
{
    // Dataパケットの送信時間(Tick)の基準になります。
    m_TickBase = nn::os::GetSystemTick();
}

//----------------------------------------------------------
bool
SpyController::StartSyncThread(int priority) NN_NOEXCEPT
{
    m_IsSyncThreadEnabled = true;

    nn::spy::detail::fnd::Thread::RunArgs args;
    args.name = SyncThreadName;
    args.handler = &m_SyncThreadHandler;
    args.priority = priority;
    args.stack = m_pSyncThreadStack;
    args.stackSize = m_SyncThreadStackSize;

    return m_SyncThread.Run(args);
}

//----------------------------------------------------------
bool
SpyController::StartDataThread(int priority) NN_NOEXCEPT
{
    m_IsDataThreadEnabled = true;

    m_PushDataEvent.Reset();

    nn::spy::detail::fnd::Thread::RunArgs args;
    args.name = DataThreadName;
    args.handler = &m_DataThreadHandler;
    args.priority = priority;
    args.stack = m_pDataThreadStack;
    args.stackSize = m_DataThreadStackSize;

    return m_DataThread.Run(args);
}

//----------------------------------------------------------
void
SpyController::SyncThreadMain(void* param) NN_NOEXCEPT
{
    NN_UNUSED(param);

    while(m_IsSyncThreadEnabled)
    {
        if(!m_Session.IsOpened())
        {
            // NOTE:
            // このタイミングでメインスレッドが SpyController::Close() を実行した場合、
            // その後に Sync スレッドがセッションを再度開いてしまいます。

            SetState(State_Opening);

            if(!m_Session.Open(SyncPort, DataPort))
            {
                nn::spy::detail::fnd::Thread::Sleep(
                    nn::spy::detail::fnd::TimeSpan::FromMilliSeconds(100));
                continue;
            }

            SetState(State_Opened);
        }

        if(!m_Session.IsConnected())
        {
            SetState(State_Connecting);

            if(m_Session.IsConnecting())
            {
                nn::spy::detail::fnd::Thread::Sleep(
                    nn::spy::detail::fnd::TimeSpan::FromMilliSeconds(100));
                continue;
            }

            if(!m_Session.Connect())
            {
                continue;
            }

            // NOTE : 初期化が終わるまで接続完了としない
        }

        if(!m_Session.ProcessSyncPacket(&m_SessionMsgHandlerAdaptor))
        {
            if (m_Session.IsOpened())
            {
                // パケットの読み込み待ち。
                nn::spy::detail::fnd::Thread::Sleep(
                    nn::spy::detail::fnd::TimeSpan::FromMilliSeconds(100));
            }
        }
    }

    // SpyController::Close() にてセッションは閉じられますが、
    // Sync スレッドがまた開いた可能性があるので確実に閉じておきます。
    SetState(State_Disconnecting);
    m_Session.Close();
}

//----------------------------------------------------------
void
SpyController::DataThreadMain(void* param) NN_NOEXCEPT
{
    NN_UNUSED(param);

    for(;;)
    {
        if(!m_IsDataThreadEnabled)
        {
            break;
        }

        if(!IsPrepared())
        {
            nn::spy::detail::fnd::Thread::Sleep(
                nn::spy::detail::fnd::TimeSpan::FromMilliSeconds(100));
            continue;
        }

        if(GetCurrentDataBuffer().currentPosition == 0)
        {
            m_PushDataEvent.Wait();
            continue;
        }

        SwapBuffer();
        SendDataPacket();
    }
}

//----------------------------------------------------------
void
SpyController::SetState(State value) NN_NOEXCEPT
{
    if(m_State == value)
    {
        return;
    }

    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

#ifdef STATE_DEBUG_ENABLED
    NN_DETAIL_SPY_INFO("[nn::spy::SpyController] State = %s --> %s\n", StateToString(m_State), StateToString(value));
#endif

    m_State = value;
}

//----------------------------------------------------------
void
SpyController::EstablishConnection() NN_NOEXCEPT
{
    InitializeDataInfo();

    // Initializeパケットを受信するとState_Preparing状態に入ります。
    // 付加的な初期化パケットを受信したのち、最後に、
    // 有効なフラグを持ったSelectDataIdパケットを受信すると
    // State_Prepared 状態に遷移します。
    SetState(State_Preparing);
}

//----------------------------------------------------------
void
SpyController::DismissConnection() NN_NOEXCEPT
{
    FinalizeDataInfo();
}

//----------------------------------------------------------
bool
SpyController::PushDataPacketAt(
    nn::spy::detail::SpyDataId dataId,
    const void* buffer,
    size_t length,
    nn::os::Tick tick) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(dataId, nn::spy::detail::SpyDataId_Min, nn::spy::detail::SpyDataId_Max);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(length > 0);

    // 準備が完了していない場合は失敗させます。
    if(!IsPrepared())
    {
        return false;
    }

    nn::spy::detail::fnd::ScopedCriticalSection dataLock(m_DataBufferLock);

    if (tick < m_TickBase)
    {
        return false;
    }

    detail::DataPacket* packet = AllocateDataPacketBuffer(length);

    if(packet == NULL)
    {
        NN_DETAIL_SPY_WARN("[%s] out of memory.\n", NN_CURRENT_FUNCTION_NAME);
        return false;
    }

    packet->SetTimestamp((tick - m_TickBase).ToTimeSpan().GetMicroSeconds());
    packet->SetLengths(length);
    packet->body.dataId = dataId;
    memcpy(packet->GetPayload(), buffer, length);

#ifdef BUFFER_DEBUG_ENABLED
    NN_DETAIL_SPY_INFO(
        "[nn::spy::SpyController] PushDataAt : Current(%8zd/%8zu), Send(%8zd/%8zu).\n",
        GetCurrentDataBuffer().currentPosition,
        GetCurrentDataBuffer().length,
        GetSendDataPacketBuffer().currentPosition,
        GetSendDataPacketBuffer().length);
#endif

    m_DataBufferUsageMax = std::max(m_DataBufferUsageMax, static_cast<size_t>(GetCurrentDataBuffer().currentPosition));

    m_PushDataEvent.Set();

    return true;
}

//----------------------------------------------------------
bool
SpyController::PushDataPacketAt(
    nn::spy::detail::SpyDataId dataId,
    const void* const buffers[],
    const size_t lengths[],
    int count,
    nn::os::Tick tick) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(dataId, nn::spy::detail::SpyDataId_Min, nn::spy::detail::SpyDataId_Max);
    NN_SDK_REQUIRES_NOT_NULL(buffers);
    NN_SDK_REQUIRES_NOT_NULL(lengths);
    NN_SDK_REQUIRES(count >= 0);

    // 準備が完了していない場合は失敗させます。
    if(!IsPrepared())
    {
        return false;
    }

    if (count <= 0)
    {
        return true;
    }

    size_t length = 0;
    for (int i = 0; i < count; ++i)
    {
        NN_SDK_ASSERT_NOT_NULL(buffers[i]);
        NN_SDK_ASSERT(lengths[i] > 0);
        length += lengths[i];
    }

    nn::spy::detail::fnd::ScopedCriticalSection dataLock(m_DataBufferLock);

    detail::DataPacket* packet = AllocateDataPacketBuffer(length);

    if(packet == NULL)
    {
        NN_DETAIL_SPY_WARN("[%s] out of memory.\n", NN_CURRENT_FUNCTION_NAME);
        return false;
    }

    packet->SetTimestamp((tick - m_TickBase).ToTimeSpan().GetMicroSeconds());
    packet->SetLengths(length);
    packet->body.dataId = dataId;
    void* pPayload = packet->GetPayload();

    for (int i = 0; i < count; ++i)
    {
        memcpy(pPayload, buffers[i], lengths[i]);
        pPayload = detail::fnd::AddOffsetToPtr(pPayload, lengths[i]);
    }

#ifdef BUFFER_DEBUG_ENABLED
    NN_DETAIL_SPY_INFO(
        "[nn::spy::SpyController] PushDataAt : Current(%8zd/%8zu), Send(%8zd/%8zu).\n",
        GetCurrentDataBuffer().currentPosition,
        GetCurrentDataBuffer().length,
        GetSendDataPacketBuffer().currentPosition,
        GetSendDataPacketBuffer().length);
#endif

    m_DataBufferUsageMax = std::max(m_DataBufferUsageMax, static_cast<size_t>(GetCurrentDataBuffer().currentPosition));

    m_PushDataEvent.Set();

    return true;
}

//----------------------------------------------------------
detail::DataPacket*
SpyController::AllocateDataPacketBuffer(size_t payloadLength) NN_NOEXCEPT
{
    // m_DataBufferLock でロックがかかった状態で呼び出されることを想定しており、
    // この関数内ではロックしません。

    size_t requiredLength = detail::DataPacket::GetRequiredMemorySize(payloadLength);

    if(GetCurrentDataBuffer().length - GetCurrentDataBuffer().currentPosition < requiredLength)
    {
        // メモリ不足
        return NULL;
    }

    void* buffer = nn::spy::detail::fnd::AddOffsetToPtr(
        GetCurrentDataBuffer().address,
        GetCurrentDataBuffer().currentPosition);

    GetCurrentDataBuffer().currentPosition += requiredLength;

    return new(buffer) detail::DataPacket;
}

//----------------------------------------------------------
void
SpyController::SendDataPacket() NN_NOEXCEPT
{
    if(GetSendDataPacketBuffer().currentPosition == 0)
    {
        return;
    }

#ifdef BUFFER_DEBUG_ENABLED
    // データが溜まっていない場合はログ出力しない
    if(GetSendDataPacketBuffer().currentPosition > 0)
    {
        NN_DETAIL_SPY_INFO(
            "[nn::spy::SpyController] SendData : (%8zd/%8zu).\n",
            GetSendDataPacketBuffer().currentPosition,
            GetSendDataPacketBuffer().length);
    }
#endif

#if defined(NN_SPY_DATA_FILE_CHANNEL_AVAILABLE)
    bool bOutputByFile = m_SpyDataFileChannel.WriteData(
        this->GetSendDataPacketBuffer().address,
        this->GetSendDataPacketBuffer().currentPosition);
#else
    bool bOutputByFile = false;
#endif

    if (bOutputByFile)
    {
        // Dataチャンネルの生存を通知。
        detail::PingPacket* packet = new(GetSendDataPacketBuffer().address) detail::PingPacket();
        m_Session.SendData(packet, sizeof(*packet));
    }
    else
    {
         // ファイルの代わりにDataチャンネルで送信する。
         m_Session.SendData(
             GetSendDataPacketBuffer().address,
             GetSendDataPacketBuffer().currentPosition);
    }

    GetSendDataPacketBuffer().currentPosition = 0;
}

//----------------------------------------------------------
bool
SpyController::SendSetOutputDirPathReplyPacket(bool active) NN_NOEXCEPT
{
#ifdef BUFFER_DEBUG_ENABLED
    NN_DETAIL_SPY_INFO("[nn::spy::SpyController] SendSetOutputDirPathReplyPacket(active=%s).\n", active? "true" : "false");
#endif

    // 準備中でない場合は失敗させます。
    if (!IsPreparing())
    {
        return false;
    }

    // 有効なデータIDフラグが設定されている場合、バックグラウンドスレッドで
    // Dataチャンネルへの出力が行われているので、ここでDataチャンネルへの
    // 書き込みを行うことはできない。
    if (m_IsDataSelected)
    {
        NN_SDK_ASSERT(!m_IsDataSelected);
        return false;
    }

    nn::spy::detail::fnd::ScopedCriticalSection lock(m_DataBufferLock);

    detail::SetOutputDirReplyPacket packet;
    packet.header.resultCode = active? 1 : 0;

    if(m_Session.SendData(&packet, sizeof(packet)) != sizeof(packet))
    {
        NN_DETAIL_SPY_WARN(NN_TEXT_SPY("[%s] SetOutputDirReplyPacket を送れませんでした。\n"), NN_CURRENT_FUNCTION_NAME);
        return false;
    }

    return true;
}

//----------------------------------------------------------
void
SpyController::SwapBuffer() NN_NOEXCEPT
{
    nn::spy::detail::fnd::ScopedCriticalSection lock(m_DataBufferLock);

    m_pCurrentDataBuffer = &GetSendDataPacketBuffer();
}

//----------------------------------------------------------
size_t
SpyController::GetDataBufferUsageMax() NN_NOEXCEPT
{
    nn::spy::detail::fnd::ScopedCriticalSection lock(m_DataBufferLock);

    size_t result = m_DataBufferUsageMax;
    m_DataBufferUsageMax = 0;

    return result;
}

//----------------------------------------------------------
void
SpyController::SelectDataIds(int flagsCount, uint32_t flags[]) NN_NOEXCEPT
{
    // Initializeパケットを受信するとState_Preparing状態に入ります。
    // 付加的な初期化パケットを受信したのち、最後に、
    // 有効なフラグを持ったSelectDataIdパケットを受信すると
    // State_Prepared 状態に遷移します。
    if (IsPreparing() && flagsCount > 0)
    {
        ResetSpyFrameBase();
        SetState(State_Prepared);
    }

    bool dataSelected = false;
    for (SpyModule* module = m_pModuleTop;
         module != NULL;
         module = module->GetNext())
    {
        int dataId = module->GetDataInfo().GetDataId();

        int index = dataId / 32;
        uint32_t mask = 1u << (dataId % 32);

        if (index < flagsCount && (flags[index] & mask) != 0)
        {
            module->OnRequested(true);
            dataSelected = true;
        }
        else
        {
            module->OnRequested(false);
        }
    }

    m_IsDataSelected = dataSelected;

    DumpAllModules();
}

//----------------------------------------------------------
void
SpyController::InitializeDataInfo() NN_NOEXCEPT
{
    int dataId = 0;
    for (SpyModule* module = m_pModuleTop;
        module != NULL;
        module = module->GetNext())
    {
        module->AllocateDataId(static_cast<nn::spy::detail::SpyDataId>(++dataId));
        module->OnSessionStarted();
    }

    m_pModule = m_pModuleTop;
    m_IsDataSelected = false;
}

//----------------------------------------------------------
void
SpyController::FinalizeDataInfo() NN_NOEXCEPT
{
    for (SpyModule* module = m_pModuleTop;
        module != NULL;
        module = module->GetNext())
    {
        module->OnRequested(false);
    }

    m_IsDataSelected = false;
}

//----------------------------------------------------------
const SpyDataInfo*
SpyController::QueryDataInfo() NN_NOEXCEPT
{
    SpyModule* module = m_pModule;

    if (module == NULL)
    {
        return NULL;
    }
    else
    {
        m_pModule = module->GetNext();
    }

    return &module->GetDataInfo();
}

//----------------------------------------------------------
void
SpyController::DumpAllModules() NN_NOEXCEPT
{
#if defined(MODULE_DEBUG_ENABLED)
    NN_DETAIL_SPY_INFO("-- all SpyModules --------------------\n");

    for (SpyModule* module = m_pModuleTop;
        module != NULL;
        module = module->GetNext())
    {
        NN_DETAIL_SPY_INFO("[0x%08x] ID=%2d Prev=0x%p Next=0x%p : %s\n",
            module,
            module->GetDataInfo().GetDataId(),
            module->GetPrevious(),
            module->GetNext(),
            module->GetDataInfo().GetDataName());
    }

    NN_DETAIL_SPY_INFO("--------------------------------------\n");
#endif
}

//----------------------------------------------------------
const char*
SpyController::StateToString(State value) NN_NOEXCEPT
{
#if defined(STATE_DEBUG_ENABLED)
    static const char* strings[] = {
        "State_NotInitialized",
        "State_Initialized",
        "State_Opening",
        "State_Opened",
        "State_Disconnecting",
        "State_Connecting",
        "State_Preparing",
        "State_Prepared"
    };

    return strings[value];
#else
    NN_UNUSED(value);
    return NULL;
#endif
}

} // namespace nn::spy
} // namespace nn

#endif // NN_BUILD_CONFIG_SPY_ENABLED
