﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "precompiled.h"

#include <nn/diag/text/diag_SdkTextSpy.h>

#include <nn/spy/detail/fnd/string/spyfnd_String.h>
#include <nn/spy/spy_PlotModule.h>
#include <nn/spy/spy_PlotNode.h>
#include <nn/spy/spy_PlotState.h>

namespace nn {
namespace spy {

NN_DEFINE_STATIC_CONSTANT(const int PlotState::MaxStateValueLength);

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

//------------------------------------------------------------------------------
PlotState::PlotState() NN_NOEXCEPT
{
    InitializePacketCommonData();
    SetColor(0xF3, 0x98, 0x00); // orange
}

//------------------------------------------------------------------------------
PlotState::PlotState(const char* name) NN_NOEXCEPT
{
    InitializePacketCommonData();
    SetName(name);
    SetColor(0xF3, 0x98, 0x00); // orange
}

//------------------------------------------------------------------------------
PlotState::PlotState(const char* name, uint8_t r, uint8_t g, uint8_t b) NN_NOEXCEPT
{
    InitializePacketCommonData();
    SetName(name);
    SetColor(r, g, b);
}

//------------------------------------------------------------------------------
void PlotState::InitializePacketCommonData() NN_NOEXCEPT
{
    std::memset(&m_MetadataPacketPayload, 0, sizeof(m_MetadataPacketPayload));
    InitializeItemMetaData(&m_MetadataPacketPayload.itemMetaData);
    m_MetadataPacketPayload.common.dataType = nn::spy::detail::SpyPlotItemPacket::DataType_StateMetadata;

    std::memset(&m_ValuePacketPayload, 0, sizeof(m_ValuePacketPayload));
    m_ValuePacketPayload.common.dataType = nn::spy::detail::SpyPlotItemPacket::DataType_StateValue;
    m_ValuePacketPayload.id = m_MetadataPacketPayload.itemMetaData.id;
}

//------------------------------------------------------------------------------
PlotState::~PlotState() NN_NOEXCEPT
{
}

//------------------------------------------------------------------------------
void PlotState::PushValue(const char* state) NN_NOEXCEPT
{
    if(!IsRequested())
    {
        return;
    }

    PushValueAt(state, nn::os::GetSystemTick());
}

//------------------------------------------------------------------------------
void PlotState::PushValue(const char* state, uint8_t r, uint8_t g, uint8_t b) NN_NOEXCEPT
{
    if(!IsRequested())
    {
        return;
    }

    PushValueAt(state, r, g, b, nn::os::GetSystemTick());
}

//------------------------------------------------------------------------------
void PlotState::PushValueAt(const char* state, nn::os::Tick tick) NN_NOEXCEPT
{
    if(!IsRequested())
    {
        return;
    }

#if defined(NN_DETAIL_ENABLE_SDK_ASSERT)
    CheckTick(tick);
#endif

    // メタデータをプッシュしていない場合は先にプッシュする
    if(!IsMetadataPushed())
    {
        PushMetaData();
    }

    m_ValuePacketPayload.state.Set(state);
    m_ValuePacketPayload.color = m_MetadataPacketPayload.itemMetaData.color;

    PushDataAt(&m_ValuePacketPayload, offsetof(nn::spy::detail::SpyPlotItemPacket::StateValuePacketPayload, state.text) + m_ValuePacketPayload.state.length + 1, tick);
}

//------------------------------------------------------------------------------
void PlotState::PushValueAt(const char* state, uint8_t r, uint8_t g, uint8_t b, nn::os::Tick tick) NN_NOEXCEPT
{
    if(!IsRequested())
    {
        return;
    }

#if defined(NN_DETAIL_ENABLE_SDK_ASSERT)
    CheckTick(tick);
#endif

    // メタデータをプッシュしていない場合は先にプッシュする
    if(!IsMetadataPushed())
    {
        PushMetaData();
    }

    m_ValuePacketPayload.state.Set(state);
    m_ValuePacketPayload.color.Set(r, g, b);

    PushDataAt(&m_ValuePacketPayload, offsetof(nn::spy::detail::SpyPlotItemPacket::StateValuePacketPayload, state.text) + m_ValuePacketPayload.state.length + 1, tick);
}

//------------------------------------------------------------------------------
void PlotState::PushOwnMetaData() NN_NOEXCEPT
{
    if(!PushData(m_MetadataPacketPayload))
    {
        NN_DETAIL_SPY_WARN(NN_TEXT_SPY("[%s] PlotState '%s' のメタデータ送信に失敗しました。\n"), NN_CURRENT_FUNCTION_NAME, GetName());
    }
}

#endif // NN_BUILD_CONFIG_SPY_ENABLED

} // namespace nn::spy
} // namespace nn
