﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "precompiled.h"

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <nn/diag/text/diag_SdkTextSpy.h>

#include <nn/spy/spy_PlotItem.h>
#include <nn/spy/spy_PlotModule.h>
#include <nn/spy/spy_PlotNode.h>
#include <nn/spy/spy_PlotState.h>
#include <nn/spy/spy_SpyController.h>

namespace nn {
namespace spy {

//----------------------------------------------------------
PlotModule::PlotModule() NN_NOEXCEPT
    : SpyModule("Plot", nn::spy::detail::SpyPlotItemPacket::PacketVersion)
    , m_pItemTop(NULL)
    , m_pItemLast(NULL)
    , m_IsMetadataPushed(false)
    , m_IsResetPushed(false)
{
}

//----------------------------------------------------------
bool PlotModule::AttachItem(PlotItem& item) NN_NOEXCEPT
{
    if(item.IsAttached())
    {
        if (item.GetModule() != this)
        {
            NN_DETAIL_SPY_WARN(NN_TEXT_SPY("[%s] SpyPlotItem は、複数の SpyPlotModule にアタッチできません。\n"), NN_CURRENT_FUNCTION_NAME);
            return false;
        }

        return true;
    }

    if(m_pItemTop == NULL)
    {
        m_pItemTop = &item;
        m_pItemLast = &item;
    }
    else
    {
        NN_SDK_ASSERT(m_pItemLast != NULL);
        m_pItemLast->SetNext(&item);
        item.SetPrevious(m_pItemLast);
        m_pItemLast = &item;
    }

    item.Attach(*this);

    return true;
}

//----------------------------------------------------------
void PlotModule::DetachItem(PlotItem& item) NN_NOEXCEPT
{
    if (!item.IsAttached())
    {
        return;
    }

    if (item.GetModule() != this)
    {
        NN_SDK_ASSERT(false, NN_TEXT_SPY("アイテムは別のモジュールにアタッチされています。"));
        return;
    }

    // Previous -> Next をつなぐ
    // Top を更新する
    if(item.GetPrevious() != NULL)
    {
        item.GetPrevious()->SetNext(item.GetNext());
    }
    else
    {
        NN_SDK_ASSERT(m_pItemTop == &item);
        m_pItemTop = item.GetNext();
    }

    // Previous <- Next をつなぐ
    // Last を更新する
    if(item.GetNext() != NULL)
    {
        item.GetNext()->SetPrevious(item.GetPrevious());
    }
    else
    {
        NN_SDK_ASSERT(m_pItemLast == &item);
        m_pItemLast = item.GetPrevious();
    }

    item.Detach();
}

//------------------------------------------------------------------------------
void PlotModule::OnSessionStarted() NN_NOEXCEPT
{
    m_IsMetadataPushed = false;
    m_IsResetPushed = false;

    SpyModule::OnSessionStarted();
}

//------------------------------------------------------------------------------
void PlotModule::OnRequested(bool isRequested) NN_NOEXCEPT
{
    SpyModule::OnRequested(isRequested);

    if(isRequested)
    {
        m_IsResetPushed = false;
    }

    // Plot データが要求されたら、メタデータを送信するように設定する
    if(isRequested && !m_IsMetadataPushed)
    {
        for (PlotItem* pItem = m_pItemTop; pItem != NULL; pItem = pItem->GetNext())
        {
            pItem->InvalidateMetadata();
        }

        m_IsMetadataPushed = true;
    }
}

bool PlotModule::PushData(const void* buffer, size_t length) NN_NOEXCEPT
{
    if (!m_IsResetPushed)
    {
        m_IsResetPushed = true;

        nn::spy::detail::SpyPlotItemPacket::ResetPacketPayload payload;
        std::memset(&payload, 0, sizeof(payload));
        payload.common.dataType = nn::spy::detail::SpyPlotItemPacket::DataType_Reset;

        SpyModule::PushData(&payload, sizeof(payload));
    }

    return SpyModule::PushData(buffer, length);
}

bool PlotModule::PushDataAt(const void* buffer, size_t length, nn::os::Tick tick) NN_NOEXCEPT
{
    if (!m_IsResetPushed)
    {
        m_IsResetPushed = true;

        nn::spy::detail::SpyPlotItemPacket::ResetPacketPayload payload;
        std::memset(&payload, 0, sizeof(payload));
        payload.common.dataType = nn::spy::detail::SpyPlotItemPacket::DataType_Reset;

        SpyModule::PushData(&payload, sizeof(payload));
    }

    return SpyModule::PushDataAt(buffer, length, tick);
}

} // namespace nn::spy
} // namespace nn

#endif // NN_BUILD_CONFIG_SPY_ENABLED
