﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <nn/spy/detail/spy_SpySession.h>

#include <nn/spy/detail/fnd/basis/spyfnd_Memory.h>
#include <nn/spy/detail/fnd/hio/spyfnd_HioChannel.h>
#include <nn/spy/detail/fnd/string/spyfnd_String.h>
#include <nn/spy/detail/spy_Packet.h>

#if defined(NN_SDK_BUILD_DEBUG)
// #define STATE_DEBUG_ENABLED
// #define COM_DEBUG_ENABLED
#endif

namespace nn {
namespace spy {
namespace detail {

//----------------------------------------------------------
SpySession::SpySession() NN_NOEXCEPT
    : m_State(State_NotInitialized)
    , m_StateChangedCallback(NULL)
    , m_StateChangedCallbackParam(NULL)
    , m_SyncChannelBuffer(NULL)
    , m_SyncChannelBufferLength(0)
{
}

//----------------------------------------------------------
void
SpySession::Initialize(void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

    if(m_State >= State_Initialized)
    {
        return;
    }

    // バッファサイズチェック
    NN_SDK_ASSERT(bufferLength >= GetRequiredMemorySize());

    m_SyncChannelBuffer = nn::spy::detail::fnd::RoundUp(buffer, nn::spy::detail::fnd::MemoryTraits::DefaultAlignment);
    m_SyncChannelBufferLength = bufferLength - nn::spy::detail::fnd::GetOffsetFromPtr(buffer, m_SyncChannelBuffer);

    SetState(State_Initialized);
}

//----------------------------------------------------------
void
SpySession::Finalize() NN_NOEXCEPT
{
    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

    if(m_State <= State_NotInitialized)
    {
        return;
    }

    Close();

    m_SyncChannelBuffer = NULL;
    m_SyncChannelBufferLength = 0;

    SetState(State_NotInitialized);
}

//----------------------------------------------------------
size_t
SpySession::GetRequiredMemorySize() NN_NOEXCEPT
{
    size_t maxPacketLength = sizeof(InitializePacket);
    maxPacketLength = nn::spy::detail::fnd::Max(maxPacketLength, sizeof(InitializeReplyPacket));
    maxPacketLength = nn::spy::detail::fnd::Max(maxPacketLength, sizeof(FinalizePacket));
    maxPacketLength = nn::spy::detail::fnd::Max(maxPacketLength, sizeof(FinalizeReplyPacket));
    maxPacketLength = nn::spy::detail::fnd::Max(maxPacketLength, sizeof(PingPacket));
    maxPacketLength = nn::spy::detail::fnd::Max(maxPacketLength, sizeof(PongPacket));
    maxPacketLength = nn::spy::detail::fnd::Max(maxPacketLength, sizeof(SetOutputDirPacket));
    maxPacketLength = nn::spy::detail::fnd::Max(maxPacketLength, sizeof(SetOutputDirReplyPacket));
    maxPacketLength = nn::spy::detail::fnd::Max(maxPacketLength, sizeof(NotifyDataReadPacket));
    maxPacketLength = nn::spy::detail::fnd::Max(maxPacketLength, sizeof(NotifyDataReadReplyPacket));
    maxPacketLength = nn::spy::detail::fnd::Max(maxPacketLength, sizeof(QueryDataInfoPacket));
    maxPacketLength = nn::spy::detail::fnd::Max(maxPacketLength, sizeof(QueryDataInfoReplyPacket));

    return
        nn::spy::detail::fnd::RoundUp(maxPacketLength, nn::spy::detail::fnd::MemoryTraits::DefaultAlignment);
}

//----------------------------------------------------------
bool
SpySession::Open(const char* syncPort, const char* dataPort) NN_NOEXCEPT
{
    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

    NN_SDK_ASSERT(IsInitialized(), "SpySession is not initialized.\n");

    if(m_State >= State_Opening)
    {
        return true;
    }

    SetState(State_Opening);

    if(!OpenSyncChannel(syncPort))
    {
        Close();
        return false;
    }

    if(!m_DataChannel.Open(dataPort))
    {
        Close();
        return false;
    }

    SetState(State_Opened);

    return true;
}

//----------------------------------------------------------
void
SpySession::Close() NN_NOEXCEPT
{
    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

    NN_SDK_ASSERT(IsInitialized(), "SpySession is not initialized.\n");

    if(m_State < State_Opening)
    {
        return;
    }

    Disconnect();

    m_DataChannel.Close();
    CloseSyncChannel();

    SetState(State_Initialized);
}

//----------------------------------------------------------
bool
SpySession::Connect() NN_NOEXCEPT
{
    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

    if(IsConnected())
    {
        return true;
    }

    if(m_State < State_Opened)
    {
        return false;
    }

    SetState(State_Connecting);

    m_StateLock.Leave();

    if(!m_SyncChannel.Connect())
    {
        m_StateLock.Enter();

        Disconnect();
        return false;
    }

    if(!m_DataChannel.Connect())
    {
        m_StateLock.Enter();

        Disconnect();
        return false;
    }

    m_StateLock.Enter();

    // Sync, Data チャンネルの Connect() 中に別スレッドでセッションが Close() 等された場合。
    if(m_State != State_Connecting)
    {
        return false;
    }

    SetState(State_Preparing);

    return true;
}

//----------------------------------------------------------
void
SpySession::Disconnect() NN_NOEXCEPT
{
    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

    if(m_State < State_Connecting)
    {
        return;
    }

    SetState(State_Disconnecting);

    m_DataChannel.Disconnect();
    m_SyncChannel.Disconnect();

    SetState(State_Opened);
}

//----------------------------------------------------------
bool
SpySession::ProcessSyncPacket(IMessageHandler* msgHandler) NN_NOEXCEPT
{
    if(!IsConnected())
    {
        return false;
    }

    if(!m_SyncChannel.IsConnected() || !m_DataChannel.IsConnected())
    {
        Close();
        return false;
    }

    const PacketHeader* packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();

    if(packetHeader == NULL)
    {
        // 次のパケットヘッダを読む。
        NN_SDK_ASSERT(m_SyncPacketReader.CanReadHeader());
        m_SyncPacketReader.ReadPacketHeader();

        packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();

        if(packetHeader == NULL)
        {
            return false;
        }
    }

    bool completeProcess = false;

    if(IsPreparing())
    {
        // 準備中は、初期化パケットしか受け付けない。
        // それ以外のパケットはスキップする。
        switch (packetHeader->id.value)
        {
        case PacketId::Type_Initialize:
            completeProcess = OnInitialize(msgHandler);
            break;

        default:
            completeProcess = m_SyncPacketReader.ReadPacketBody(GetPacketBodyBuffer(), GetPacketBodyBufferSize());
            break;
        }
    }
    else
    {
        switch (packetHeader->id.value)
        {
        case PacketId::Type_Ping:
            completeProcess = OnPing();
            break;

        case PacketId::Type_SelectDataId:
            completeProcess = OnSelectDataId(msgHandler);
            break;

        case PacketId::Type_SetOutputDir:
            completeProcess = OnSetOutputDir(msgHandler);
            break;

        case PacketId::Type_NotifyDataRead:
            completeProcess = OnNotifyDataRead(msgHandler);
            break;

        case PacketId::Type_Finalize:
            completeProcess = OnFinalize(msgHandler);
            break;

        case PacketId::Type_QueryDataInfo:
            completeProcess = OnQueryDataInfo(msgHandler);
            break;

        default:
            NN_ABORT("unknown packet '%08x'\n", packetHeader->id.value);
            break;
        }
    }

    if(!completeProcess)
    {
        return true;
    }

    // パケット処理が完了した場合は、GetCurrentPacketHeader() で NULL が返るように Next() する
    if(m_SyncPacketReader.GetState() == SpyPacketReader::State_ReadBody)
    {
        m_SyncPacketReader.Next();
    }

    return true;
}

//----------------------------------------------------------
size_t
SpySession::SendData(const void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);

    if(!m_DataChannel.IsConnected())
    {
        return 0;
    }

    return m_DataChannel.Write(buffer, length);
}

//----------------------------------------------------------
void
SpySession::SetState(State value) NN_NOEXCEPT
{
    if(m_State == value)
    {
        return;
    }

#ifdef STATE_DEBUG_ENABLED
    NN_DETAIL_SPY_INFO("[SpySession] State = %s --> %s\n", StateToString(m_State), StateToString(value));
#endif

    m_State = value;

    if(m_StateChangedCallback != NULL)
    {
        m_StateChangedCallback(m_StateChangedCallbackParam);
    }
}

//----------------------------------------------------------
bool
SpySession::OpenSyncChannel(const char* port) NN_NOEXCEPT
{
    if(!m_SyncChannel.Open(port))
    {
        return false;
    }

    m_SyncPacketReader.Initialize(&m_SyncChannel, m_SyncChannelBuffer);

    return true;
}

//----------------------------------------------------------
void
SpySession::CloseSyncChannel() NN_NOEXCEPT
{
    m_SyncPacketReader.Finalize();
    m_SyncChannel.Close();
}

//----------------------------------------------------------
void*
SpySession::GetPacketBodyBuffer() NN_NOEXCEPT
{
    if(m_SyncChannelBuffer == NULL) { return NULL; }

    if(m_SyncPacketReader.GetState() < SpyPacketReader::State_ReadHeader)
    {
        return NULL;
    }

    return nn::spy::detail::fnd::AddOffsetToPtr(m_SyncChannelBuffer, sizeof(PacketHeader));
}

//----------------------------------------------------------
bool
SpySession::OnInitialize(IMessageHandler* msgHandler) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(msgHandler);
    NN_SDK_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::State_ReadHeader);

    if(!m_SyncPacketReader.ReadPacketBody(GetPacketBodyBuffer(), GetPacketBodyBufferSize()))
    {
        return false;
    }

    const PacketHeader* packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();
    NN_SDK_ASSERT_NOT_NULL(packetHeader);
    NN_SDK_ASSERT(packetHeader->bodyLength >= sizeof(InitializePacket::Body));
    NN_UNUSED(packetHeader);

#if defined(COM_DEBUG_ENABLED)
    NN_DETAIL_SPY_INFO("[SpySession] OnInitialize()\n");
#endif

    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

    if (m_State != State_Preparing)
    {
        // セッションが外部から閉じられた場合。
        return false;
    }

    if(msgHandler != NULL)
    {
        msgHandler->OnInitializeSession();
    }

    InitializeReplyPacket* replyPacket = new(m_SyncChannelBuffer) InitializeReplyPacket();
    m_SyncChannel.Write(replyPacket, sizeof(InitializeReplyPacket));

    SetState(State_Prepared);

    return true;
}

//----------------------------------------------------------
bool
SpySession::OnSetOutputDir(IMessageHandler* msgHandler) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(msgHandler);
    NN_SDK_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::State_ReadHeader);

    if(!m_SyncPacketReader.ReadPacketBody(GetPacketBodyBuffer(), GetPacketBodyBufferSize()))
    {
        return false;
    }

    const PacketHeader* packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();
    NN_SDK_ASSERT_NOT_NULL(packetHeader);
    NN_SDK_ASSERT(packetHeader->bodyLength >= offsetof(SetOutputDirPacket::Body, path));

    SetOutputDirPacket::Body* body = reinterpret_cast<SetOutputDirPacket::Body*>(GetPacketBodyBuffer());

    // パス文字列がパケットに収まっているか。
    if (body != NULL && !(body->length > 0 &&
        offsetof(SetOutputDirPacket::Body, length) + body->length <= packetHeader->bodyLength))
    {
        NN_ABORT("invalid SetOutputDirPacket (%d).", body->length);
        body = NULL;
    }

    // パス文字列はヌル終端されているか。
    if (body != NULL && !(body->path[body->length - 1] == '\0'))
    {
        NN_ABORT("SetOutputDirPacket::path string is not null-terminated.");
        body = NULL;
    }

#if defined(COM_DEBUG_ENABLED)
    NN_DETAIL_SPY_INFO("[SpySession] OnSetOutputDir(path = '%s').\n", body? body->path : "<invalid>");
#endif

    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

    if (m_State != State_Prepared)
    {
        // セッションが外部から閉じられた場合。
        return false;
    }

    if(msgHandler != NULL)
    {
        msgHandler->OnSetOutputDirPath(body? body->path : NULL);
    }

    SetOutputDirReplyPacket* replyPacket = new(m_SyncChannelBuffer) SetOutputDirReplyPacket();
    m_SyncChannel.Write(replyPacket, sizeof(SetOutputDirReplyPacket));

    return true;
}

//----------------------------------------------------------
bool
SpySession::OnSelectDataId(IMessageHandler* msgHandler) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(msgHandler);
    NN_SDK_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::State_ReadHeader);

    if(!m_SyncPacketReader.ReadPacketBody(GetPacketBodyBuffer(), GetPacketBodyBufferSize()))
    {
        return false;
    }

    const PacketHeader* packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();
    NN_SDK_ASSERT_NOT_NULL(packetHeader);
    NN_SDK_ASSERT(packetHeader->bodyLength >= offsetof(SelectDataIdPacket::Body, selectionFlags));
    NN_UNUSED(packetHeader);

    SelectDataIdPacket::Body* body = reinterpret_cast<SelectDataIdPacket::Body*>(GetPacketBodyBuffer());
    NN_SDK_ASSERT_NOT_NULL(body);
    NN_SDK_ASSERT(packetHeader->bodyLength == offsetof(SelectDataIdPacket::Body, selectionFlags) + sizeof(uint32_t) * body->selectionFlagsLength);

#if defined(COM_DEBUG_ENABLED)
    NN_DETAIL_SPY_INFO("[SpySession] OnSelectDataId() : selectionFlagsLength = %d.\n", body->selectionFlagsLength);
    for (int i = 0; i < static_cast<int>(body->selectionFlagsLength); ++i)
    {
        NN_DETAIL_SPY_INFO("  flags[%d] = %08x\n", i, body->selectionFlags[i]);
    }
#endif

    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

    if (m_State != State_Prepared)
    {
        // セッションが外部から閉じられた場合。
        return false;
    }

    if(msgHandler != NULL && body != NULL)
    {
        msgHandler->OnSelectDataId(body->selectionFlagsLength, body->selectionFlags);
    }

    SelectDataIdReplyPacket* replyPacket = new(m_SyncChannelBuffer) SelectDataIdReplyPacket();
    m_SyncChannel.Write(replyPacket, sizeof(SelectDataIdReplyPacket));

    return true;
}

//----------------------------------------------------------
bool
SpySession::OnNotifyDataRead(IMessageHandler* msgHandler) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(msgHandler);
    NN_SDK_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::State_ReadHeader);

    if(!m_SyncPacketReader.ReadPacketBody(GetPacketBodyBuffer(), GetPacketBodyBufferSize()))
    {
        return false;
    }

    const PacketHeader* packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();
    NN_SDK_ASSERT_NOT_NULL(packetHeader);
    NN_SDK_ASSERT(packetHeader->bodyLength >= sizeof(NotifyDataReadPacket::Body));
    NN_UNUSED(packetHeader);

    NotifyDataReadPacket::Body* body = reinterpret_cast<NotifyDataReadPacket::Body*>(GetPacketBodyBuffer());
    NN_SDK_ASSERT_NOT_NULL(body);

#if defined(COM_DEBUG_ENABLED)
    NN_DETAIL_SPY_INFO("[SpySession] OnNotifyDataRead() : fileIndex=%d.\n", body->fileIndex);
#endif

    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

    if (m_State != State_Prepared)
    {
        // セッションが外部から閉じられた場合。
        return false;
    }

    if(msgHandler != NULL && body != NULL)
    {
        msgHandler->OnDataRead(body->fileIndex);
    }

    NotifyDataReadReplyPacket* replyPacket = new(m_SyncChannelBuffer) NotifyDataReadReplyPacket();
    m_SyncChannel.Write(replyPacket, sizeof(NotifyDataReadReplyPacket));

    return true;
}


//----------------------------------------------------------
bool
SpySession::OnFinalize(IMessageHandler* msgHandler) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(msgHandler);
    NN_SDK_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::State_ReadHeader);

#if defined(COM_DEBUG_ENABLED)
    NN_DETAIL_SPY_INFO("[SpySession] OnFinalize().\n");
#endif

    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

    if (m_State != State_Prepared)
    {
        // セッションが外部から閉じられた場合。
        return false;
    }

    if(msgHandler != NULL)
    {
        msgHandler->OnFinalizeSession();
    }

    FinalizeReplyPacket* replyPacket = new(m_SyncChannelBuffer) FinalizeReplyPacket();
    m_SyncChannel.Write(replyPacket, sizeof(FinalizeReplyPacket));

    // 一度 Close() しないと再接続できないプラットフォームがあるため、
    // Disconnect() ではなく Close() する。
    Close();

    return true;
}

//----------------------------------------------------------
bool
SpySession::OnPing() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::State_ReadHeader);

#if defined(COM_DEBUG_ENABLED)
    NN_DETAIL_SPY_INFO("[SpySession] OnPing().\n");
#endif

    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

    if (m_State != State_Prepared)
    {
        // セッションが外部から閉じられた場合。
        return false;
    }

    PongPacket* replyPacket = new(m_SyncChannelBuffer) PongPacket();

    if(m_SyncChannel.Write(replyPacket, sizeof(PongPacket)) != sizeof(PongPacket))
    {
        Disconnect();
    }

    return true;
}

//----------------------------------------------------------
bool
SpySession::OnQueryDataInfo(IMessageHandler* msgHandler) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(msgHandler);
    NN_SDK_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::State_ReadHeader);

    if (m_SyncPacketReader.GetState() == SpyPacketReader::State_ReadHeader &&
        !m_SyncPacketReader.ReadPacketBody(GetPacketBodyBuffer(), GetPacketBodyBufferSize()))
    {
        return false;
    }

    const PacketHeader* packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();
    NN_SDK_ASSERT_NOT_NULL(packetHeader);
    NN_UNUSED(packetHeader);

#if defined(COM_DEBUG_ENABLED)
    NN_DETAIL_SPY_INFO("[SpySession] OnQueryDataInfo()\n");
#endif

    nn::spy::detail::fnd::ScopedCriticalSection lock(m_StateLock);

    if (m_State != State_Prepared)
    {
        // セッションが外部から閉じられた場合。
        return false;
    }

    QueryDataInfoReplyPacket* replyPacket = new(m_SyncChannelBuffer) QueryDataInfoReplyPacket();

    const SpyDataInfo* dataInfo = NULL;

    if(msgHandler != NULL)
    {
        // NULLが返った場合はデータタイプの終了を表します。
        dataInfo = msgHandler->OnQueryDataInfo();
    }

    if (dataInfo != NULL)
    {
        static const char PADDING[4] = { '\0', '\0', '\0', '\0' };

        const char* dataName = dataInfo->GetDataName();
        size_t dataNameLength = strlen(dataName);
        size_t dataNameLengthPadded = nn::spy::detail::fnd::RoundUp(dataNameLength, 4);

        replyPacket->body.dataVersion = dataInfo->GetDataVersion();
        replyPacket->body.dataId = static_cast<uint16_t>(dataInfo->GetDataId());
        replyPacket->body.dataNameLength = static_cast<uint16_t>(dataNameLength);
        replyPacket->SetPacketSize();
        m_SyncChannel.Write(replyPacket, sizeof(QueryDataInfoReplyPacket));
        m_SyncChannel.Write(dataName, dataNameLength);
        if (dataNameLength != dataNameLengthPadded)
        {
            m_SyncChannel.Write(PADDING, dataNameLengthPadded - dataNameLength);
        }
    }
    else
    {
        replyPacket->body.dataVersion = 0;
        replyPacket->body.dataId = SpyDataId_Invalid;
        replyPacket->body.dataNameLength = 0;
        replyPacket->SetPacketSize();
        m_SyncChannel.Write(replyPacket, sizeof(QueryDataInfoReplyPacket));
    }

    return true;
}

//----------------------------------------------------------
const char*
SpySession::StateToString(State value) NN_NOEXCEPT
{
#if defined(STATE_DEBUG_ENABLED)
    static const char* strings[] = {
        "State_NotInitialized",
        "State_Initialized",
        "State_Opening",
        "State_Opened",
        "State_Disconnecting",
        "State_Connecting",
        "State_Preparing",
        "State_Prepared"
    };

    return strings[value];
#else
    NN_UNUSED(value);
    return NULL;
#endif
}

} // namespace nn::spy::detail
} // namespace nn::spy
} // namespace nn

#endif // NN_BUILD_CONFIG_SPY_ENABLED
