﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../precompiled.h"

#include <nn/nn_TimeSpan.h>

#include <nn/spy/detail/fnd/os/spyfnd_Thread.h>

namespace nn {
namespace spy {
namespace detail {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  スレッドのメイン関数オブジェクトです。
//---------------------------------------------------------------------------
class Thread::ThreadMain
{
public:
    static void Run(void* param) NN_NOEXCEPT
    {
        Thread* owner = reinterpret_cast<Thread*>(param);

        NN_SDK_ASSERT_NOT_NULL(owner);
        NN_SDK_ASSERT_NOT_NULL(owner->m_Handler);

        owner->m_IsTerminated = false;
        owner->OnRun();

        owner->m_Handler->Run(owner->m_Param);

        owner->OnExit();
        owner->m_IsTerminated = true;
    }
};

//---------------------------------------------------------------------------
Thread::Thread() NN_NOEXCEPT
    : m_State(State_NotRun)
    , m_Priority(DefaultThreadPriority)
    , m_Handler(NULL)
    , m_IsTerminated(false)
{
}

//---------------------------------------------------------------------------
void
Thread::SetPriority(int value) NN_NOEXCEPT
{
    nn::os::ChangeThreadPriority(&m_Handle, value);
}

//---------------------------------------------------------------------------
void
Thread::Sleep(const TimeSpan& timeSpan) NN_NOEXCEPT
{
    nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(timeSpan.ToNanoSeconds()));
}

//---------------------------------------------------------------------------
bool
Thread::Create(Handle& handle, const RunArgs& args) NN_NOEXCEPT
{
    if (!nn::os::CreateThread(
        &handle,
        ThreadMain::Run,
        this,
        args.stack,
        args.stackSize,
        args.priority).IsSuccess())
    {
        return false;
    }

    nn::os::StartThread(&handle);

    return true;
}

//---------------------------------------------------------------------------
void
Thread::Detach() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsTerminated()); // 事前に Join が呼ばれていれば ThreadMain::Run() の処理は完了しているはず

    nn::os::DestroyThread(&m_Handle);
}

//---------------------------------------------------------------------------
void
Thread::SetName(const char* name) NN_NOEXCEPT
{
    nn::os::SetThreadNamePointer(&m_Handle, name == NULL ? "" : name);
}

//---------------------------------------------------------------------------
void
Thread::SetAffinityMask(AffinityMask value) NN_NOEXCEPT
{
    // 対応APIが存在しないため未対応
    NN_UNUSED(value);
}

//---------------------------------------------------------------------------
void
Thread::Resume() NN_NOEXCEPT
{
}

//---------------------------------------------------------------------------
void
Thread::Join() NN_NOEXCEPT
{
    nn::os::WaitThread(&m_Handle);
}

//---------------------------------------------------------------------------
bool
Thread::IsTerminated() const NN_NOEXCEPT
{
    return m_IsTerminated;
}

} // namespace nn::spy::detail::fnd
} // namespace nn::spy::detail
} // namespace nn::spy
} // namespace nn
