﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../precompiled.h"

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <climits>

#include <nn/spy/detail/fnd/hio/spyfnd_HioChannel.h>

#include <WinSock2.h>

namespace nn {
namespace spy {
namespace detail {
namespace fnd {

NN_DEFINE_STATIC_CONSTANT(const PortType HioChannel::InvalidPort);

//----------------------------------------------------------
HioChannel::HioChannel() NN_NOEXCEPT
    : m_Port(InvalidPort)
    , m_Socket(INVALID_SOCKET)
    , m_ClientSocket(INVALID_SOCKET)
{
}

//----------------------------------------------------------
bool
HioChannel::Open(const PortType port) NN_NOEXCEPT
{
    m_Socket = socket(AF_INET, SOCK_STREAM, 0);

    if(m_Socket == INVALID_SOCKET)
    {
        return false;
    }

    struct sockaddr_in addr;
    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);
    addr.sin_addr.S_un.S_addr = htonl(INADDR_LOOPBACK);

    m_Port = port;

    if(bind(m_Socket, reinterpret_cast<const struct sockaddr*>(&addr), sizeof(struct sockaddr_in)) == SOCKET_ERROR)
    {
        Close();
        return false;
    }

    if(listen(m_Socket, 1) == SOCKET_ERROR)
    {
        Close();
        return false;
    }

    return true;
}

//----------------------------------------------------------
void
HioChannel::Close() NN_NOEXCEPT
{
    if(m_ClientSocket != INVALID_SOCKET)
    {
        closesocket(m_ClientSocket);
        m_ClientSocket = INVALID_SOCKET;
    }

    if(m_Socket != INVALID_SOCKET)
    {
        closesocket(m_Socket);
        m_Socket = INVALID_SOCKET;
    }

    m_Port = InvalidPort;
}

//----------------------------------------------------------
bool
HioChannel::Connect() NN_NOEXCEPT
{
    if(!IsOpened())
    {
        NN_SDK_ASSERT(false, "HioChannel is not opened.");
        return false;
    }

    if(IsConnected())
    {
        return true;
    }

    struct sockaddr_in client;
    int addrlen = sizeof(client);

    m_ClientSocket = accept(m_Socket, reinterpret_cast<struct sockaddr*>(&client), &addrlen);

    if(m_ClientSocket == INVALID_SOCKET)
    {
        Disconnect();
        return false;
    }

    return true;
}

//----------------------------------------------------------
void
HioChannel::Disconnect() NN_NOEXCEPT
{
    if(m_ClientSocket != INVALID_SOCKET)
    {
        closesocket(m_ClientSocket);
        m_ClientSocket = INVALID_SOCKET;
    }
}

//----------------------------------------------------------
bool
HioChannel::IsOpened() const NN_NOEXCEPT
{
    return m_Port != InvalidPort;
}

//----------------------------------------------------------
bool
HioChannel::IsConnected() const NN_NOEXCEPT
{
    return IsOpened() && m_ClientSocket != INVALID_SOCKET;
}

//----------------------------------------------------------
HioChannel::ConstPortType
HioChannel::GetPort() const NN_NOEXCEPT
{
    return m_Port;
}

//----------------------------------------------------------
size_t
HioChannel::Read(void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT_LESS(length, static_cast<size_t>(std::numeric_limits<int>::max()));

    if(!IsOpened())
    {
        return 0;
    }

    if(!IsConnected())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    char* current = reinterpret_cast<char*>(buffer);
    size_t readLength = 0;

    while(readLength < length)
    {
        int recvLength = recv(m_ClientSocket, current, static_cast<int>(length - readLength), 0);

        // ブロッキングモードの場合、0 以下を返したら切断されたとみなす
        if(recvLength <= 0)
        {
            Disconnect();
            break;
        }

        current += recvLength;
        readLength += recvLength;
    }

    return readLength;
}

//----------------------------------------------------------
size_t
HioChannel::Write(const void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT_LESS(length, static_cast<size_t>(std::numeric_limits<int>::max()));

    if(!IsOpened())
    {
        return 0;
    }

    if(!IsConnected())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    const char* current = reinterpret_cast<const char*>(buffer);
    size_t writtenLength = 0;

    while(writtenLength < length)
    {
        int sendLength = send(m_ClientSocket, current, static_cast<int>(length - writtenLength), 0);

        // ブロッキングモードの場合、0 以下を返したら切断されたとみなす
        if(sendLength <= 0)
        {
            Disconnect();
            break;
        }

        current += sendLength;
        writtenLength += sendLength;
    }

    return writtenLength;
}

} // namespace nn::spy::detail::fnd
} // namespace nn::spy::detail
} // namespace nn::spy
} // namespace nn

#endif // NN_BUILD_CONFIG_SPY_ENABLED
